/* -*- Mode: c++ -*- */
/*
 * Copyright 2001,2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRSIGSOURCE_H_
#define _VRSIGSOURCE_H_

extern "C" {
#include <math.h>
#include <sys/time.h>
#include <unistd.h>
}

#include <VrSource.h>
#include <stdlib.h>

enum {VR_SQUARE_WAVE, VR_TRIANGLE_WAVE, VR_SIN_WAVE, VR_DC_WAVE, VR_COS_WAVE};

template<class oType> 
class VrSigSource : public VrSource<oType> {
protected:
  int wform;
  double freq;
  double amp;
  double time;
  oType* buffer;
  void incTime() {time = time + 1/(double)getSamplingFrequency();}
  double slope() {return 4 * amp * getSamplingFrequency() / freq;}
  void createBuffers();
public:
  virtual int work2(VrSampleRange output, void *o[]);

  virtual void initialize();

  int getWaveform() {return wform;}
  double getAmplitude() {return amp;}
  double getFrequency() {return freq;}

  void setWaveform(int w) {wform = w; initialize ();}

  void setAmplitude(double a) {amp = a; initialize ();}

  void setFrequency(double f) {
    if (2*f > getSamplingFrequency()) 
      freq = getSamplingFrequency()/2; 
    else
      freq = f;
    initialize();
  }


  VrSigSource(double sampling_freq, int wf, double f, double a)  
    :wform(wf),freq(f),time(0),buffer(0)
  {
    setSamplingFrequency (sampling_freq);
    amp = a;
  }
    
  VrSigSource(double sampling_freq)
    :wform(VR_SIN_WAVE),freq(1000),time(0),buffer(0)
  { 
    setSamplingFrequency (sampling_freq);
    amp = (1 << 8*sizeof(oType) - 1);
  }
};

template<class oType> void
VrSigSource<oType>::createBuffers()
{
  unsigned int	i;
  
  switch (wform) {
  case VR_TRIANGLE_WAVE:
    for (i = 0;i < getOutputSize(); i++) {
      if ((i % (int)(getSamplingFrequency() / freq)) < 
	  getSamplingFrequency() / 2*freq)
	buffer[i] = (oType)(slope() * i - amp);
      else
	buffer[i] = (oType)(amp - slope() * 
			    (i - getSamplingFrequency() / 2*freq));
    }
    break;

  case VR_SIN_WAVE:
    for (i = 0; i < getOutputSize(); i++) {
      buffer[i] = (oType)(amp * sin(2*M_PI*freq*time));
      incTime();
    }
    
    // average last point to minimize discontinuity on wrap around
    i--;	// last point
    buffer[i] = (buffer[i-1] + buffer[0]) / 2;
    break;

  case VR_COS_WAVE:
    for (i = 0; i < getOutputSize(); i++) {
      buffer[i] = (oType)(amp * cos(2*M_PI*freq*time));
      incTime();
    }
    
    // average last point to minimize discontinuity on wrap around
    i--;	// last point
    buffer[i] = (buffer[i-1] + buffer[0]) / 2;
    break;

  case VR_SQUARE_WAVE:
    fprintf (stderr, "VrSigSource: VR_SQUARE_WAVE is broken\a\n");	// FIXME

    for (i = 0; i < getOutputSize();i++) {
      if (((int)time % (int)(getSamplingFrequency() / freq)) <  
	  getSamplingFrequency() / 2*freq)
	buffer[i] = (oType)amp;
      else
	buffer[i] = (oType)amp;
    }
    break;

  case VR_DC_WAVE:
    for (i = 0; i < getOutputSize(); i++) {
      buffer[i] = (oType)amp;
    }
    break;

  default:
    abort ();
  }
}

template<class oType> void
VrSigSource<oType>::initialize()
{
#if 1
  // pick a burst length that is the maximum of 50 ms or 2048 samples.
  double burst_length = 4096.0 / getSamplingFrequency ();
  if (0.050 > burst_length)
    burst_length = 0.050;
#else
  double burst_length = 0.050;
#endif

  // Output is in bursts, rounded to a full period.
  // outputSize changes when frequency is changed.

  if (freq == 0.0)	// DC wave generation
    setOutputSize ((int) (burst_length * getSamplingFrequency ()));

  else {
    double samples_per_cycle = getSamplingFrequency() / freq;
    int	 cycles_in_burst = (int) ((getSamplingFrequency() * burst_length) / samples_per_cycle);

    setOutputSize ((int) (cycles_in_burst * samples_per_cycle + 0.5));
  }
  
  if (buffer != 0)
    delete[] buffer; 
  buffer = new oType[getOutputSize()];
  createBuffers();
}

template<class oType> int
VrSigSource<oType>::work2(VrSampleRange output, void *ao[])
{ 
 oType **o = (oType **)ao;
  unsigned int size = output.size;
  while (size > 0) {
    memcpy((char *)(o[0]),(char *)buffer,getOutputSize()*sizeof(oType));
    size -= getOutputSize();
    o[0]+=getOutputSize();
  }
  return output.size;
}

#endif 
