/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gr_packed_to_unpacked_bb.h>
#include <gr_io_signature.h>
#include <assert.h>

static const int BITS_PER_BYTE = 8;

gr_packed_to_unpacked_bb_sptr 
gr_make_packed_to_unpacked_bb (unsigned int bits_per_chunk, gr_endianness_t endianness)
{
  return gr_packed_to_unpacked_bb_sptr 
    (new gr_packed_to_unpacked_bb (bits_per_chunk,endianness));
}

gr_packed_to_unpacked_bb::gr_packed_to_unpacked_bb (unsigned int bits_per_chunk, 
						    gr_endianness_t endianness)
  : gr_block ("packed_to_unpacked_bb",
	      gr_make_io_signature (1, 1, sizeof (unsigned char)),
	      gr_make_io_signature (1, 1, sizeof (unsigned char))),
    d_bits_per_chunk(bits_per_chunk),d_endianness(endianness),d_index(0)
{
  assert (bits_per_chunk <= 8);
  assert (bits_per_chunk > 0);

  set_relative_rate (8.0 / bits_per_chunk);
}

void
gr_packed_to_unpacked_bb::forecast(int noutput_items, gr_vector_int &ninput_items_required)
{
  ninput_items_required[0] =
    (int) ceil((d_index + noutput_items * d_bits_per_chunk) / 8.0);
  // printf("Forecast want %d need %d\n",noutput_items,ninput_items_required[0]);
}

int
get_bit_le (const unsigned char *in_vector,unsigned int bit_addr)
{
  unsigned char x = in_vector[bit_addr>>3];
  return (x>>(bit_addr&7))&1;
}

int
get_bit_be (const unsigned char *in_vector,unsigned int bit_addr)
{
  unsigned char x = in_vector[bit_addr>>3];
  return (x>>(7-(bit_addr&7)))&1;
}

int
gr_packed_to_unpacked_bb::general_work (int noutput_items,
					gr_vector_int &ninput_items,
					gr_vector_const_void_star &input_items,
					gr_vector_void_star &output_items)
{
  const unsigned char *in = (unsigned char *) input_items[0];
  unsigned char *out = (unsigned char *) output_items[0];

  //assert (noutput_items % BITS_PER_BYTE == 0);

  if(d_endianness == GR_MSB_FIRST){
    for (int i = 0; i < noutput_items; i++){
      //printf("here msb %d\n",i);
      unsigned char x = 0;
      for(unsigned int j=0; j<d_bits_per_chunk; j++, d_index++)
	x = (x<<1) | get_bit_be(in, d_index);

      out[i] = x;
    }
  }
  else {
    for (int i = 0; i < noutput_items; i++){
      //printf("here lsb %d\n",i);
      unsigned char x = 0;
      for(unsigned int j=0; j<d_bits_per_chunk; j++, d_index++)
	x = (x<<1) | get_bit_le(in, d_index);

      out[i] = x;
    }
  }

  //printf("almost got to end\n");
  assert(ninput_items[0] >= (int) ((d_index+7)>>3));
  consume_each (d_index >> 3);
  d_index = d_index & 7;
  //printf("got to end\n");
  return noutput_items;
}
