// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "mwi.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

static unsigned idx(const char *str)
{
	unsigned id = 0;

	while(*str)
		id ^= (id << 2) ^ (*(str++) & 0x1f);

	return id;
}

MWI::MWI() :
Module(), Mutex(), Keydata("/bayonne/switch"), Sync()
{
	ifstream cfg;
	char buf[32];
	char *cp;
	mwidata_t *mwi;

	static Keydata::Define keys[] = {
	{"group", "*"},	// default delivery target
	{"protocol", "generic"},	// default subject
	{NULL, NULL}};

	load(keys);

	driver->addModule(this);
	updated = false;

	freelist = NULL;
	memset(index, 0, sizeof(index));

	byport = new mwidata_t*[driver->getTrunkCount()];
	memset(byport, 0, sizeof(mwidata_t *) * driver->getTrunkCount());

	cfg.open("save/mwi.data");
	if(!cfg.is_open())
		return;

	while(!cfg.eof())
	{
		cfg.getline(buf, 32);
		cp = strchr(buf, ':');
		if(!cp)
			continue;
		*(cp++) = 0;
		mwi = addNumber(buf);
		if(!mwi)
			continue;
		mwi->count = mwi->prior = atoi(cp);		
	}
	cfg.close();
}

MWI::~MWI()
{
	delete[] byport;
	if(updated)
		saveDatabase();
}

mwidata_t *MWI::getNumber(const char *key)
{
	mwidata_t *node;
	enterMutex();
	node = index[idx(key)];

	while(node)
	{
		if(!stricmp(key, node->number))
			break;
		node = node->next;
	}
	leaveMutex();
	return node;
}

mwidata_t *MWI::addNumber(const char *key)
{
	unsigned id = idx(key);
	mwidata_t *node;

	enterMutex();
	node = index[id];
	while(node)
	{
		if(!stricmp(key, node->number))
			break;
		node = node->next;
	}

	if(!node)
	{
		updated = true;
		if(freelist)
		{
			node = freelist;
			freelist = node->byport;
		}
		else
			node = (mwidata_t *)alloc(sizeof(mwidata_t));
		node->next = index[id];
		index[id] = node;
		strcpy(node->number, key);
		node->count = node->prior = 0;
		node->byport = NULL;
		node->purge = false;
	}
	leaveMutex();	
	return node;
}	

void MWI::delNumber(const char *key)
{
	mwidata_t **link;
	mwidata_t *node;
	
	enterMutex();
	link = &index[idx(key)];
	while(*link)
	{
		node = *link;
		if(!stricmp(key, node->number))
			break;
		link = &node->next;
	}
	if(*link)
	{
		if(node->byport)
			node->purge = true;
		else
		{
			*link = node->next;
			node->byport = freelist;
			freelist = node;
			updated = true;
		}
	}
	leaveMutex();
}

void MWI::saveDatabase(void)
{
	mwidata_t	*mwi;
	char buf[25];
	int i;
	int fd;
	bool err = false;

	remove("save/mwi.temp");
	fd = creat("save/mwi.temp", 0660);
	if(fd < 0)
	{
		slog(Slog::levelError) << "mwi: save/mwi.save: cannot create" << endl;
		return;
	}

	for(i = 0; i < MWI_INDEX_SIZE; ++i)
	{
		mwi = index[i];
		while(mwi)
		{
			sprintf(buf, "%s:%d\n", mwi->number, mwi->count);
			if(write(fd, buf, strlen(buf)) < 1)
			{
				slog(Slog::levelError) << "mwi: write error" << endl;
				err = true;
				i = MWI_INDEX_SIZE;
				break;
			}
			mwi = mwi->next;
		}
	}
	close(fd);
	if(!err)
		rename("save/mwi.temp", "save/mwi.data");

	updated = false;
}

mwidata_t * MWI::getLink(unsigned id, const char *data)
{
	mwidata_t *mwi;

	enterMutex();
	mwi = getNumber(data);
	if(!mwi)
	{
		leaveMutex();
		return NULL;
	}
	if(!mwi->byport)
	{
		mwi->byport = byport[id];
		byport[id] = mwi;
	}
	leaveMutex();
	return mwi;
}

mwidata_t *MWI::addLink(unsigned id, const char *data)
{
	mwidata_t *mwi;

	enterMutex();
	mwi = addNumber(data);
	if(!mwi)
	{
		leaveMutex();
		return NULL;
	}
	if(!mwi->byport)
	{
		mwi->byport = byport[id];
		byport[id] = mwi;
	}
	leaveMutex();
	return mwi;
}

void	MWI::delLink(unsigned id, const char *number)
{
	mwidata_t **link, *node = NULL;

	enterMutex();
	link = &byport[id];

	while(*link)
	{
		node = *link;
		if(!stricmp(node->number, number))
			break;
		link = &node->byport;
	}
	if(*link)
	{
		*link = node->next;
		node->byport = NULL;
		node->prior = node->count;		
		if(node->purge)
			delNumber(number);
	}
	leaveMutex();
}

mwidata_t *MWI::getFirst(unsigned id)
{
	mwidata_t *node;

	enterMutex();
	while(NULL != (node = byport[id]))
	{
		if(node->prior != node->count)
			break;

		byport[id] = node->next;
		node->byport = NULL;
		if(node->purge)
			delNumber(node->number);
	}
	return node;
}

char *MWI::dispatch(Trunk *trunk)
{
	return "not-supported";
}

#ifdef	CCXX_NAMESPACES
};
#endif
