;; DFA-based pipeline descriptions for MIPS32 5K processor family
;; Contributed by David Ung (davidu@mips.com)
;;   and Nigel Stephens (nigel@mips.com)
;;
;; References:
;;   "MIPS64 5K Processor Core Family Software User's Manual,
;;     Doc no: MD00012, Rev 2.09, Jan 28, 2005."
;;
;; 5Kc - Single issue with no floating point unit.
;; 5kf - Separate floating point pipe which can dual-issue with the
;;       integer pipe.
;;
;; Copyright (C) 2005-2013 Free Software Foundation, Inc.
;;
;; This file is part of GCC.
;;
;; GCC is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 3, or (at your
;; option) any later version.

;; GCC is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GCC; see the file COPYING3.  If not see
;; <http://www.gnu.org/licenses/>.

(define_automaton "r5k_cpu, r5k_mdu, r5k_fpu")

;; Integer execution unit.
(define_cpu_unit "r5k_ixu_arith"       "r5k_cpu")
(define_cpu_unit "r5k_ixu_mpydiv"      "r5k_mdu")
(define_cpu_unit "r5kf_fpu_arith"      "r5k_fpu")

(define_insn_reservation "r5k_int_load" 2
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "load"))
  "r5k_ixu_arith")

(define_insn_reservation "r5k_int_prefetch" 1
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "prefetch,prefetchx"))
  "r5k_ixu_arith")

(define_insn_reservation "r5k_int_store" 1
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "store"))
  "r5k_ixu_arith")

;; Divides
(define_insn_reservation "r5k_int_divsi" 34
  (and (eq_attr "cpu" "5kc,5kf")
       (and (eq_attr "type" "idiv")
	    (eq_attr "mode" "!DI")))
 "r5k_ixu_arith+(r5k_ixu_mpydiv*34)")

(define_insn_reservation "r5k_int_divdi" 66
  (and (eq_attr "cpu" "5kc,5kf")
       (and (eq_attr "type" "idiv")
	    (eq_attr "mode" "DI")))
  "r5k_ixu_arith+(r5k_ixu_mpydiv*66)")

;; 32x32 multiply
;; 32x16 is faster, but there's no way to detect this
(define_insn_reservation "r5k_int_mult" 2
  (and (eq_attr "cpu" "5kc,5kf")
       (and (eq_attr "type" "imul,imadd")
	    (eq_attr "mode" "SI")))
  "r5k_ixu_arith+(r5k_ixu_mpydiv*2)")

;; 64x64 multiply
(define_insn_reservation "r5k_int_mult_64" 9
  (and (eq_attr "cpu" "5kc,5kf")
       (and (eq_attr "type" "imul,imadd")
	    (eq_attr "mode" "DI")))
  "r5k_ixu_arith+(r5k_ixu_mpydiv*2)")

;; 3 operand MUL 32x32
(define_insn_reservation "r5k_int_mul" 4
  (and (eq_attr "cpu" "5kc,5kf")
       (and (eq_attr "type" "imul3")
	    (eq_attr "mode" "SI")))
  "r5k_ixu_arith+(r5k_ixu_mpydiv*2)")

;; Move to HI/LO -> MADD/MSUB,MFHI/MFLO has a 1 cycle latency.
(define_insn_reservation "r5k_int_mthilo" 1
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "mthi,mtlo"))
  "r5k_ixu_arith+r5k_ixu_mpydiv")

;; Move from HI/LO -> integer operation has a 2 cycle latency.
(define_insn_reservation "r5k_int_mfhilo" 2
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "mfhi,mflo"))
  "r5k_ixu_arith+r5k_ixu_mpydiv")

;; All other integer insns.
(define_insn_reservation "r5k_int_alu" 1
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "arith,condmove,const,logical,move,nop,shift,signext,slt"))
  "r5k_ixu_arith")

(define_insn_reservation "r5k_int_branch" 1
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "branch"))
  "r5k_ixu_arith")

;; JR/JALR always cause one pipeline bubble because of interlock.
(define_insn_reservation "r5k_int_jump" 2
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "jump,call"))
  "r5k_ixu_arith")

;; Any    -> JR/JALR (without dependency) : 1 clock issue delay
;; Any    -> JR/JALR (with dependency)    : 2 clock issue delay
;; load   -> JR/JALR (with dependency)    : 3 clock issue delay
;; mfhilo -> JR/JALR (with dependency)    : 3 clock issue delay
;; mul    -> JR/JALR (with dependency)    : 3 clock issue delay
(define_bypass 2 "r5k_int_alu"    "r5k_int_jump")
(define_bypass 3 "r5k_int_load"   "r5k_int_jump")
(define_bypass 3 "r5k_int_mfhilo" "r5k_int_jump")
(define_bypass 3 "r5k_int_mul"    "r5k_int_jump")

;; Unknown or multi - single issue
(define_insn_reservation "r5k_int_unknown" 1
  (and (eq_attr "cpu" "5kc,5kf")
       (eq_attr "type" "unknown,multi,atomic,syncloop"))
  "r5k_ixu_arith+r5k_ixu_mpydiv")


;; Floating Point Instructions
;; The 5Kf is a partial dual-issue cpu which can dual issue an integer
;; and floating-point instruction in the same cycle.

;; fadd, fabs, fneg
(define_insn_reservation "r5kf_fadd" 4
  (and (eq_attr "cpu" "5kf")
       (eq_attr "type" "fadd,fabs,fneg"))
  "r5kf_fpu_arith")

;; fmove, fcmove
(define_insn_reservation "r5kf_fmove" 4
  (and (eq_attr "cpu" "5kf")
       (eq_attr "type" "fmove"))
  "r5kf_fpu_arith")

;; fload
(define_insn_reservation "r5kf_fload" 3
  (and (eq_attr "cpu" "5kf")
       (eq_attr "type" "fpload,fpidxload"))
  "r5kf_fpu_arith")

;; fstore
(define_insn_reservation "r5kf_fstore" 1
  (and (eq_attr "cpu" "5kf")
       (eq_attr "type" "fpstore"))
  "r5kf_fpu_arith")

;; fmul, fmadd
(define_insn_reservation "r5kf_fmul_sf" 4
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "fmul,fmadd")
	    (eq_attr "mode" "SF")))
  "r5kf_fpu_arith")

(define_insn_reservation "r5kf_fmul_df" 5
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "fmul,fmadd")
	    (eq_attr "mode" "DF")))
  "r5kf_fpu_arith*2")

;; fdiv, fsqrt, frsqrt
(define_insn_reservation "r5kf_fdiv_sf" 17
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "fdiv,fsqrt,frsqrt")
	    (eq_attr "mode" "SF")))
  "r5kf_fpu_arith*14")

(define_insn_reservation "r5kf_fdiv_df" 32
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "fdiv,fsqrt")
	    (eq_attr "mode" "DF")))
  "r5kf_fpu_arith*29")

;; frsqrt
(define_insn_reservation "r5kf_frsqrt_df" 35
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "frsqrt")
	    (eq_attr "mode" "DF")))
  "r5kf_fpu_arith*31")

;; fcmp
(define_insn_reservation "r5kf_fcmp" 2
  (and (eq_attr "cpu" "5kf")
       (eq_attr "type" "fcmp"))
  "r5kf_fpu_arith")

;; fcmp -> movf.fmt & movt.fmt bypass (dependency must be on condition)
(define_bypass 1 "r5kf_fcmp" "r5kf_fmove")

;; fcvt (cvt.d.s, cvt.[sd].[wl]
(define_insn_reservation "r5kf_fcvt_d2s" 4
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "fcvt")
	    (eq_attr "cnv_mode" "I2S,I2D,S2D")))
  "r5kf_fpu_arith")

;; fcvt (cvt.s.d)
(define_insn_reservation "r5kf_fcvt_s2d" 6
  (and (eq_attr "cpu" "5kc")
       (and (eq_attr "type" "fcvt")
	    (eq_attr "cnv_mode" "D2S")))
  "r5kf_fpu_arith")

;; fcvt (cvt.[wl].[sd], etc)
(define_insn_reservation "r5kf_fcvt_f2i" 5
  (and (eq_attr "cpu" "5kf")
       (and (eq_attr "type" "fcvt")
	    (eq_attr "cnv_mode" "S2I,D2I")))
  "r5kf_fpu_arith")

;; fxfer (mfc1, mfhc1, mtc1, mthc1) - single issue
(define_insn_reservation "r5kf_fxfer" 2
  (and (eq_attr "cpu" "5kf")
       (eq_attr "type" "mfc,mtc"))
  "r5k_ixu_arith+r5kf_fpu_arith")
