/* Copyright 2013 The Chromium OS Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 *
 * Register map for IT83xx processor
 */

#ifndef __CROS_EC_REGISTERS_H
#define __CROS_EC_REGISTERS_H

#include "common.h"
#include "compile_time_macros.h"

#define __ram_code __attribute__((section(__RAM_CODE_SECTION_NAME)))

/* IRQ numbers */
/* Group 0 */
#define IT83XX_IRQ_WKO20           1
#define IT83XX_IRQ_KBC_OUT         2
#define IT83XX_IRQ_PMC_OUT         3
#define IT83XX_IRQ_SMB_D           4
#define IT83XX_IRQ_WKINTAD         5
#define IT83XX_IRQ_WKO23           6
#define IT83XX_IRQ_PWM             7
/* Group 1 */
#define IT83XX_IRQ_ADC             8
#define IT83XX_IRQ_SMB_A           9
#define IT83XX_IRQ_SMB_B          10
#define IT83XX_IRQ_KB_MATRIX      11
#define IT83XX_IRQ_WKO26          12
#define IT83XX_IRQ_WKINTC         13
#define IT83XX_IRQ_WKO25          14
#define IT83XX_IRQ_CIR            15
/* Group 2 */
#define IT83XX_IRQ_SMB_C          16
#define IT83XX_IRQ_WKO24          17
#define IT83XX_IRQ_PS2_2          18
#define IT83XX_IRQ_PS2_1          19
#define IT83XX_IRQ_PS2_0          20
#define IT83XX_IRQ_WKO22          21
#define IT83XX_IRQ_SMFI           22
#define IT83XX_IRQ_USB            23
/* Group 3 */
#define IT83XX_IRQ_KBC_IN         24
#define IT83XX_IRQ_PMC_IN         25
#define IT83XX_IRQ_PMC2_OUT       26
#define IT83XX_IRQ_PMC2_IN        27
#define IT83XX_IRQ_GINT           28
#define IT83XX_IRQ_EGPC           29
#define IT83XX_IRQ_EXT_TIMER1     30
#define IT83XX_IRQ_WKO21          31
/* Group 4 */
#define IT83XX_IRQ_GPINT0         32
#define IT83XX_IRQ_GPINT1         33
#define IT83XX_IRQ_GPINT2         34
#define IT83XX_IRQ_GPINT3         35
#define IT83XX_IRQ_CIR_GPINT      36
#define IT83XX_IRQ_SSPI           37
#define IT83XX_IRQ_UART1          38
#define IT83XX_IRQ_UART2          39
/* Group 5 */
#define IT83XX_IRQ_WKO50          40
#define IT83XX_IRQ_WKO51          41
#define IT83XX_IRQ_WKO52          42
#define IT83XX_IRQ_WKO53          43
#define IT83XX_IRQ_WKO54          44
#define IT83XX_IRQ_WKO55          45
#define IT83XX_IRQ_WKO56          46
#define IT83XX_IRQ_WKO57          47
/* Group 6 */
#define IT83XX_IRQ_WKO60          48
#define IT83XX_IRQ_WKO61          49
#define IT83XX_IRQ_WKO62          50
#define IT83XX_IRQ_WKO63          51
#define IT83XX_IRQ_WKO64          52
#define IT83XX_IRQ_WKO65          53
#define IT83XX_IRQ_WKO66          54
#define IT83XX_IRQ_WKO67          55
/* Group 7 */
#define IT83XX_IRQ_RTCT_ALARM1    56
#define IT83XX_IRQ_RTCT_ALARM2    57
#define IT83XX_IRQ_EXT_TIMER2     58
#define IT83XX_IRQ_DEFERRED_SPI   59
#define IT83XX_IRQ_TMR_A0         60
#define IT83XX_IRQ_TMR_A1         61
#define IT83XX_IRQ_TMR_B0         62
#define IT83XX_IRQ_TMR_B1         63
/* Group 8 */
#define IT83XX_IRQ_PMC2EX_OUT     64
#define IT83XX_IRQ_PMC2EX_IN      65
#define IT83XX_IRQ_PMC3_OUT       66
#define IT83XX_IRQ_PMC3_IN        67
#define IT83XX_IRQ_PMC4_OUT       68
#define IT83XX_IRQ_PMC4_IN        69
#define IT83XX_IRQ_I2BRAM         71
/* Group 9 */
#define IT83XX_IRQ_WKO70          72
#define IT83XX_IRQ_WKO71          73
#define IT83XX_IRQ_WKO72          74
#define IT83XX_IRQ_WKO73          75
#define IT83XX_IRQ_WKO74          76
#define IT83XX_IRQ_WKO75          77
#define IT83XX_IRQ_WKO76          78
#define IT83XX_IRQ_WKO77          79
/* Group 10 */
#define IT83XX_IRQ_EXT_TMR8       80
#define IT83XX_IRQ_SMB_CLOCK_HELD 81
#define IT83XX_IRQ_CEC            82
#define IT83XX_IRQ_H2RAM_LPC      83
#define IT83XX_IRQ_HW_KB_SCAN     84
#define IT83XX_IRQ_WKO88          85
#define IT83XX_IRQ_WKO89          86
#define IT83XX_IRQ_WKO90          87
/* Group 11 */
#define IT83XX_IRQ_WKO80          88
#define IT83XX_IRQ_WKO81          89
#define IT83XX_IRQ_WKO82          90
#define IT83XX_IRQ_WKO83          91
#define IT83XX_IRQ_WKO84          92
#define IT83XX_IRQ_WKO85          93
#define IT83XX_IRQ_WKO86          94
#define IT83XX_IRQ_WKO87          95
/* Group 12 */
#define IT83XX_IRQ_WKO91          96
#define IT83XX_IRQ_WKO92          97
#define IT83XX_IRQ_WKO93          98
#define IT83XX_IRQ_WKO94          99
#define IT83XX_IRQ_WKO95          100
#define IT83XX_IRQ_WKO96          101
#define IT83XX_IRQ_WKO97          102
#define IT83XX_IRQ_WKO98          103
/* Group 13 */
#define IT83XX_IRQ_WKO99          104
#define IT83XX_IRQ_WKO100         105
#define IT83XX_IRQ_WKO101         106
#define IT83XX_IRQ_WKO102         107
#define IT83XX_IRQ_WKO103         108
#define IT83XX_IRQ_WKO104         109
#define IT83XX_IRQ_WKO105         110
#define IT83XX_IRQ_WKO106         111
/* Group 14 */
#define IT83XX_IRQ_WKO107         112
#define IT83XX_IRQ_WKO108         113
#define IT83XX_IRQ_WKO109         114
#define IT83XX_IRQ_WKO110         115
#define IT83XX_IRQ_WKO111         116
#define IT83XX_IRQ_WKO112         117
#define IT83XX_IRQ_WKO113         118
#define IT83XX_IRQ_WKO114         119
/* Group 15 */
#define IT83XX_IRQ_WKO115         120
#define IT83XX_IRQ_WKO116         121
#define IT83XX_IRQ_WKO117         122
#define IT83XX_IRQ_WKO118         123
#define IT83XX_IRQ_WKO119         124
#define IT83XX_IRQ_WKO120         125
#define IT83XX_IRQ_WKO121         126
#define IT83XX_IRQ_WKO122         127
/* Group 16 */
#define IT83XX_IRQ_WKO128         128
#define IT83XX_IRQ_WKO129         129
#define IT83XX_IRQ_WKO130         130
#define IT83XX_IRQ_WKO131         131
#define IT83XX_IRQ_WKO132         132
#define IT83XX_IRQ_WKO133         133
#define IT83XX_IRQ_WKO134         134
#define IT83XX_IRQ_WKO135         135
/* Group 17 */
#define IT83XX_IRQ_WKO136         136
#define IT83XX_IRQ_WKO137         137
#define IT83XX_IRQ_WKO138         138
#define IT83XX_IRQ_WKO139         139
#define IT83XX_IRQ_WKO140         140
#define IT83XX_IRQ_WKO141         141
#define IT83XX_IRQ_WKO142         142
#define IT83XX_IRQ_WKO143         143
/* Group 18 */
#define IT83XX_IRQ_WKO123         144
#define IT83XX_IRQ_WKO124         145
#define IT83XX_IRQ_WKO125         146
#define IT83XX_IRQ_WKO126         147
#define IT83XX_IRQ_PMC5_OUT       149
#define IT83XX_IRQ_PMC5_IN        150
#define IT83XX_IRQ_V_COMP         151
/* Group 19 */
#define IT83XX_IRQ_SMB_E          152
#define IT83XX_IRQ_SMB_F          153
#define IT83XX_IRQ_OSC_DMA        154
#define IT83XX_IRQ_EXT_TIMER3     155
#define IT83XX_IRQ_EXT_TIMER4     156
#define IT83XX_IRQ_EXT_TIMER5     157
#define IT83XX_IRQ_EXT_TIMER6     158
#define IT83XX_IRQ_EXT_TIMER7     159
/* Group 20 */
#define IT83XX_IRQ_PECI           160
#define IT83XX_IRQ_SOFTWARE       161
#define IT83XX_IRQ_ESPI           162
#define IT83XX_IRQ_ESPI_VW        163
#define IT83XX_IRQ_PCH_P80        164
#define IT83XX_IRQ_USBPD0         165
#define IT83XX_IRQ_USBPD1         166
/* Group 21 */
#if defined(CHIP_FAMILY_IT8320)
#define IT83XX_IRQ_WKO40          168
#define IT83XX_IRQ_WKO45          169
#define IT83XX_IRQ_WKO46          170
#define IT83XX_IRQ_WKO144         171
#define IT83XX_IRQ_WKO145         172
#define IT83XX_IRQ_WKO146         173
#define IT83XX_IRQ_WKO147         174
#define IT83XX_IRQ_WKO148         175
/* Group 22 */
#define IT83XX_IRQ_WKO149         176
#define IT83XX_IRQ_WKO150         177

#define IT83XX_IRQ_COUNT          178
#elif defined(CHIP_FAMILY_IT8XXX1) || defined(CHIP_FAMILY_IT8XXX2)
/* Group 21 */
#define IT83XX_IRQ_AUDIO_IF       170
#define IT83XX_IRQ_SPI_SLAVE      171
#define IT83XX_IRQ_DSP_ENGINE     172
#define IT83XX_IRQ_NN_ENGINE      173
#define IT83XX_IRQ_USBPD2         174
#define IT83XX_IRQ_CRYPTO         175
/* Group 22 */
#define IT83XX_IRQ_WKO40          176
#define IT83XX_IRQ_WKO45          177
#define IT83XX_IRQ_WKO46          178
#define IT83XX_IRQ_WKO144         179
#define IT83XX_IRQ_WKO145         180
#define IT83XX_IRQ_WKO146         181
#define IT83XX_IRQ_WKO147         182
#define IT83XX_IRQ_WKO148         183
/* Group 23 */
#define IT83XX_IRQ_WKO149         184
#define IT83XX_IRQ_WKO150         185
#define IT83XX_IRQ_SSPI1          191
/* Group 24 */
#define IT83XX_IRQ_XLPIN0         192
#define IT83XX_IRQ_XLPIN1         193
#define IT83XX_IRQ_XLPIN2         194
#define IT83XX_IRQ_XLPIN3         195
#define IT83XX_IRQ_XLPIN4         196
#define IT83XX_IRQ_XLPIN5         197
#define IT83XX_IRQ_WEEK_ALARM     199
/* Group 25 */
#define IT83XX_IRQ_GPO0           200
#define IT83XX_IRQ_GPO1           201
#define IT83XX_IRQ_GPO2           202
#define IT83XX_IRQ_GPO3           203
/* Group 26 */
#define IT83XX_IRQ_GPP0           208
#define IT83XX_IRQ_GPP1           209
#define IT83XX_IRQ_GPP2           210
#define IT83XX_IRQ_GPP3           211
#define IT83XX_IRQ_GPP4           212
#define IT83XX_IRQ_GPP5           213
#define IT83XX_IRQ_GPP6           214
/* Group 27 */
#define IT83XX_IRQ_GPQ0           216
#define IT83XX_IRQ_GPQ1           217
#define IT83XX_IRQ_GPQ2           218
#define IT83XX_IRQ_GPQ3           219
#define IT83XX_IRQ_GPQ4           220
#define IT83XX_IRQ_GPQ5           221
/* Group 28 */
#define IT83XX_IRQ_GPR0           224
#define IT83XX_IRQ_GPR1           225
#define IT83XX_IRQ_GPR2           226
#define IT83XX_IRQ_GPR3           227
#define IT83XX_IRQ_GPR4           228
#define IT83XX_IRQ_GPR5           229

#define IT83XX_IRQ_COUNT          230
#endif /* !defined(CHIP_FAMILY_IT8320) */

/* IRQ dispatching to CPU INT vectors */
#define IT83XX_CPU_INT_IRQ_1       2
#define IT83XX_CPU_INT_IRQ_2       5
#define IT83XX_CPU_INT_IRQ_3       4
#define IT83XX_CPU_INT_IRQ_4       6
#define IT83XX_CPU_INT_IRQ_5       2
#define IT83XX_CPU_INT_IRQ_6       2
#define IT83XX_CPU_INT_IRQ_7       4
#define IT83XX_CPU_INT_IRQ_8       7
#define IT83XX_CPU_INT_IRQ_9       6
#define IT83XX_CPU_INT_IRQ_10      6
#define IT83XX_CPU_INT_IRQ_11      5
#define IT83XX_CPU_INT_IRQ_12      2
#define IT83XX_CPU_INT_IRQ_13      2
#define IT83XX_CPU_INT_IRQ_14      2
#define IT83XX_CPU_INT_IRQ_15      8
#define IT83XX_CPU_INT_IRQ_16      6
#define IT83XX_CPU_INT_IRQ_17      2
#define IT83XX_CPU_INT_IRQ_18      8
#define IT83XX_CPU_INT_IRQ_19      8
#define IT83XX_CPU_INT_IRQ_20      8
#define IT83XX_CPU_INT_IRQ_21      2
#define IT83XX_CPU_INT_IRQ_22     12
#define IT83XX_CPU_INT_IRQ_23     12
#define IT83XX_CPU_INT_IRQ_24      5
#define IT83XX_CPU_INT_IRQ_25      4
#define IT83XX_CPU_INT_IRQ_26      4
#define IT83XX_CPU_INT_IRQ_27      4
#define IT83XX_CPU_INT_IRQ_28     11
#define IT83XX_CPU_INT_IRQ_29     11
#define IT83XX_CPU_INT_IRQ_30      3
#define IT83XX_CPU_INT_IRQ_31      2
#define IT83XX_CPU_INT_IRQ_32     11
#define IT83XX_CPU_INT_IRQ_33     11
#define IT83XX_CPU_INT_IRQ_34     11
#define IT83XX_CPU_INT_IRQ_35     11
#define IT83XX_CPU_INT_IRQ_36      8
#define IT83XX_CPU_INT_IRQ_37      9
#define IT83XX_CPU_INT_IRQ_38      9
#define IT83XX_CPU_INT_IRQ_39      9
#define IT83XX_CPU_INT_IRQ_40      2
#define IT83XX_CPU_INT_IRQ_41      2
#define IT83XX_CPU_INT_IRQ_42      2
#define IT83XX_CPU_INT_IRQ_43      2
#define IT83XX_CPU_INT_IRQ_44      2
#define IT83XX_CPU_INT_IRQ_45      2
#define IT83XX_CPU_INT_IRQ_46      2
#define IT83XX_CPU_INT_IRQ_47      2
#define IT83XX_CPU_INT_IRQ_48      2
#define IT83XX_CPU_INT_IRQ_49      2
#define IT83XX_CPU_INT_IRQ_50      2
#define IT83XX_CPU_INT_IRQ_51      2
#define IT83XX_CPU_INT_IRQ_52      2
#define IT83XX_CPU_INT_IRQ_53      2
#define IT83XX_CPU_INT_IRQ_54      2
#define IT83XX_CPU_INT_IRQ_55      2
#define IT83XX_CPU_INT_IRQ_56     10
#define IT83XX_CPU_INT_IRQ_57     10
#define IT83XX_CPU_INT_IRQ_58      3
#define IT83XX_CPU_INT_IRQ_59     12
#define IT83XX_CPU_INT_IRQ_60      3
#define IT83XX_CPU_INT_IRQ_61      3
#define IT83XX_CPU_INT_IRQ_62      3
#define IT83XX_CPU_INT_IRQ_63      3
#define IT83XX_CPU_INT_IRQ_64      4
#define IT83XX_CPU_INT_IRQ_65      4
#define IT83XX_CPU_INT_IRQ_66      4
#define IT83XX_CPU_INT_IRQ_67      4
#define IT83XX_CPU_INT_IRQ_68      4
#define IT83XX_CPU_INT_IRQ_69      4
#define IT83XX_CPU_INT_IRQ_71     12
#define IT83XX_CPU_INT_IRQ_72      2
#define IT83XX_CPU_INT_IRQ_73      2
#define IT83XX_CPU_INT_IRQ_74      2
#define IT83XX_CPU_INT_IRQ_75      2
#define IT83XX_CPU_INT_IRQ_76      2
#define IT83XX_CPU_INT_IRQ_77      2
#define IT83XX_CPU_INT_IRQ_78      2
#define IT83XX_CPU_INT_IRQ_79      2
#define IT83XX_CPU_INT_IRQ_80      3
#define IT83XX_CPU_INT_IRQ_81      6
#define IT83XX_CPU_INT_IRQ_82     12
#define IT83XX_CPU_INT_IRQ_83     12
#define IT83XX_CPU_INT_IRQ_84      5
#define IT83XX_CPU_INT_IRQ_85      2
#define IT83XX_CPU_INT_IRQ_86      2
#define IT83XX_CPU_INT_IRQ_87      2
#define IT83XX_CPU_INT_IRQ_88      2
#define IT83XX_CPU_INT_IRQ_89      2
#define IT83XX_CPU_INT_IRQ_90      2
#define IT83XX_CPU_INT_IRQ_91      2
#define IT83XX_CPU_INT_IRQ_92      2
#define IT83XX_CPU_INT_IRQ_93      2
#define IT83XX_CPU_INT_IRQ_94      2
#define IT83XX_CPU_INT_IRQ_95      2
#define IT83XX_CPU_INT_IRQ_96      2
#define IT83XX_CPU_INT_IRQ_97      2
#define IT83XX_CPU_INT_IRQ_98      2
#define IT83XX_CPU_INT_IRQ_99      2
#define IT83XX_CPU_INT_IRQ_100     2
#define IT83XX_CPU_INT_IRQ_101     2
#define IT83XX_CPU_INT_IRQ_102     2
#define IT83XX_CPU_INT_IRQ_103     2
#define IT83XX_CPU_INT_IRQ_104     2
#define IT83XX_CPU_INT_IRQ_105     2
#define IT83XX_CPU_INT_IRQ_106     2
#define IT83XX_CPU_INT_IRQ_107     2
#define IT83XX_CPU_INT_IRQ_108     2
#define IT83XX_CPU_INT_IRQ_109     2
#define IT83XX_CPU_INT_IRQ_110     2
#define IT83XX_CPU_INT_IRQ_111     2
#define IT83XX_CPU_INT_IRQ_112     2
#define IT83XX_CPU_INT_IRQ_113     2
#define IT83XX_CPU_INT_IRQ_114     2
#define IT83XX_CPU_INT_IRQ_115     2
#define IT83XX_CPU_INT_IRQ_116     2
#define IT83XX_CPU_INT_IRQ_117     2
#define IT83XX_CPU_INT_IRQ_118     2
#define IT83XX_CPU_INT_IRQ_119     2
#define IT83XX_CPU_INT_IRQ_120     2
#define IT83XX_CPU_INT_IRQ_121     2
#define IT83XX_CPU_INT_IRQ_122     2
#define IT83XX_CPU_INT_IRQ_123     2
#define IT83XX_CPU_INT_IRQ_124     2
#define IT83XX_CPU_INT_IRQ_125     2
#define IT83XX_CPU_INT_IRQ_126     2
#define IT83XX_CPU_INT_IRQ_127     2
#define IT83XX_CPU_INT_IRQ_128     2
#define IT83XX_CPU_INT_IRQ_129     2
#define IT83XX_CPU_INT_IRQ_130     2
#define IT83XX_CPU_INT_IRQ_131     2
#define IT83XX_CPU_INT_IRQ_132     2
#define IT83XX_CPU_INT_IRQ_133     2
#define IT83XX_CPU_INT_IRQ_134     2
#define IT83XX_CPU_INT_IRQ_135     2
#define IT83XX_CPU_INT_IRQ_136     2
#define IT83XX_CPU_INT_IRQ_137     2
#define IT83XX_CPU_INT_IRQ_138     2
#define IT83XX_CPU_INT_IRQ_139     2
#define IT83XX_CPU_INT_IRQ_140     2
#define IT83XX_CPU_INT_IRQ_141     2
#define IT83XX_CPU_INT_IRQ_142     2
#define IT83XX_CPU_INT_IRQ_143     2
#define IT83XX_CPU_INT_IRQ_144     2
#define IT83XX_CPU_INT_IRQ_145     2
#define IT83XX_CPU_INT_IRQ_146     2
#define IT83XX_CPU_INT_IRQ_147     2
#define IT83XX_CPU_INT_IRQ_149     4
#define IT83XX_CPU_INT_IRQ_150     4
#define IT83XX_CPU_INT_IRQ_151     7
#define IT83XX_CPU_INT_IRQ_152     6
#define IT83XX_CPU_INT_IRQ_153     6
#define IT83XX_CPU_INT_IRQ_154    12
#define IT83XX_CPU_INT_IRQ_155     3
#define IT83XX_CPU_INT_IRQ_156     3
#define IT83XX_CPU_INT_IRQ_157     3
#define IT83XX_CPU_INT_IRQ_158     3
#define IT83XX_CPU_INT_IRQ_159     3
#define IT83XX_CPU_INT_IRQ_160    12
#define IT83XX_CPU_INT_IRQ_161    12
#define IT83XX_CPU_INT_IRQ_162    12
#define IT83XX_CPU_INT_IRQ_163    12
#define IT83XX_CPU_INT_IRQ_164    12
#define IT83XX_CPU_INT_IRQ_165    12
#define IT83XX_CPU_INT_IRQ_166    12
#define IT83XX_CPU_INT_IRQ_167    12
#define IT83XX_CPU_INT_IRQ_168     2
#define IT83XX_CPU_INT_IRQ_169     2
#if defined(CHIP_FAMILY_IT8320)
#define IT83XX_CPU_INT_IRQ_170     2
#define IT83XX_CPU_INT_IRQ_171     2
#define IT83XX_CPU_INT_IRQ_172     2
#define IT83XX_CPU_INT_IRQ_173     2
#define IT83XX_CPU_INT_IRQ_174     2
#define IT83XX_CPU_INT_IRQ_175     2
#elif defined(CHIP_FAMILY_IT8XXX1) || defined(CHIP_FAMILY_IT8XXX2)
#define IT83XX_CPU_INT_IRQ_170    12
#define IT83XX_CPU_INT_IRQ_171    12
#define IT83XX_CPU_INT_IRQ_172    12
#define IT83XX_CPU_INT_IRQ_173    12
#define IT83XX_CPU_INT_IRQ_174    12
#define IT83XX_CPU_INT_IRQ_175    12
#endif
#define IT83XX_CPU_INT_IRQ_176     2
#define IT83XX_CPU_INT_IRQ_177     2
#define IT83XX_CPU_INT_IRQ_178     2
#define IT83XX_CPU_INT_IRQ_179     2
#define IT83XX_CPU_INT_IRQ_180     2
#define IT83XX_CPU_INT_IRQ_181     2
#define IT83XX_CPU_INT_IRQ_182     2
#define IT83XX_CPU_INT_IRQ_183     2
#define IT83XX_CPU_INT_IRQ_184     2
#define IT83XX_CPU_INT_IRQ_185     2
#define IT83XX_CPU_INT_IRQ_191     2
#define IT83XX_CPU_INT_IRQ_192     2
#define IT83XX_CPU_INT_IRQ_193     2
#define IT83XX_CPU_INT_IRQ_194     2
#define IT83XX_CPU_INT_IRQ_195     2
#define IT83XX_CPU_INT_IRQ_196     2
#define IT83XX_CPU_INT_IRQ_197     2
#define IT83XX_CPU_INT_IRQ_199     2
#define IT83XX_CPU_INT_IRQ_200     2
#define IT83XX_CPU_INT_IRQ_201     2
#define IT83XX_CPU_INT_IRQ_202     2
#define IT83XX_CPU_INT_IRQ_203     2
#define IT83XX_CPU_INT_IRQ_208     2
#define IT83XX_CPU_INT_IRQ_209     2
#define IT83XX_CPU_INT_IRQ_210     2
#define IT83XX_CPU_INT_IRQ_211     2
#define IT83XX_CPU_INT_IRQ_212     2
#define IT83XX_CPU_INT_IRQ_213     2
#define IT83XX_CPU_INT_IRQ_214     2
#define IT83XX_CPU_INT_IRQ_216     2
#define IT83XX_CPU_INT_IRQ_217     2
#define IT83XX_CPU_INT_IRQ_218     2
#define IT83XX_CPU_INT_IRQ_219     2
#define IT83XX_CPU_INT_IRQ_220     2
#define IT83XX_CPU_INT_IRQ_221     2
#define IT83XX_CPU_INT_IRQ_224     2
#define IT83XX_CPU_INT_IRQ_225     2
#define IT83XX_CPU_INT_IRQ_226     2
#define IT83XX_CPU_INT_IRQ_227     2
#define IT83XX_CPU_INT_IRQ_228     2
#define IT83XX_CPU_INT_IRQ_229     2

/* "Fake" IRQ to declare in readable fashion all WKO IRQ routed to INT#2 */
#define CPU_INT_2_ALL_GPIOS      255
#define IT83XX_CPU_INT_IRQ_255     2

#define CPU_INT_GROUP_5          254
#define IT83XX_CPU_INT_IRQ_254     5

#define CPU_INT_GROUP_4          252
#define IT83XX_CPU_INT_IRQ_252     4

#define CPU_INT_GROUP_12         253
#define IT83XX_CPU_INT_IRQ_253    12

#define CPU_INT_GROUP_3          251
#define IT83XX_CPU_INT_IRQ_251     3

#define CPU_INT_GROUP_6          250
#define IT83XX_CPU_INT_IRQ_250     6

#define CPU_INT_GROUP_9          249
#define IT83XX_CPU_INT_IRQ_249     9

#define CPU_INT_GROUP_7          248
#define IT83XX_CPU_INT_IRQ_248     7

#define CPU_INT(irq) CONCAT2(IT83XX_CPU_INT_IRQ_, irq)

/* --- INTC --- */
#define IT83XX_INTC_BASE    CHIP_EC_INTC_BASE

#define IT83XX_INTC_REG(n)  REG8(IT83XX_INTC_BASE+(n))

#define IT83XX_INTC_AIVCT   REG8(IT83XX_INTC_BASE+0x10)

#define IT83XX_INTC_IER0    REG8(IT83XX_INTC_BASE+0x04)
#define IT83XX_INTC_IER1    REG8(IT83XX_INTC_BASE+0x05)
#define IT83XX_INTC_IER2    REG8(IT83XX_INTC_BASE+0x06)
#define IT83XX_INTC_IER3    REG8(IT83XX_INTC_BASE+0x07)
#define IT83XX_INTC_IER4    REG8(IT83XX_INTC_BASE+0x15)
#define IT83XX_INTC_IER5    REG8(IT83XX_INTC_BASE+0x19)
#define IT83XX_INTC_IER6    REG8(IT83XX_INTC_BASE+0x1d)
#define IT83XX_INTC_IER7    REG8(IT83XX_INTC_BASE+0x21)
#define IT83XX_INTC_IER8    REG8(IT83XX_INTC_BASE+0x25)
#define IT83XX_INTC_IER9    REG8(IT83XX_INTC_BASE+0x29)
#define IT83XX_INTC_IER10   REG8(IT83XX_INTC_BASE+0x2d)
#define IT83XX_INTC_IER11   REG8(IT83XX_INTC_BASE+0x31)
#define IT83XX_INTC_IER12   REG8(IT83XX_INTC_BASE+0x35)
#define IT83XX_INTC_IER13   REG8(IT83XX_INTC_BASE+0x39)
#define IT83XX_INTC_IER14   REG8(IT83XX_INTC_BASE+0x3d)
#define IT83XX_INTC_IER15   REG8(IT83XX_INTC_BASE+0x41)
#define IT83XX_INTC_IER16   REG8(IT83XX_INTC_BASE+0x45)
#define IT83XX_INTC_IER17   REG8(IT83XX_INTC_BASE+0x49)
#define IT83XX_INTC_IER18   REG8(IT83XX_INTC_BASE+0x4d)
#define IT83XX_INTC_IER19   REG8(IT83XX_INTC_BASE+0x51)
#define IT83XX_INTC_IER20   REG8(IT83XX_INTC_BASE+0x55)
#define IT83XX_INTC_IER21   REG8(IT83XX_INTC_BASE+0x59)
#define IT83XX_INTC_IER22   REG8(IT83XX_INTC_BASE+0x5d)
#define IT83XX_INTC_IER23   REG8(IT83XX_INTC_BASE+0x91)
#define IT83XX_INTC_IER24   REG8(IT83XX_INTC_BASE+0x95)
#define IT83XX_INTC_IER25   REG8(IT83XX_INTC_BASE+0x99)
#define IT83XX_INTC_IER26   REG8(IT83XX_INTC_BASE+0x9d)
#define IT83XX_INTC_IER27   REG8(IT83XX_INTC_BASE+0xa1)
#define IT83XX_INTC_IER28   REG8(IT83XX_INTC_BASE+0xa5)

#define IT83XX_INTC_ISR0    REG8(IT83XX_INTC_BASE+0x00)
#define IT83XX_INTC_ISR1    REG8(IT83XX_INTC_BASE+0x01)
#define IT83XX_INTC_ISR2    REG8(IT83XX_INTC_BASE+0x02)
#define IT83XX_INTC_ISR3    REG8(IT83XX_INTC_BASE+0x03)
#define IT83XX_INTC_ISR4    REG8(IT83XX_INTC_BASE+0x14)
#define IT83XX_INTC_ISR5    REG8(IT83XX_INTC_BASE+0x18)
#define IT83XX_INTC_ISR6    REG8(IT83XX_INTC_BASE+0x1c)
#define IT83XX_INTC_ISR7    REG8(IT83XX_INTC_BASE+0x20)
#define IT83XX_INTC_ISR8    REG8(IT83XX_INTC_BASE+0x24)
#define IT83XX_INTC_ISR9    REG8(IT83XX_INTC_BASE+0x28)
#define IT83XX_INTC_ISR10   REG8(IT83XX_INTC_BASE+0x2c)
#define IT83XX_INTC_ISR11   REG8(IT83XX_INTC_BASE+0x30)
#define IT83XX_INTC_ISR12   REG8(IT83XX_INTC_BASE+0x34)
#define IT83XX_INTC_ISR13   REG8(IT83XX_INTC_BASE+0x38)
#define IT83XX_INTC_ISR14   REG8(IT83XX_INTC_BASE+0x3c)
#define IT83XX_INTC_ISR15   REG8(IT83XX_INTC_BASE+0x40)
#define IT83XX_INTC_ISR16   REG8(IT83XX_INTC_BASE+0x44)
#define IT83XX_INTC_ISR17   REG8(IT83XX_INTC_BASE+0x48)
#define IT83XX_INTC_ISR18   REG8(IT83XX_INTC_BASE+0x4c)
#define IT83XX_INTC_ISR19   REG8(IT83XX_INTC_BASE+0x50)
#define IT83XX_INTC_ISR20   REG8(IT83XX_INTC_BASE+0x54)
#define IT83XX_INTC_ISR21   REG8(IT83XX_INTC_BASE+0x58)
#define IT83XX_INTC_ISR22   REG8(IT83XX_INTC_BASE+0x5c)
#define IT83XX_INTC_ISR23   REG8(IT83XX_INTC_BASE+0x90)
#define IT83XX_INTC_ISR24   REG8(IT83XX_INTC_BASE+0x94)
#define IT83XX_INTC_ISR25   REG8(IT83XX_INTC_BASE+0x98)
#define IT83XX_INTC_ISR26   REG8(IT83XX_INTC_BASE+0x9c)
#define IT83XX_INTC_ISR27   REG8(IT83XX_INTC_BASE+0xa0)
#define IT83XX_INTC_ISR28   REG8(IT83XX_INTC_BASE+0xa4)

#define IT83XX_INTC_IELMR10 REG8(IT83XX_INTC_BASE+0x2E)
#define IT83XX_INTC_IPOLR10 REG8(IT83XX_INTC_BASE+0x2F)
#define IT83XX_INTC_IELMR19 REG8(IT83XX_INTC_BASE+0x52)
#define IT83XX_INTC_IPOLR19 REG8(IT83XX_INTC_BASE+0x53)

#define IT83XX_INTC_EXT_IER_OFF(n) (0x60 + (n))
#define IT83XX_INTC_IVCT(i)     REG8(IT83XX_INTC_BASE+0x80+(i))

/* --- EC Access to the Host Controlled Modules (EC2I Bridge) --- */
#define IT83XX_EC2I_BASE  0x00F01200

#define IT83XX_EC2I_IHIOA       REG8(IT83XX_EC2I_BASE+0x00)
#define IT83XX_EC2I_IHD         REG8(IT83XX_EC2I_BASE+0x01)
#define IT83XX_EC2I_LSIOHA      REG8(IT83XX_EC2I_BASE+0x02)
#define IT83XX_EC2I_SIOLV       REG8(IT83XX_EC2I_BASE+0x03)
#define IT83XX_EC2I_IBMAE       REG8(IT83XX_EC2I_BASE+0x04)
#define IT83XX_EC2I_IBCTL       REG8(IT83XX_EC2I_BASE+0x05)

/* --- System Wake-UP Control (SWUC) --- */
#define IT83XX_SWUC_BASE  0x00F01400
#define IT83XX_SWUC_SWCTL1      REG8(IT83XX_SWUC_BASE+0x00)

/* --- Wake-Up Control (WUC) --- */
#define IT83XX_WUC_BASE   0x00F01B00

#define IT83XX_WUC_WUEMR1 (IT83XX_WUC_BASE+0x00)
#define IT83XX_WUC_WUEMR5 (IT83XX_WUC_BASE+0x0c)
#define IT83XX_WUC_WUESR1 (IT83XX_WUC_BASE+0x04)
#define IT83XX_WUC_WUESR5 (IT83XX_WUC_BASE+0x0d)
#define IT83XX_WUC_WUBEMR1 (IT83XX_WUC_BASE+0x3c)
#define IT83XX_WUC_WUBEMR5 (IT83XX_WUC_BASE+0x0f)

#define IT83XX_WUC_WUESR10    REG8(IT83XX_WUC_BASE+0x21)
#define IT83XX_WUC_WUESR11    REG8(IT83XX_WUC_BASE+0x25)

#define IT83XX_WUC_WUEMR3     REG8(IT83XX_WUC_BASE+0x02)
#define IT83XX_WUC_WUESR3     REG8(IT83XX_WUC_BASE+0x06)
#define IT83XX_WUC_WUENR3     REG8(IT83XX_WUC_BASE+0x0A)

#define IT83XX_WUC_WUEMR4     REG8(IT83XX_WUC_BASE+0x03)
#define IT83XX_WUC_WUESR4     REG8(IT83XX_WUC_BASE+0x07)
#define IT83XX_WUC_WUENR4     REG8(IT83XX_WUC_BASE+0x0B)

/* --- UART --- */
#define IT83XX_UART0_BASE 0x00F02700
#define IT83XX_UART1_BASE 0x00F02800

#define IT83XX_UART_BASE(n)        CONCAT3(IT83XX_UART, n, _BASE)
#define IT83XX_UART_REG(n, offset) REG8(IT83XX_UART_BASE(n) + (offset))

#define IT83XX_UART_DLL(n)         IT83XX_UART_REG(n, 0x00)
#define IT83XX_UART_DLM(n)         IT83XX_UART_REG(n, 0x01)
#define IT83XX_UART_RBR(n)         IT83XX_UART_REG(n, 0x00)
#define IT83XX_UART_THR(n)         IT83XX_UART_REG(n, 0x00)
#define IT83XX_UART_IER(n)         IT83XX_UART_REG(n, 0x01)
#define IT83XX_UART_IIR(n)         IT83XX_UART_REG(n, 0x02)
#define IT83XX_UART_FCR(n)         IT83XX_UART_REG(n, 0x02)
#define IT83XX_UART_LCR(n)         IT83XX_UART_REG(n, 0x03)
#define IT83XX_UART_MCR(n)         IT83XX_UART_REG(n, 0x04)
#define IT83XX_UART_LSR(n)         IT83XX_UART_REG(n, 0x05)
#define IT83XX_UART_MSR(n)         IT83XX_UART_REG(n, 0x06)
#define IT83XX_UART_SCR(n)         IT83XX_UART_REG(n, 0x07)
#define IT83XX_UART_ECSMPR(n)      IT83XX_UART_REG(n, 0x08)
#define IT83XX_UART_CSSR(n)        IT83XX_UART_REG(n, 0x09)

/* --- GPIO --- */

#define IT83XX_GPIO_BASE  0x00F01600
#define IT83XX_GPIO2_BASE 0x00F03E00

#define IT83XX_GPIO_GCR         REG8(IT83XX_GPIO_BASE+0x00)
#define IT83XX_GPIO_GCR_LPC_RST_B7      0x1
#define IT83XX_GPIO_GCR_LPC_RST_D2      0x2
#define IT83XX_GPIO_GCR_LPC_RST_DISABLE 0x3
#define IT83XX_GPIO_GCR_LPC_RST_POS     1

#define IT83XX_GPIO_GPDRA       REG8(IT83XX_GPIO_BASE+0x01)
#define IT83XX_GPIO_GPDRB       REG8(IT83XX_GPIO_BASE+0x02)
#define IT83XX_GPIO_GPDRC       REG8(IT83XX_GPIO_BASE+0x03)
#define IT83XX_GPIO_GPDRE       REG8(IT83XX_GPIO_BASE+0x05)
#define IT83XX_GPIO_GPDRF       REG8(IT83XX_GPIO_BASE+0x06)
#define IT83XX_GPIO_GPDRH       REG8(IT83XX_GPIO_BASE+0x08)

#define IT83XX_GPIO_GPCRA0      REG8(IT83XX_GPIO_BASE+0x10)
#define IT83XX_GPIO_GPCRA1      REG8(IT83XX_GPIO_BASE+0x11)
#define IT83XX_GPIO_GPCRA2      REG8(IT83XX_GPIO_BASE+0x12)
#define IT83XX_GPIO_GPCRA3      REG8(IT83XX_GPIO_BASE+0x13)
#define IT83XX_GPIO_GPCRA4      REG8(IT83XX_GPIO_BASE+0x14)
#define IT83XX_GPIO_GPCRA5      REG8(IT83XX_GPIO_BASE+0x15)
#define IT83XX_GPIO_GPCRA6      REG8(IT83XX_GPIO_BASE+0x16)
#define IT83XX_GPIO_GPCRA7      REG8(IT83XX_GPIO_BASE+0x17)

#define IT83XX_GPIO_GPCRB0      REG8(IT83XX_GPIO_BASE+0x18)
#define IT83XX_GPIO_GPCRB1      REG8(IT83XX_GPIO_BASE+0x19)
#define IT83XX_GPIO_GPCRB2      REG8(IT83XX_GPIO_BASE+0x1A)
#define IT83XX_GPIO_GPCRB3      REG8(IT83XX_GPIO_BASE+0x1B)
#define IT83XX_GPIO_GPCRB4      REG8(IT83XX_GPIO_BASE+0x1C)
#define IT83XX_GPIO_GPCRB5      REG8(IT83XX_GPIO_BASE+0x1D)
#define IT83XX_GPIO_GPCRB6      REG8(IT83XX_GPIO_BASE+0x1E)
#define IT83XX_GPIO_GPCRB7      REG8(IT83XX_GPIO_BASE+0x1F)

#define IT83XX_GPIO_GPCRC0      REG8(IT83XX_GPIO_BASE+0x20)
#define IT83XX_GPIO_GPCRC1      REG8(IT83XX_GPIO_BASE+0x21)
#define IT83XX_GPIO_GPCRC2      REG8(IT83XX_GPIO_BASE+0x22)
#define IT83XX_GPIO_GPCRC3      REG8(IT83XX_GPIO_BASE+0x23)
#define IT83XX_GPIO_GPCRC4      REG8(IT83XX_GPIO_BASE+0x24)
#define IT83XX_GPIO_GPCRC5      REG8(IT83XX_GPIO_BASE+0x25)
#define IT83XX_GPIO_GPCRC6      REG8(IT83XX_GPIO_BASE+0x26)
#define IT83XX_GPIO_GPCRC7      REG8(IT83XX_GPIO_BASE+0x27)

#define IT83XX_GPIO_GPCRE0      REG8(IT83XX_GPIO_BASE+0x30)
#define IT83XX_GPIO_GPCRE1      REG8(IT83XX_GPIO_BASE+0x31)
#define IT83XX_GPIO_GPCRE2      REG8(IT83XX_GPIO_BASE+0x32)
#define IT83XX_GPIO_GPCRE3      REG8(IT83XX_GPIO_BASE+0x33)
#define IT83XX_GPIO_GPCRE4      REG8(IT83XX_GPIO_BASE+0x34)
#define IT83XX_GPIO_GPCRE5      REG8(IT83XX_GPIO_BASE+0x35)
#define IT83XX_GPIO_GPCRE6      REG8(IT83XX_GPIO_BASE+0x36)
#define IT83XX_GPIO_GPCRE7      REG8(IT83XX_GPIO_BASE+0x37)

#define IT83XX_GPIO_GPCRF0      REG8(IT83XX_GPIO_BASE+0x38)
#define IT83XX_GPIO_GPCRF1      REG8(IT83XX_GPIO_BASE+0x39)
#define IT83XX_GPIO_GPCRF2      REG8(IT83XX_GPIO_BASE+0x3A)
#define IT83XX_GPIO_GPCRF3      REG8(IT83XX_GPIO_BASE+0x3B)
#define IT83XX_GPIO_GPCRF4      REG8(IT83XX_GPIO_BASE+0x3C)
#define IT83XX_GPIO_GPCRF5      REG8(IT83XX_GPIO_BASE+0x3D)
#define IT83XX_GPIO_GPCRF6      REG8(IT83XX_GPIO_BASE+0x3E)
#define IT83XX_GPIO_GPCRF7      REG8(IT83XX_GPIO_BASE+0x3F)

#define IT83XX_GPIO_GPCRH0      REG8(IT83XX_GPIO_BASE+0x48)
#define IT83XX_GPIO_GPCRH1      REG8(IT83XX_GPIO_BASE+0x49)
#define IT83XX_GPIO_GPCRH2      REG8(IT83XX_GPIO_BASE+0x4A)
#define IT83XX_GPIO_GPCRH3      REG8(IT83XX_GPIO_BASE+0x4B)
#define IT83XX_GPIO_GPCRH4      REG8(IT83XX_GPIO_BASE+0x4C)
#define IT83XX_GPIO_GPCRH5      REG8(IT83XX_GPIO_BASE+0x4D)
#define IT83XX_GPIO_GPCRH6      REG8(IT83XX_GPIO_BASE+0x4E)
#define IT83XX_GPIO_GPCRH7      REG8(IT83XX_GPIO_BASE+0x4F)

#define IT83XX_GPIO_GPCRI0      REG8(IT83XX_GPIO_BASE+0x50)
#define IT83XX_GPIO_GPCRI1      REG8(IT83XX_GPIO_BASE+0x51)
#define IT83XX_GPIO_GPCRI2      REG8(IT83XX_GPIO_BASE+0x52)
#define IT83XX_GPIO_GPCRI3      REG8(IT83XX_GPIO_BASE+0x53)
#define IT83XX_GPIO_GPCRI4      REG8(IT83XX_GPIO_BASE+0x54)
#define IT83XX_GPIO_GPCRI5      REG8(IT83XX_GPIO_BASE+0x55)
#define IT83XX_GPIO_GPCRI6      REG8(IT83XX_GPIO_BASE+0x56)
#define IT83XX_GPIO_GPCRI7      REG8(IT83XX_GPIO_BASE+0x57)

#define IT83XX_GPIO_GPCRM5      REG8(IT83XX_GPIO_BASE+0xA5)

#define IT83XX_GPIO_GPDMRA      REG8(IT83XX_GPIO_BASE+0x61)
#define IT83XX_GPIO_GPDMRB      REG8(IT83XX_GPIO_BASE+0x62)
#define IT83XX_GPIO_GPDMRC      REG8(IT83XX_GPIO_BASE+0x63)
#define IT83XX_GPIO_GPDMRE      REG8(IT83XX_GPIO_BASE+0x65)
#define IT83XX_GPIO_GPDMRF      REG8(IT83XX_GPIO_BASE+0x66)
#define IT83XX_GPIO_GPDMRH      REG8(IT83XX_GPIO_BASE+0x68)

#define IT83XX_GPIO_GPCRL0      REG8(IT83XX_GPIO_BASE+0x98)
#define IT83XX_GPIO_GPCRL1      REG8(IT83XX_GPIO_BASE+0x99)
#define IT83XX_GPIO_GPCRL2      REG8(IT83XX_GPIO_BASE+0x9A)
#define IT83XX_GPIO_GPCRL3      REG8(IT83XX_GPIO_BASE+0x9B)
#define IT83XX_GPIO_GPCRP0      REG8(IT83XX_GPIO2_BASE+0x18)
#define IT83XX_GPIO_GPCRP1      REG8(IT83XX_GPIO2_BASE+0x19)

#define IT83XX_GPIO_GRC1        REG8(IT83XX_GPIO_BASE+0xF0)
#define IT83XX_GPIO_GRC2        REG8(IT83XX_GPIO_BASE+0xF1)
#define IT83XX_GPIO_GRC3        REG8(IT83XX_GPIO_BASE+0xF2)
#define IT83XX_GPIO_GRC4        REG8(IT83XX_GPIO_BASE+0xF3)
#define IT83XX_GPIO_GRC5        REG8(IT83XX_GPIO_BASE+0xF4)
#define IT83XX_GPIO_GRC6        REG8(IT83XX_GPIO_BASE+0xF5)
#define IT83XX_GPIO_GRC7        REG8(IT83XX_GPIO_BASE+0xF6)
#define IT83XX_GPIO_GRC8        REG8(IT83XX_GPIO_BASE+0xF7)
#define IT83XX_GPIO_GRC19       REG8(IT83XX_GPIO_BASE+0xE4)
#define IT83XX_GPIO_GRC20       REG8(IT83XX_GPIO_BASE+0xE5)
#define IT83XX_GPIO_GRC21       REG8(IT83XX_GPIO_BASE+0xE6)
#define IT83XX_GPIO_GRC22       REG8(IT83XX_GPIO_BASE+0xE7)
#define IT83XX_GPIO_GRC23       REG8(IT83XX_GPIO_BASE+0xE8)
#define IT83XX_GPIO_GRC24       REG8(IT83XX_GPIO_BASE+0xE9)
#define IT83XX_GPIO_GCR25       REG8(IT83XX_GPIO_BASE+0xD1)
#define IT83XX_GPIO_GCR26       REG8(IT83XX_GPIO_BASE+0xD2)
#define IT83XX_GPIO_GCR27       REG8(IT83XX_GPIO_BASE+0xD3)
#define IT83XX_GPIO_GCR28       REG8(IT83XX_GPIO_BASE+0xD4)
#define IT83XX_GPIO_GCR29       REG8(IT83XX_GPIO_BASE+0xEE)
#define IT83XX_GPIO_GCR30       REG8(IT83XX_GPIO_BASE+0xED)
#define IT83XX_GPIO_GCR31       REG8(IT83XX_GPIO_BASE+0xD5)
#define IT83XX_GPIO_GCR32       REG8(IT83XX_GPIO_BASE+0xD6)
#define IT83XX_GPIO_GCR33       REG8(IT83XX_GPIO_BASE+0xD7)

#define IT83XX_VBATPC_BGPOPSCR  REG8(IT83XX_GPIO2_BASE+0xF0)
#define IT83XX_VBATPC_XLPIER    REG8(IT83XX_GPIO2_BASE+0xF5)

enum {
	/* GPIO group index */
	GPIO_A = 0x1,
	GPIO_B = 0x2,
	GPIO_C = 0x3,
	GPIO_D = 0x4,
	GPIO_E = 0x5,
	GPIO_F = 0x6,
	GPIO_G = 0x7,
	GPIO_H = 0x8,
	GPIO_I = 0x9,
	GPIO_J = 0xa,
	GPIO_K = 0xb,
	GPIO_L = 0xc,
	GPIO_M = 0xd,
#if defined(CHIP_FAMILY_IT8XXX1) || defined(CHIP_FAMILY_IT8XXX2)
	GPIO_O = 0xe,
	GPIO_P = 0xf,
	GPIO_Q = 0x10,
	GPIO_R = 0x11,
#endif
	GPIO_PORT_COUNT,

	/*
	 * NOTE: support flags when KSI/KSO are configured as GPIO
	 * 1) it8320bx:
	 * output: GPIO_OUTPUT, GPIO_OPEN_DRAIN, GPIO_HIGH, GPIO_LOW
	 * input: GPIO_INPUT
	 * 2) it8320dx, it8xxx1, and it8xxx2:
	 * output: GPIO_OUTPUT, GPIO_OPEN_DRAIN(always internal pullup),
	 *         GPIO_HIGH, GPIO_LOW
	 * input: GPIO_INPUT, GPIO_PULL_UP
	 */
	/* KSI[7-0]  GPIO data mirror register. */
	GPIO_KSI,
	/* KSO[15-8] GPIO data mirror register. */
	GPIO_KSO_H,
	/* KSO[7-0]  GPIO data mirror register. */
	GPIO_KSO_L,
	/* Compiler check COUNT and gpio_group_to_reg member cnt match or not */
	COUNT,
};

struct gpio_reg_t {
	/* GPIO and KSI/KSO port data register (bit mapping to pin) */
	uint32_t reg_gpdr;
	/* GPIO and KSI/KSO port data mirror register (bit mapping to pin) */
	uint32_t reg_gpdmr;
	/* GPIO and KSI/KSO port output type register (bit mapping to pin) */
	uint32_t reg_gpotr;
	/* GPIO port control register (byte mapping to pin) */
	uint32_t reg_gpcr;
};

/* GPIO group index convert to GPIO data/output type/ctrl group address */
static const struct gpio_reg_t gpio_group_to_reg[] = {
	/*               GPDR(set),  GPDMR(get), GPOTR,      GPCR      */
	[GPIO_A]     = { 0x00F01601, 0x00F01661, 0x00F01671, 0x00F01610 },
	[GPIO_B]     = { 0x00F01602, 0x00F01662, 0x00F01672, 0x00F01618 },
	[GPIO_C]     = { 0x00F01603, 0x00F01663, 0x00F01673, 0x00F01620 },
	[GPIO_D]     = { 0x00F01604, 0x00F01664, 0x00F01674, 0x00F01628 },
	[GPIO_E]     = { 0x00F01605, 0x00F01665, 0x00F01675, 0x00F01630 },
	[GPIO_F]     = { 0x00F01606, 0x00F01666, 0x00F01676, 0x00F01638 },
	[GPIO_G]     = { 0x00F01607, 0x00F01667, 0x00F01677, 0x00F01640 },
	[GPIO_H]     = { 0x00F01608, 0x00F01668, 0x00F01678, 0x00F01648 },
	[GPIO_I]     = { 0x00F01609, 0x00F01669, 0x00F01679, 0x00F01650 },
	[GPIO_J]     = { 0x00F0160A, 0x00F0166A, 0x00F0167A, 0x00F01658 },
	[GPIO_K]     = { 0x00F0160B, 0x00F0166B, 0x00F0167B, 0x00F01690 },
	[GPIO_L]     = { 0x00F0160C, 0x00F0166C, 0x00F0167C, 0x00F01698 },
	[GPIO_M]     = { 0x00F0160D, 0x00F0166D, 0x00F0167D, 0x00F016a0 },
#if defined(CHIP_FAMILY_IT8XXX1) || defined(CHIP_FAMILY_IT8XXX2)
	[GPIO_O]     = { 0x00F03E01, 0x00F03E61, 0x00F03E71, 0x00F03E10 },
	[GPIO_P]     = { 0x00F03E02, 0x00F03E62, 0x00F03E72, 0x00F03E18 },
	[GPIO_Q]     = { 0x00F03E03, 0x00F03E63, 0x00F03E73, 0x00F03E20 },
	[GPIO_R]     = { 0x00F03E04, 0x00F03E64, 0x00F03E74, 0x00F03E28 },
#endif
	[GPIO_KSI]   = { 0x00F01D08, 0x00F01D09, 0x00F01D26, 0xFFFFFFFF },
	[GPIO_KSO_H] = { 0x00F01D01, 0x00F01D0C, 0x00F01D27, 0xFFFFFFFF },
	[GPIO_KSO_L] = { 0x00F01D00, 0x00F01D0F, 0x00F01D28, 0xFFFFFFFF },
};
BUILD_ASSERT(ARRAY_SIZE(gpio_group_to_reg) == (COUNT));

#define UNIMPLEMENTED_GPIO_BANK GPIO_A

#define IT83XX_GPIO_DATA(port)                 \
	REG8(gpio_group_to_reg[port].reg_gpdr)
#define IT83XX_GPIO_DATA_MIRROR(port)          \
	REG8(gpio_group_to_reg[port].reg_gpdmr)
#define IT83XX_GPIO_GPOT(port)                 \
	REG8(gpio_group_to_reg[port].reg_gpotr)
#define IT83XX_GPIO_CTRL(port, pin_offset)     \
	REG8(gpio_group_to_reg[port].reg_gpcr + pin_offset)
#define GPCR_PORT_PIN_MODE_INPUT          BIT(7)
#define GPCR_PORT_PIN_MODE_OUTPUT         BIT(6)
#define GPCR_PORT_PIN_MODE_PULLUP         BIT(2)
#define GPCR_PORT_PIN_MODE_PULLDOWN       BIT(1)

/* --- Clock and Power Management (ECPM) --- */

#define IT83XX_ECPM_BASE  0x00F01E00

#define IT83XX_ECPM_CGCTRL1R_OFF 0x01
#define IT83XX_ECPM_CGCTRL2R_OFF 0x02
#define IT83XX_ECPM_CGCTRL3R_OFF 0x05
#define IT83XX_ECPM_CGCTRL4R_OFF 0x09

#define IT83XX_ECPM_PLLCTRL	REG8(IT83XX_ECPM_BASE+0x03)
enum ec_pll_ctrl {
	EC_PLL_DOZE = 0,
	EC_PLL_SLEEP = 1,
	EC_PLL_DEEP_DOZE = 3,
};

#define IT83XX_ECPM_AUTOCG	REG8(IT83XX_ECPM_BASE+0x04)
#define IT83XX_ECPM_PLLFREQR	REG8(IT83XX_ECPM_BASE+0x06)
#define IT83XX_ECPM_PLLCSS	REG8(IT83XX_ECPM_BASE+0x08)
#define IT83XX_ECPM_SCDCR0	REG8(IT83XX_ECPM_BASE+0x0c)
#define IT83XX_ECPM_SCDCR1	REG8(IT83XX_ECPM_BASE+0x0d)
#define IT83XX_ECPM_SCDCR2	REG8(IT83XX_ECPM_BASE+0x0e)
#define IT83XX_ECPM_SCDCR3	REG8(IT83XX_ECPM_BASE+0x0f)
#define IT83XX_ECPM_SCDCR4	REG8(IT83XX_ECPM_BASE+0x10)

/*
 * The clock gate offsets combine the register offset from ECPM_BASE and the
 * mask within that register into one value. These are used for
 * clock_enable_peripheral() and clock_disable_peripheral()
 */
enum clock_gate_offsets {
	CGC_OFFSET_EGPC  = ((IT83XX_ECPM_CGCTRL2R_OFF << 8) | 0x40),
	CGC_OFFSET_CIR   = ((IT83XX_ECPM_CGCTRL2R_OFF << 8) | 0x20),
	CGC_OFFSET_SWUC  = ((IT83XX_ECPM_CGCTRL2R_OFF << 8) | 0x10),
	CGC_OFFSET_USB   = ((IT83XX_ECPM_CGCTRL3R_OFF << 8) | 0x20),
	CGC_OFFSET_PECI  = ((IT83XX_ECPM_CGCTRL3R_OFF << 8) | 0x08),
	CGC_OFFSET_UART  = ((IT83XX_ECPM_CGCTRL3R_OFF << 8) | 0x04),
	CGC_OFFSET_SSPI  = ((IT83XX_ECPM_CGCTRL3R_OFF << 8) | 0x02),
	CGC_OFFSET_DBGR  = ((IT83XX_ECPM_CGCTRL3R_OFF << 8) | 0x01),
	CGC_OFFSET_SMBF  = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x80),
	CGC_OFFSET_SMBE  = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x40),
	CGC_OFFSET_SMBD  = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x20),
	CGC_OFFSET_SMBC  = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x10),
	CGC_OFFSET_SMBB  = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x08),
	CGC_OFFSET_SMBA  = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x04),
	CGC_OFFSET_SMB   = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x02),
	CGC_OFFSET_CEC   = ((IT83XX_ECPM_CGCTRL4R_OFF << 8) | 0x01)
};

/* --- Timer (TMR) --- */
#define IT83XX_TMR_BASE         0x00F02900

#define IT83XX_TMR_PRSC         REG8(IT83XX_TMR_BASE+0x00)
#define IT83XX_TMR_GCSMS        REG8(IT83XX_TMR_BASE+0x01)
#define IT83XX_TMR_CTR_A0       REG8(IT83XX_TMR_BASE+0x02)
#define IT83XX_TMR_CTR_A1       REG8(IT83XX_TMR_BASE+0x03)
#define IT83XX_TMR_CTR_B0       REG8(IT83XX_TMR_BASE+0x04)
#define IT83XX_TMR_CTR_B1       REG8(IT83XX_TMR_BASE+0x05)
#define IT83XX_TMR_DCR_A0       REG8(IT83XX_TMR_BASE+0x06)
#define IT83XX_TMR_DCR_A1       REG8(IT83XX_TMR_BASE+0x07)
#define IT83XX_TMR_DCR_B0       REG8(IT83XX_TMR_BASE+0x08)
#define IT83XX_TMR_DCR_B1       REG8(IT83XX_TMR_BASE+0x09)
#define IT83XX_TMR_CCGSR        REG8(IT83XX_TMR_BASE+0x0A)
#define IT83XX_TMR_TMRCE        REG8(IT83XX_TMR_BASE+0x0B)
#define IT83XX_TMR_TMRIE        REG8(IT83XX_TMR_BASE+0x0C)

/* --- External Timer and Watchdog (ETWD) --- */
#define IT83XX_ETWD_BASE  0x00F01F00

#define IT83XX_ETWD_ETWCFG        REG8(IT83XX_ETWD_BASE+0x01)
#define IT83XX_ETWD_ET1PSR        REG8(IT83XX_ETWD_BASE+0x02)
#define IT83XX_ETWD_ET1CNTLHR     REG8(IT83XX_ETWD_BASE+0x03)
#define IT83XX_ETWD_ET1CNTLLR     REG8(IT83XX_ETWD_BASE+0x04)
#define IT83XX_ETWD_ETWCTRL       REG8(IT83XX_ETWD_BASE+0x05)
#define IT83XX_ETWD_EWDCNTLLR     REG8(IT83XX_ETWD_BASE+0x06)
#define IT83XX_ETWD_EWDKEYR       REG8(IT83XX_ETWD_BASE+0x07)
#define IT83XX_ETWD_EWDCNTLHR     REG8(IT83XX_ETWD_BASE+0x09)
#define IT83XX_ETWD_ETXCTRL(n)    REG8(IT83XX_ETWD_BASE + 0x10 + (n << 3))
#define IT83XX_ETWD_ETXPSR(n)     REG8(IT83XX_ETWD_BASE + 0x11 + (n << 3))
#define IT83XX_ETWD_ETXCNTLR(n)   REG32(IT83XX_ETWD_BASE + 0x14 + (n << 3))
#define IT83XX_ETWD_ETXCNTOR(n)   REG32(IT83XX_ETWD_BASE + 0x48 + (n << 2))

/* --- General Control (GCTRL) --- */
#define IT83XX_GCTRL_BASE 0x00F02000

#ifdef IT83XX_CHIP_ID_3BYTES
#define IT83XX_GCTRL_CHIPID1      REG8(IT83XX_GCTRL_BASE+0x85)
#define IT83XX_GCTRL_CHIPID2      REG8(IT83XX_GCTRL_BASE+0x86)
#define IT83XX_GCTRL_CHIPID3      REG8(IT83XX_GCTRL_BASE+0x87)
#else
#define IT83XX_GCTRL_CHIPID1      REG8(IT83XX_GCTRL_BASE+0x00)
#define IT83XX_GCTRL_CHIPID2      REG8(IT83XX_GCTRL_BASE+0x01)
#endif
#define IT83XX_GCTRL_CHIPVER      REG8(IT83XX_GCTRL_BASE+0x02)
#define IT83XX_GCTRL_DBGROS       REG8(IT83XX_GCTRL_BASE+0x03)
#define IT83XX_SMB_DBGR           BIT(0)
#define IT83XX_GCTRL_WNCKR        REG8(IT83XX_GCTRL_BASE+0x0B)
#define IT83XX_GCTRL_RSTS         REG8(IT83XX_GCTRL_BASE+0x06)
#define IT83XX_GCTRL_BADRSEL      REG8(IT83XX_GCTRL_BASE+0x0A)
#define IT83XX_GCTRL_SPCTRL1      REG8(IT83XX_GCTRL_BASE+0x0D)
#define IT83XX_GCTRL_RSTDMMC      REG8(IT83XX_GCTRL_BASE+0x10)
#define IT83XX_GCTRL_RSTC4        REG8(IT83XX_GCTRL_BASE+0x11)
#define IT83XX_GCTRL_SPCTRL4      REG8(IT83XX_GCTRL_BASE+0x1C)
#define IT83XX_GCTRL_MCCR3        REG8(IT83XX_GCTRL_BASE+0x20)
#define IT83XX_GCTRL_SPISLVPFE    BIT(6)
#define IT83XX_GCTRL_RSTC5        REG8(IT83XX_GCTRL_BASE+0x21)
#define IT83XX_GCTRL_MCCR         REG8(IT83XX_GCTRL_BASE+0x30)
#define IT83XX_GCTRL_PMER1        REG8(IT83XX_GCTRL_BASE+0x32)
#define IT83XX_GCTRL_PMER2        REG8(IT83XX_GCTRL_BASE+0x33)
#define IT83XX_GCTRL_EPLR         REG8(IT83XX_GCTRL_BASE+0x37)
#define IT83XX_GCTRL_IVTBAR       REG8(IT83XX_GCTRL_BASE+0x41)
#define IT83XX_GCTRL_MCCR2        REG8(IT83XX_GCTRL_BASE+0x44)
#define IT83XX_GCTRL_PIN_MUX0     REG8(IT83XX_GCTRL_BASE+0x46)
#define IT83XX_DLM14_ENABLE       BIT(5)
#define IT83XX_GCTRL_SSCR         REG8(IT83XX_GCTRL_BASE+0x4A)
#define IT83XX_GCTRL_ETWDUARTCR   REG8(IT83XX_GCTRL_BASE+0x4B)
#define IT83XX_GCTRL_WMCR         REG8(IT83XX_GCTRL_BASE+0x4C)
#define IT83XX_GCTRL_H2ROFSR      REG8(IT83XX_GCTRL_BASE+0x53)
/* bit[0] = 0 or 1 : disable or enable ETWD hardware reset */
#define ETWD_HW_RST_EN            BIT(0)
#define IT83XX_GCTRL_RVILMCR0     REG8(IT83XX_GCTRL_BASE+0x5D)
#define ILMCR_ILM0_ENABLE         BIT(0)
#define ILMCR_ILM2_ENABLE         BIT(2)
#define IT83XX_GCTRL_EWPR0PFH(i)  REG8(IT83XX_GCTRL_BASE+0x60+i)
#define IT83XX_GCTRL_EWPR0PFD(i)  REG8(IT83XX_GCTRL_BASE+0xA0+i)
#define IT83XX_GCTRL_EWPR0PFEC(i) REG8(IT83XX_GCTRL_BASE+0xC0+i)

/* --- Pulse Width Modulation (PWM) --- */
#define IT83XX_PWM_BASE   0x00F01800

#define IT83XX_PWM_C0CPRS       REG8(IT83XX_PWM_BASE+0x00)
#define IT83XX_PWM_CTR          REG8(IT83XX_PWM_BASE+0x01)
#define IT83XX_PWM_DCR0         REG8(IT83XX_PWM_BASE+0x02)
#define IT83XX_PWM_DCR1         REG8(IT83XX_PWM_BASE+0x03)
#define IT83XX_PWM_DCR2         REG8(IT83XX_PWM_BASE+0x04)
#define IT83XX_PWM_DCR3         REG8(IT83XX_PWM_BASE+0x05)
#define IT83XX_PWM_DCR4         REG8(IT83XX_PWM_BASE+0x06)
#define IT83XX_PWM_DCR5         REG8(IT83XX_PWM_BASE+0x07)
#define IT83XX_PWM_DCR6         REG8(IT83XX_PWM_BASE+0x08)
#define IT83XX_PWM_DCR7         REG8(IT83XX_PWM_BASE+0x09)
#define IT83XX_PWM_PWMPOL       REG8(IT83XX_PWM_BASE+0x0A)
#define IT83XX_PWM_PCFSR        REG8(IT83XX_PWM_BASE+0x0B)
#define IT83XX_PWM_PCSSGL       REG8(IT83XX_PWM_BASE+0x0C)
#define IT83XX_PWM_PCSSGH       REG8(IT83XX_PWM_BASE+0x0D)
#define IT83XX_PWM_CR256PCSSG   REG8(IT83XX_PWM_BASE+0x0E)
#define IT83XX_PWM_PCSGR        REG8(IT83XX_PWM_BASE+0x0F)
#define IT83XX_PWM_CTR1M        REG8(IT83XX_PWM_BASE+0x10)
#define IT83XX_PWM_F1TLRR       REG8(IT83XX_PWM_BASE+0x1E)
#define IT83XX_PWM_F1TMRR       REG8(IT83XX_PWM_BASE+0x1F)
#define IT83XX_PWM_F2TLRR       REG8(IT83XX_PWM_BASE+0x20)
#define IT83XX_PWM_F2TMRR       REG8(IT83XX_PWM_BASE+0x21)
#define IT83XX_PWM_ZINTSCR      REG8(IT83XX_PWM_BASE+0x22)
#define IT83XX_PWM_ZTIER        REG8(IT83XX_PWM_BASE+0x23)
#define IT83XX_PWM_TSWCTLR      REG8(IT83XX_PWM_BASE+0x24)
#define IT83XX_PWM_C4CPRS       REG8(IT83XX_PWM_BASE+0x27)
#define IT83XX_PWM_C4MCPRS      REG8(IT83XX_PWM_BASE+0x28)
#define IT83XX_PWM_C6CPRS       REG8(IT83XX_PWM_BASE+0x2B)
#define IT83XX_PWM_C6MCPRS      REG8(IT83XX_PWM_BASE+0x2C)
#define IT83XX_PWM_C7CPRS       REG8(IT83XX_PWM_BASE+0x2D)
#define IT83XX_PWM_C7MCPRS      REG8(IT83XX_PWM_BASE+0x2E)
#define IT83XX_PWM_CLK6MSEL     REG8(IT83XX_PWM_BASE+0x40)
#define IT83XX_PWM_CTR1         REG8(IT83XX_PWM_BASE+0x41)
#define IT83XX_PWM_CTR2         REG8(IT83XX_PWM_BASE+0x42)
#define IT83XX_PWM_CTR3         REG8(IT83XX_PWM_BASE+0x43)
#define IT83XX_PWM_PWM5TOCTRL   REG8(IT83XX_PWM_BASE+0x44)
#define IT83XX_PWM_CFLRR        REG8(IT83XX_PWM_BASE+0x45)
#define IT83XX_PWM_CFMRR        REG8(IT83XX_PWM_BASE+0x46)
#define IT83XX_PWM_CFINTCTRL    REG8(IT83XX_PWM_BASE+0x47)
#define IT83XX_PWM_TSWCTRL      REG8(IT83XX_PWM_BASE+0x48)
#define IT83XX_PWM_PWMODENR     REG8(IT83XX_PWM_BASE+0x49)

/* Analog to Digital Converter (ADC) */
#define IT83XX_ADC_BASE   0x00F01900

#define IT83XX_ADC_ADCSTS       REG8(IT83XX_ADC_BASE+0x00)
#define IT83XX_ADC_ADCCFG       REG8(IT83XX_ADC_BASE+0x01)
#define IT83XX_ADC_ADCCTL       REG8(IT83XX_ADC_BASE+0x02)
#define IT83XX_ADC_ADCGCR       REG8(IT83XX_ADC_BASE+0x03)
#define IT83XX_ADC_DBKEN            BIT(7) /* ADC data buffer keep enable. */
#define IT83XX_ADC_VCH0CTL      REG8(IT83XX_ADC_BASE+0x04)
#define IT83XX_ADC_KDCTL        REG8(IT83XX_ADC_BASE+0x05)
#define IT83XX_ADC_AHCE             BIT(7)
#define IT83XX_ADC_VCH1CTL      REG8(IT83XX_ADC_BASE+0x06)
#define IT83XX_ADC_VCH1DATL     REG8(IT83XX_ADC_BASE+0x07)
#define IT83XX_ADC_VCH1DATM     REG8(IT83XX_ADC_BASE+0x08)
#define IT83XX_ADC_VCH2CTL      REG8(IT83XX_ADC_BASE+0x09)
#define IT83XX_ADC_VCH2DATL     REG8(IT83XX_ADC_BASE+0x0A)
#define IT83XX_ADC_VCH2DATM     REG8(IT83XX_ADC_BASE+0x0B)
#define IT83XX_ADC_VCH3CTL      REG8(IT83XX_ADC_BASE+0x0C)
#define IT83XX_ADC_VCH3DATL     REG8(IT83XX_ADC_BASE+0x0D)
#define IT83XX_ADC_VCH3DATM     REG8(IT83XX_ADC_BASE+0x0E)
#define IT83XX_ADC_VHSCDBL      REG8(IT83XX_ADC_BASE+0x14)
#define IT83XX_ADC_VHSCDBM      REG8(IT83XX_ADC_BASE+0x15)
#define IT83XX_ADC_VCH0DATL     REG8(IT83XX_ADC_BASE+0x18)
#define IT83XX_ADC_VCH0DATM     REG8(IT83XX_ADC_BASE+0x19)
#define IT83XX_ADC_VHSGCDBL     REG8(IT83XX_ADC_BASE+0x1C)
#define IT83XX_ADC_VHSGCDBM     REG8(IT83XX_ADC_BASE+0x1D)
#define IT83XX_ADC_ADCSAR       REG8(IT83XX_ADC_BASE+0x32)
#define IT83XX_ADC_VCMPSCP      REG8(IT83XX_ADC_BASE+0x37)
#define IT83XX_ADC_VCH4CTL      REG8(IT83XX_ADC_BASE+0x38)
#define IT83XX_ADC_VCH4DATM     REG8(IT83XX_ADC_BASE+0x39)
#define IT83XX_ADC_VCH4DATL     REG8(IT83XX_ADC_BASE+0x3A)
#define IT83XX_ADC_VCH5CTL      REG8(IT83XX_ADC_BASE+0x3B)
#define IT83XX_ADC_VCH5DATM     REG8(IT83XX_ADC_BASE+0x3C)
#define IT83XX_ADC_VCH5DATL     REG8(IT83XX_ADC_BASE+0x3D)
#define IT83XX_ADC_VCH6CTL      REG8(IT83XX_ADC_BASE+0x3E)
#define IT83XX_ADC_VCH6DATM     REG8(IT83XX_ADC_BASE+0x3F)
#define IT83XX_ADC_VCH6DATL     REG8(IT83XX_ADC_BASE+0x40)
#define IT83XX_ADC_VCH7CTL      REG8(IT83XX_ADC_BASE+0x41)
#define IT83XX_ADC_VCH7DATM     REG8(IT83XX_ADC_BASE+0x42)
#define IT83XX_ADC_VCH7DATL     REG8(IT83XX_ADC_BASE+0x43)
#define IT83XX_ADC_ADCDVSTS     REG8(IT83XX_ADC_BASE+0x44)
#define IT83XX_ADC_VCMPSTS      REG8(IT83XX_ADC_BASE+0x45)
#define IT83XX_ADC_VCMP0CTL     REG8(IT83XX_ADC_BASE+0x46)
#define ADC_VCMP_CMPEN              BIT(7)
#define ADC_VCMP_CMPINTEN           BIT(6)
#define ADC_VCMP_GREATER_THRESHOLD  BIT(5)
#define ADC_VCMP_EDGE_TRIGGER       BIT(4)
#define ADC_VCMP_GPIO_ACTIVE_LOW    BIT(3)
#define IT83XX_ADC_CMP0THRDATM  REG8(IT83XX_ADC_BASE+0x47)
#define IT83XX_ADC_CMP0THRDATL  REG8(IT83XX_ADC_BASE+0x48)
#define IT83XX_ADC_VCMP1CTL     REG8(IT83XX_ADC_BASE+0x49)
#define IT83XX_ADC_CMP1THRDATM  REG8(IT83XX_ADC_BASE+0x4A)
#define IT83XX_ADC_CMP1THRDATL  REG8(IT83XX_ADC_BASE+0x4B)
#define IT83XX_ADC_VCMP2CTL     REG8(IT83XX_ADC_BASE+0x4C)
#define IT83XX_ADC_CMP2THRDATM  REG8(IT83XX_ADC_BASE+0x4D)
#define IT83XX_ADC_CMP2THRDATL  REG8(IT83XX_ADC_BASE+0x4E)
#define IT83XX_ADC_VCH13CTL     REG8(IT83XX_ADC_BASE+0x60)
#define IT83XX_ADC_VCH13DATM    REG8(IT83XX_ADC_BASE+0x61)
#define IT83XX_ADC_VCH13DATL    REG8(IT83XX_ADC_BASE+0x62)
#define IT83XX_ADC_VCH14CTL     REG8(IT83XX_ADC_BASE+0x63)
#define IT83XX_ADC_VCH14DATM    REG8(IT83XX_ADC_BASE+0x64)
#define IT83XX_ADC_VCH14DATL    REG8(IT83XX_ADC_BASE+0x65)
#define IT83XX_ADC_VCH15CTL     REG8(IT83XX_ADC_BASE+0x66)
#define IT83XX_ADC_VCH15DATM    REG8(IT83XX_ADC_BASE+0x67)
#define IT83XX_ADC_VCH15DATL    REG8(IT83XX_ADC_BASE+0x68)
#define IT83XX_ADC_VCH16CTL     REG8(IT83XX_ADC_BASE+0x69)
#define IT83XX_ADC_VCH16DATM    REG8(IT83XX_ADC_BASE+0x6A)
#define IT83XX_ADC_VCH16DATL    REG8(IT83XX_ADC_BASE+0x6B)
#define IT83XX_ADC_ADCDVSTS2    REG8(IT83XX_ADC_BASE+0x6C)
#define IT83XX_ADC_VCMPSTS2     REG8(IT83XX_ADC_BASE+0x6D)
#define IT83XX_ADC_VCMP3CTL     REG8(IT83XX_ADC_BASE+0x6E)
#define IT83XX_ADC_CMP3THRDATM  REG8(IT83XX_ADC_BASE+0x6F)
#define IT83XX_ADC_CMP3THRDATL  REG8(IT83XX_ADC_BASE+0x70)
#define IT83XX_ADC_VCMP4CTL     REG8(IT83XX_ADC_BASE+0x71)
#define IT83XX_ADC_CMP4THRDATM  REG8(IT83XX_ADC_BASE+0x72)
#define IT83XX_ADC_CMP4THRDATL  REG8(IT83XX_ADC_BASE+0x73)
#define IT83XX_ADC_VCMP5CTL     REG8(IT83XX_ADC_BASE+0x74)
#define IT83XX_ADC_CMP5THRDATM  REG8(IT83XX_ADC_BASE+0x75)
#define IT83XX_ADC_CMP5THRDATL  REG8(IT83XX_ADC_BASE+0x76)
#define IT83XX_ADC_VCMP0CSELM   REG8(IT83XX_ADC_BASE+0x77)
#define ADC_VCMP_VCMPCSELM          BIT(0)
#define IT83XX_ADC_VCMP1CSELM   REG8(IT83XX_ADC_BASE+0x78)
#define IT83XX_ADC_VCMP2CSELM   REG8(IT83XX_ADC_BASE+0x79)
#define IT83XX_ADC_VCMP3CSELM   REG8(IT83XX_ADC_BASE+0x7A)
#define IT83XX_ADC_VCMP4CSELM   REG8(IT83XX_ADC_BASE+0x7B)
#define IT83XX_ADC_VCMP5CSELM   REG8(IT83XX_ADC_BASE+0x7C)

/* Digital to Analog Converter (DAC) */
#define IT83XX_DAC_BASE   0x00F01A00

#define IT83XX_DAC_DACPDREG     REG8(IT83XX_DAC_BASE+0x01)
#define IT83XX_DAC_POWDN(ch)    BIT(ch)
#define IT83XX_DAC_DACDAT(ch)   REG8(IT83XX_DAC_BASE+0x02+ch)

/* Keyboard Controller (KBC) */
#define IT83XX_KBC_BASE   0x00F01300

#define IT83XX_KBC_KBHICR       REG8(IT83XX_KBC_BASE+0x00)
#define IT83XX_KBC_KBIRQR       REG8(IT83XX_KBC_BASE+0x02)
#define IT83XX_KBC_KBHISR       REG8(IT83XX_KBC_BASE+0x04)
#define IT83XX_KBC_KBHIKDOR     REG8(IT83XX_KBC_BASE+0x06)
#define IT83XX_KBC_KBHIMDOR     REG8(IT83XX_KBC_BASE+0x08)
#define IT83XX_KBC_KBHIDIR      REG8(IT83XX_KBC_BASE+0x0A)

/* Power Management Channel (PMC) */
#define IT83XX_PMC_BASE   0x00F01500

#define IT83XX_PMC_PM1STS       REG8(IT83XX_PMC_BASE+0x00)
#define IT83XX_PMC_PM1DO        REG8(IT83XX_PMC_BASE+0x01)
#define IT83XX_PMC_PM1DOSCI     REG8(IT83XX_PMC_BASE+0x02)
#define IT83XX_PMC_PM1DOSMI     REG8(IT83XX_PMC_BASE+0x03)
#define IT83XX_PMC_PM1DI        REG8(IT83XX_PMC_BASE+0x04)
#define IT83XX_PMC_PM1DISCI     REG8(IT83XX_PMC_BASE+0x05)
#define IT83XX_PMC_PM1CTL       REG8(IT83XX_PMC_BASE+0x06)
#define IT83XX_PMC_PM1IC        REG8(IT83XX_PMC_BASE+0x07)
#define IT83XX_PMC_PM1IE        REG8(IT83XX_PMC_BASE+0x08)
#define IT83XX_PMC_PM2STS       REG8(IT83XX_PMC_BASE+0x10)
#define IT83XX_PMC_PM2DO        REG8(IT83XX_PMC_BASE+0x11)
#define IT83XX_PMC_PM2DOSCI     REG8(IT83XX_PMC_BASE+0x12)
#define IT83XX_PMC_PM2DOSMI     REG8(IT83XX_PMC_BASE+0x13)
#define IT83XX_PMC_PM2DI        REG8(IT83XX_PMC_BASE+0x14)
#define IT83XX_PMC_PM2DISCI     REG8(IT83XX_PMC_BASE+0x15)
#define IT83XX_PMC_PM2CTL       REG8(IT83XX_PMC_BASE+0x16)
#define IT83XX_PMC_PM2IC        REG8(IT83XX_PMC_BASE+0x17)
#define IT83XX_PMC_PM2IE        REG8(IT83XX_PMC_BASE+0x18)
#define IT83XX_PMC_PM3STS       REG8(IT83XX_PMC_BASE+0x20)
#define IT83XX_PMC_PM3DO        REG8(IT83XX_PMC_BASE+0x21)
#define IT83XX_PMC_PM3DI        REG8(IT83XX_PMC_BASE+0x22)
#define IT83XX_PMC_PM3CTL       REG8(IT83XX_PMC_BASE+0x23)
#define IT83XX_PMC_PM3IC        REG8(IT83XX_PMC_BASE+0x24)
#define IT83XX_PMC_PM3IE        REG8(IT83XX_PMC_BASE+0x25)
#define IT83XX_PMC_PM4STS       REG8(IT83XX_PMC_BASE+0x30)
#define IT83XX_PMC_PM4DO        REG8(IT83XX_PMC_BASE+0x31)
#define IT83XX_PMC_PM4DI        REG8(IT83XX_PMC_BASE+0x32)
#define IT83XX_PMC_PM4CTL       REG8(IT83XX_PMC_BASE+0x33)
#define IT83XX_PMC_PM4IC        REG8(IT83XX_PMC_BASE+0x34)
#define IT83XX_PMC_PM4IE        REG8(IT83XX_PMC_BASE+0x35)
#define IT83XX_PMC_PM5STS       REG8(IT83XX_PMC_BASE+0x40)
#define IT83XX_PMC_PM5DO        REG8(IT83XX_PMC_BASE+0x41)
#define IT83XX_PMC_PM5DI        REG8(IT83XX_PMC_BASE+0x42)
#define IT83XX_PMC_PM5CTL       REG8(IT83XX_PMC_BASE+0x43)
#define IT83XX_PMC_PM5IC        REG8(IT83XX_PMC_BASE+0x44)
#define IT83XX_PMC_PM5IE        REG8(IT83XX_PMC_BASE+0x45)
#define IT83XX_PMC_MBXCTRL      REG8(IT83XX_PMC_BASE+0x19)
#define IT83XX_PMC_MBXEC_00     REG8(IT83XX_PMC_BASE+0xF0)
#define IT83XX_PMC_MBXEC_01     REG8(IT83XX_PMC_BASE+0xF1)
#define IT83XX_PMC_MBXEC_02     REG8(IT83XX_PMC_BASE+0xF2)
#define IT83XX_PMC_MBXEC_03     REG8(IT83XX_PMC_BASE+0xF3)
#define IT83XX_PMC_MBXEC_04     REG8(IT83XX_PMC_BASE+0xF4)
#define IT83XX_PMC_MBXEC_05     REG8(IT83XX_PMC_BASE+0xF5)
#define IT83XX_PMC_MBXEC_06     REG8(IT83XX_PMC_BASE+0xF6)
#define IT83XX_PMC_MBXEC_07     REG8(IT83XX_PMC_BASE+0xF7)
#define IT83XX_PMC_MBXEC_08     REG8(IT83XX_PMC_BASE+0xF8)
#define IT83XX_PMC_MBXEC_09     REG8(IT83XX_PMC_BASE+0xF9)
#define IT83XX_PMC_MBXEC_10     REG8(IT83XX_PMC_BASE+0xFA)
#define IT83XX_PMC_MBXEC_11     REG8(IT83XX_PMC_BASE+0xFB)
#define IT83XX_PMC_MBXEC_12     REG8(IT83XX_PMC_BASE+0xFC)
#define IT83XX_PMC_MBXEC_13     REG8(IT83XX_PMC_BASE+0xFD)
#define IT83XX_PMC_MBXEC_14     REG8(IT83XX_PMC_BASE+0xFE)
#define IT83XX_PMC_MBXEC_15     REG8(IT83XX_PMC_BASE+0xFF)
#define IT83XX_PMC_PMSTS(ch)    REG8(IT83XX_PMC_BASE + 0x00 + (ch << 4))
#define IT83XX_PMC_PMDO(ch)     REG8(IT83XX_PMC_BASE + 0x01 + (ch << 4))
#define IT83XX_PMC_PMDI(ch)     \
REG8(IT83XX_PMC_BASE + (ch > LPC_PM2 ? 2 : 4) + (ch << 4))
#define IT83XX_PMC_PMCTL(ch)     \
REG8(IT83XX_PMC_BASE + (ch > LPC_PM2 ? 3 : 6) + (ch << 4))
#define IT83XX_PMC_PMIE(ch)     \
REG8(IT83XX_PMC_BASE + (ch > LPC_PM2 ? 5 : 8) + (ch << 4))

/* Keyboard Matrix Scan control (KBS) */
#define IT83XX_KBS_BASE   0x00F01D00

#define IT83XX_KBS_KSOL         REG8(IT83XX_KBS_BASE+0x00)
#define IT83XX_KBS_KSOH1        REG8(IT83XX_KBS_BASE+0x01)
#define IT83XX_KBS_KSOCTRL      REG8(IT83XX_KBS_BASE+0x02)
#define IT83XX_KBS_KSOH2        REG8(IT83XX_KBS_BASE+0x03)
#define IT83XX_KBS_KSI          REG8(IT83XX_KBS_BASE+0x04)
#define IT83XX_KBS_KSICTRL      REG8(IT83XX_KBS_BASE+0x05)
#define IT83XX_KBS_KSIGCTRL     REG8(IT83XX_KBS_BASE+0x06)
#define IT83XX_KBS_KSIGOEN      REG8(IT83XX_KBS_BASE+0x07)
#define IT83XX_KBS_KSIGDAT      REG8(IT83XX_KBS_BASE+0x08)
#define IT83XX_KBS_KSIGDMRR     REG8(IT83XX_KBS_BASE+0x09)
#define IT83XX_KBS_KSOHGCTRL    REG8(IT83XX_KBS_BASE+0x0A)
#define IT83XX_KBS_KSOHGOEN     REG8(IT83XX_KBS_BASE+0x0B)
#define IT83XX_KBS_KSOHGDMRR    REG8(IT83XX_KBS_BASE+0x0C)
#define IT83XX_KBS_KSOLGCTRL    REG8(IT83XX_KBS_BASE+0x0D)
#define IT83XX_KBS_KSOLGOEN     REG8(IT83XX_KBS_BASE+0x0E)
#define IT83XX_KBS_KSOLGDMRR    REG8(IT83XX_KBS_BASE+0x0F)
#define IT83XX_KBS_KSO0LSDR     REG8(IT83XX_KBS_BASE+0x10)
#define IT83XX_KBS_KSO1LSDR     REG8(IT83XX_KBS_BASE+0x11)
#define IT83XX_KBS_KSO2LSDR     REG8(IT83XX_KBS_BASE+0x12)
#define IT83XX_KBS_KSO3LSDR     REG8(IT83XX_KBS_BASE+0x13)
#define IT83XX_KBS_KSO4LSDR     REG8(IT83XX_KBS_BASE+0x14)
#define IT83XX_KBS_KSO5LSDR     REG8(IT83XX_KBS_BASE+0x15)
#define IT83XX_KBS_KSO6LSDR     REG8(IT83XX_KBS_BASE+0x16)
#define IT83XX_KBS_KSO7LSDR     REG8(IT83XX_KBS_BASE+0x17)
#define IT83XX_KBS_KSO8LSDR     REG8(IT83XX_KBS_BASE+0x18)
#define IT83XX_KBS_KSO9LSDR     REG8(IT83XX_KBS_BASE+0x19)
#define IT83XX_KBS_KSO10LSDR    REG8(IT83XX_KBS_BASE+0x1A)
#define IT83XX_KBS_KSO11LSDR    REG8(IT83XX_KBS_BASE+0x1B)
#define IT83XX_KBS_KSO12LSDR    REG8(IT83XX_KBS_BASE+0x1C)
#define IT83XX_KBS_KSO13LSDR    REG8(IT83XX_KBS_BASE+0x1D)
#define IT83XX_KBS_KSO14LSDR    REG8(IT83XX_KBS_BASE+0x1E)
#define IT83XX_KBS_KSO15LSDR    REG8(IT83XX_KBS_BASE+0x1F)
#define IT83XX_KBS_KSO16LSDR    REG8(IT83XX_KBS_BASE+0x20)
#define IT83XX_KBS_KSO17LSDR    REG8(IT83XX_KBS_BASE+0x21)
#define IT83XX_KBS_SDC1R        REG8(IT83XX_KBS_BASE+0x22)
#define IT83XX_KBS_SDC2R        REG8(IT83XX_KBS_BASE+0x23)
#define IT83XX_KBS_SDC3R        REG8(IT83XX_KBS_BASE+0x24)
#define IT83XX_KBS_SDSR         REG8(IT83XX_KBS_BASE+0x25)
#define IT83XX_KBS_KSIGPODR     REG8(IT83XX_KBS_BASE+0x26)
#define IT83XX_KBS_KSOHGPODR    REG8(IT83XX_KBS_BASE+0x27)
#define IT83XX_KBS_KSOLGPODR    REG8(IT83XX_KBS_BASE+0x28)

/* Shared Memory Flash Interface Bridge (SMFI) */
#define IT83XX_SMFI_BASE  0x00F01000

#define IT83XX_SMFI_SMECCS      REG8(IT83XX_SMFI_BASE+0x20)
#define IT83XX_SMFI_MASK_HOSTWA 	BIT(5)
#define IT83XX_SMFI_HRAMWC      REG8(IT83XX_SMFI_BASE+0x5A)
#define IT83XX_SMFI_HRAMW0BA    REG8(IT83XX_SMFI_BASE+0x5B)
#define IT83XX_SMFI_HRAMW1BA    REG8(IT83XX_SMFI_BASE+0x5C)
#define IT83XX_SMFI_HRAMW0AAS   REG8(IT83XX_SMFI_BASE+0x5D)
#define IT83XX_SMFI_HRAMW1AAS   REG8(IT83XX_SMFI_BASE+0x5E)
#define IT83XX_SMFI_HRAMW2BA    REG8(IT83XX_SMFI_BASE+0x76)
#define IT83XX_SMFI_HRAMW3BA    REG8(IT83XX_SMFI_BASE+0x77)
#define IT83XX_SMFI_HRAMW2AAS   REG8(IT83XX_SMFI_BASE+0x78)
#define IT83XX_SMFI_HRAMW3AAS   REG8(IT83XX_SMFI_BASE+0x79)
#define IT83XX_SMFI_H2RAMECSIE  REG8(IT83XX_SMFI_BASE+0x7A)
#define IT83XX_SMFI_H2RAMECSA   REG8(IT83XX_SMFI_BASE+0x7B)
#define IT83XX_SMFI_H2RAMHSS    REG8(IT83XX_SMFI_BASE+0x7C)
#define IT83XX_SMFI_ECINDAR0    REG8(IT83XX_SMFI_BASE+0x3B)
#define IT83XX_SMFI_ECINDAR1    REG8(IT83XX_SMFI_BASE+0x3C)
#define IT83XX_SMFI_ECINDAR2    REG8(IT83XX_SMFI_BASE+0x3D)
#define IT83XX_SMFI_ECINDAR3    REG8(IT83XX_SMFI_BASE+0x3E)
#define EC_INDIRECT_READ_INTERNAL_FLASH BIT(6)
#define IT83XX_SMFI_ECINDDR     REG8(IT83XX_SMFI_BASE+0x3F)
#define IT83XX_SMFI_SCAR0L      REG8(IT83XX_SMFI_BASE+0x40)
#define IT83XX_SMFI_SCAR0M      REG8(IT83XX_SMFI_BASE+0x41)
#define IT83XX_SMFI_SCAR0H      REG8(IT83XX_SMFI_BASE+0x42)
#define IT83XX_SMFI_SCAR2L      REG8(IT83XX_SMFI_BASE+0x46)
#define IT83XX_SMFI_SCAR2M      REG8(IT83XX_SMFI_BASE+0x47)
#define IT83XX_SMFI_SCAR2H      REG8(IT83XX_SMFI_BASE+0x48)
#define IT83XX_SMFI_FLHCTRL3R   REG8(IT83XX_SMFI_BASE+0x63)
#define IT83XX_SMFI_STCDMACR    REG8(IT83XX_SMFI_BASE+0x80)
#define IT83XX_SMFI_FLHCTRL6R   REG8(IT83XX_SMFI_BASE+0xA2)
/* Enable EC-indirect page program command */
#define IT83XX_SMFI_MASK_ECINDPP BIT(3)

/* Serial Peripheral Interface (SSPI) */
#define IT83XX_SSPI_BASE  0x00F02600

#define IT83XX_SSPI_SPIDATA     REG8(IT83XX_SSPI_BASE+0x00)
#define IT83XX_SSPI_SPICTRL1    REG8(IT83XX_SSPI_BASE+0x01)
#define IT83XX_SSPI_SPICTRL2    REG8(IT83XX_SSPI_BASE+0x02)
#define IT83XX_SSPI_SPISTS      REG8(IT83XX_SSPI_BASE+0x03)
#define IT83XX_SSPI_SPICTRL3    REG8(IT83XX_SSPI_BASE+0x04)

/* Serial Peripheral Interface (SPI) */
#define IT83XX_SPI_BASE  0x00F03A00

#define IT83XX_SPI_SPISGCR      REG8(IT83XX_SPI_BASE+0x00)
#define IT83XX_SPI_SPISCEN         BIT(0)
#define IT83XX_SPI_TXRXFAR      REG8(IT83XX_SPI_BASE+0x01)
#define IT83XX_SPI_CPURXF2A        BIT(4)
#define IT83XX_SPI_CPURXF1A        BIT(3)
#define IT83XX_SPI_CPUTFA          BIT(1)
#define IT83XX_SPI_TXFCR        REG8(IT83XX_SPI_BASE+0x02)
#define IT83XX_SPI_TXFCMR          BIT(2)
#define IT83XX_SPI_TXFR            BIT(1)
#define IT83XX_SPI_TXFS            BIT(0)
#define IT83XX_SPI_GCR2         REG8(IT83XX_SPI_BASE+0x03)
#define IT83XX_SPI_RXF2OC          BIT(4)
#define IT83XX_SPI_RXF1OC          BIT(3)
#define IT83XX_SPI_RXFAR           BIT(0)
#define IT83XX_SPI_IMR          REG8(IT83XX_SPI_BASE+0x04)
#define IT83XX_SPI_RX_FIFO_FULL    BIT(7)
#define IT83XX_SPI_RX_REACH        BIT(5)
#define IT83XX_SPI_EDIM            BIT(2)
#define IT83XX_SPI_ISR          REG8(IT83XX_SPI_BASE+0x05)
#define IT83XX_SPI_TXFSR        REG8(IT83XX_SPI_BASE+0x06)
#define IT83XX_SPI_ENDDETECTINT    BIT(2)
#define IT83XX_SPI_RXFSR        REG8(IT83XX_SPI_BASE+0x07)
#define IT83XX_SPI_RXFFSM          (BIT(4) | BIT(3))
#define IT83XX_SPI_RXF2FS          BIT(2)
#define IT83XX_SPI_RXF1FS          BIT(1)
#ifdef CHIP_VARIANT_IT83202BX
#define IT83XX_SPI_SPISRDR      REG8(IT83XX_SPI_BASE+0x08)
#else
#define IT83XX_SPI_SPISRDR      REG8(IT83XX_SPI_BASE+0x0b)
#endif
#define IT83XX_SPI_CPUWTFDB0    REG32(IT83XX_SPI_BASE+0x08)
#define IT83XX_SPI_FCR          REG8(IT83XX_SPI_BASE+0x09)
#define IT83XX_SPI_SPISRTXF        BIT(2)
#define IT83XX_SPI_RXFR            BIT(1)
#define IT83XX_SPI_RXFCMR          BIT(0)
#define IT83XX_SPI_RXFRDRB0     REG32(IT83XX_SPI_BASE+0x0C)
#define IT83XX_SPI_FTCB0R       REG8(IT83XX_SPI_BASE+0x18)
#define IT83XX_SPI_FTCB1R       REG8(IT83XX_SPI_BASE+0x19)
#define IT83XX_SPI_TCCB0        REG8(IT83XX_SPI_BASE+0x1A)
#define IT83XX_SPI_TCCB1        REG8(IT83XX_SPI_BASE+0x1B)
#define IT83XX_SPI_HPR2         REG8(IT83XX_SPI_BASE+0x1E)
#define IT83XX_SPI_EMMCBMR      REG8(IT83XX_SPI_BASE+0x21)
#define IT83XX_SPI_EMMCABM         BIT(1) /* eMMC Alternative Boot Mode */
#define IT83XX_SPI_RX_VLISMR    REG8(IT83XX_SPI_BASE+0x26)
#define IT83XX_SPI_RVLIM           BIT(0)
#define IT83XX_SPI_RX_VLISR     REG8(IT83XX_SPI_BASE+0x27)
#define IT83XX_SPI_RVLI            BIT(0)

/* Platform Environment Control Interface (PECI) */
#define IT83XX_PECI_BASE  0x00F02C00

#define IT83XX_PECI_HOSTAR      REG8(IT83XX_PECI_BASE+0x00)
#define IT83XX_PECI_HOCTLR      REG8(IT83XX_PECI_BASE+0x01)
#define IT83XX_PECI_HOCMDR      REG8(IT83XX_PECI_BASE+0x02)
#define IT83XX_PECI_HOTRADDR    REG8(IT83XX_PECI_BASE+0x03)
#define IT83XX_PECI_HOWRLR      REG8(IT83XX_PECI_BASE+0x04)
#define IT83XX_PECI_HORDLR      REG8(IT83XX_PECI_BASE+0x05)
#define IT83XX_PECI_HOWRDR      REG8(IT83XX_PECI_BASE+0x06)
#define IT83XX_PECI_HORDDR      REG8(IT83XX_PECI_BASE+0x07)
#define IT83XX_PECI_HOCTL2R     REG8(IT83XX_PECI_BASE+0x08)
#define IT83XX_PECI_RWFCSV      REG8(IT83XX_PECI_BASE+0x09)
#define IT83XX_PECI_RRFCSV      REG8(IT83XX_PECI_BASE+0x0A)
#define IT83XX_PECI_WFCSV       REG8(IT83XX_PECI_BASE+0x0B)
#define IT83XX_PECI_RFCSV       REG8(IT83XX_PECI_BASE+0x0C)
#define IT83XX_PECI_AWFCSV      REG8(IT83XX_PECI_BASE+0x0D)
#define IT83XX_PECI_PADCTLR     REG8(IT83XX_PECI_BASE+0x0E)

/*
 * The count number of the counter for 25 ms register.
 * The 25 ms register is calculated by (count number *1.024 kHz).
 */
#define I2C_CLK_LOW_TIMEOUT  255 /* ~=249 ms */

/* SMBus/I2C Interface (SMB/I2C) */
#define IT83XX_SMB_BASE   0x00F01C00

#define IT83XX_SMB_4P7USL       REG8(IT83XX_SMB_BASE+0x00)
#define IT83XX_SMB_4P0USL       REG8(IT83XX_SMB_BASE+0x01)
#define IT83XX_SMB_300NS        REG8(IT83XX_SMB_BASE+0x02)
#define IT83XX_SMB_250NS        REG8(IT83XX_SMB_BASE+0x03)
#define IT83XX_SMB_25MS         REG8(IT83XX_SMB_BASE+0x04)
#define IT83XX_SMB_45P3USL      REG8(IT83XX_SMB_BASE+0x05)
#define IT83XX_SMB_45P3USH      REG8(IT83XX_SMB_BASE+0x06)
#define IT83XX_SMB_4P7A4P0H     REG8(IT83XX_SMB_BASE+0x07)
#define IT83XX_SMB_SLVISELR     REG8(IT83XX_SMB_BASE+0x08)
#define IT83XX_SMB_SCLKTS(ch)   REG8(IT83XX_SMB_BASE+0x09+ch)
#define IT83XX_SMB_CHSEF        REG8(IT83XX_SMB_BASE+0x11)
#define IT83XX_SMB_CHSAB        REG8(IT83XX_SMB_BASE+0x20)
#define IT83XX_SMB_CHSCD        REG8(IT83XX_SMB_BASE+0x21)
#define IT83XX_SMB_HOSTA(ch)    REG8(IT83XX_SMB_BASE+0x40+(ch << 6))
#define IT83XX_SMB_HOCTL(ch)    REG8(IT83XX_SMB_BASE+0x41+(ch << 6))
#define IT83XX_SMB_HOCMD(ch)    REG8(IT83XX_SMB_BASE+0x42+(ch << 6))
#define IT83XX_SMB_TRASLA(ch)   REG8(IT83XX_SMB_BASE+0x43+(ch << 6))
#define IT83XX_SMB_D0REG(ch)    REG8(IT83XX_SMB_BASE+0x44+(ch << 6))
#define IT83XX_SMB_D1REG(ch)    REG8(IT83XX_SMB_BASE+0x45+(ch << 6))
#define IT83XX_SMB_HOBDB(ch)    REG8(IT83XX_SMB_BASE+0x46+(ch << 6))
#define IT83XX_SMB_PECERC(ch)   REG8(IT83XX_SMB_BASE+0x47+(ch << 6))
#define IT83XX_SMB_SMBPCTL(ch)  REG8(IT83XX_SMB_BASE+0x4A+(ch << 6))
#define IT83XX_SMB_HOCTL2(ch)   REG8(IT83XX_SMB_BASE+0x50+(ch << 6))
#define IT83XX_SMB_SLVEN        (1 << 5)
#define IT83XX_SMB_RESLADR      REG8(IT83XX_SMB_BASE+0x48)
#define IT83XX_SMB_SLDA         REG8(IT83XX_SMB_BASE+0x49)
#define IT83XX_SMB_SLSTA        REG8(IT83XX_SMB_BASE+0x4B)
#define IT83XX_SMB_SPDS         (1 << 5)
#define IT83XX_SMB_RCS          (1 << 3)
#define IT83XX_SMB_STS          (1 << 2)
#define IT83XX_SMB_SDS          (1 << 1)
#define IT83XX_SMB_SICR         REG8(IT83XX_SMB_BASE+0x4C)
#define IT83XX_SMB_RESLADR2     REG8(IT83XX_SMB_BASE+0x51)
#define IT83XX_SMB_ENADDR2      (1 << 7)
#define IT83XX_SMB_SFFCTL       REG8(IT83XX_SMB_BASE+0x55)
#define IT83XX_SMB_HSAPE        BIT(1)
#define IT83XX_SMB_SAFE         (1 << 0)
#define IT83XX_SMB_SFFSTA       REG8(IT83XX_SMB_BASE+0x56)
#define IT83XX_SMB_SFFFULL      (1 << 6)

/* BRAM */
#define IT83XX_BRAM_BASE  0x00F02200

/* offset 0 ~ 0x7f */
#define IT83XX_BRAM_BANK0(i)    REG8(IT83XX_BRAM_BASE + i)
/* Battery backed RAM indices. */
enum bram_indices {
	/* reset flags uses 4 bytes */
	BRAM_IDX_RESET_FLAGS0 = 0,
	BRAM_IDX_RESET_FLAGS1 = 1,
	BRAM_IDX_RESET_FLAGS2 = 2,
	BRAM_IDX_RESET_FLAGS3 = 3,

	/* PD state data for CONFIG_USB_PD_DUAL_ROLE uses 1 byte per port */
	BRAM_IDX_PD0 =           4,
	BRAM_IDX_PD1 =           5,
	BRAM_IDX_PD2 =           6,

	/* index 7 is reserved */

	BRAM_IDX_SCRATCHPAD0  = 8,
	BRAM_IDX_SCRATCHPAD1  = 9,
	BRAM_IDX_SCRATCHPAD2  = 0xa,
	BRAM_IDX_SCRATCHPAD3  = 0xb,

	/* EC logs status */
	BRAM_IDX_EC_LOG_STATUS = 0xc,

	/* offset 0x0d ~ 0x1f are reserved for future use. */
#if defined(CONFIG_HOSTCMD_LPC) || defined(CONFIG_HOSTCMD_ESPI)
	/*
	 * offset 0x20 ~ 0x7b are reserved for future use.
	 * (apply to x86 platform)
	 */

	/* This field is used to indicate BRAM is valid or not. */
	BRAM_IDX_VALID_FLAGS0  = 0x7c,
	BRAM_IDX_VALID_FLAGS1  = 0x7d,
	BRAM_IDX_VALID_FLAGS2  = 0x7e,
	BRAM_IDX_VALID_FLAGS3  = 0x7f
	/* offset 0x7f is the end of BRAM bank 0. */
#else

	/* panic data uses 144 bytes (offset 0x20 ~ 0xaf) */
	BRAM_PANIC_DATA_START  = 0x20,
	BRAM_PANIC_DATA_END    = 0xaf,

	/* This field is used to indicate BRAM is valid or not. */
	BRAM_IDX_VALID_FLAGS0  = 0xbc,
	BRAM_IDX_VALID_FLAGS1  = 0xbd,
	BRAM_IDX_VALID_FLAGS2  = 0xbe,
	BRAM_IDX_VALID_FLAGS3  = 0xbf
	/* offset 0xbf is the end of BRAM bank 1. */
#endif
};
#define BRAM_RESET_FLAGS0       IT83XX_BRAM_BANK0(BRAM_IDX_RESET_FLAGS0)
#define BRAM_RESET_FLAGS1       IT83XX_BRAM_BANK0(BRAM_IDX_RESET_FLAGS1)
#define BRAM_RESET_FLAGS2       IT83XX_BRAM_BANK0(BRAM_IDX_RESET_FLAGS2)
#define BRAM_RESET_FLAGS3       IT83XX_BRAM_BANK0(BRAM_IDX_RESET_FLAGS3)

#define BRAM_SCRATCHPAD0        IT83XX_BRAM_BANK0(BRAM_IDX_SCRATCHPAD0)
#define BRAM_SCRATCHPAD1        IT83XX_BRAM_BANK0(BRAM_IDX_SCRATCHPAD1)
#define BRAM_SCRATCHPAD2        IT83XX_BRAM_BANK0(BRAM_IDX_SCRATCHPAD2)
#define BRAM_SCRATCHPAD3        IT83XX_BRAM_BANK0(BRAM_IDX_SCRATCHPAD3)

#define BRAM_EC_LOG_STATUS      IT83XX_BRAM_BANK0(BRAM_IDX_EC_LOG_STATUS)
enum bram_ec_logs_status {
	EC_LOG_SAVED_IN_FLASH = 1,
	EC_LOG_SAVED_IN_MEMORY
};

#define BRAM_VALID_FLAGS0       IT83XX_BRAM_BANK0(BRAM_IDX_VALID_FLAGS0)
#define BRAM_VALID_FLAGS1       IT83XX_BRAM_BANK0(BRAM_IDX_VALID_FLAGS1)
#define BRAM_VALID_FLAGS2       IT83XX_BRAM_BANK0(BRAM_IDX_VALID_FLAGS2)
#define BRAM_VALID_FLAGS3       IT83XX_BRAM_BANK0(BRAM_IDX_VALID_FLAGS3)

/*
 * These 128 bytes are use to latch port 80h data on x86 platform.
 * And they will be used to save panic data if the GPG1 reset mechanism
 * is enabled.
 */
#if defined(CONFIG_HOSTCMD_LPC) || defined(CONFIG_HOSTCMD_ESPI)
/* offset 0x80 ~ 0xbf */
#define IT83XX_BRAM_BANK1(i)    REG8(IT83XX_BRAM_BASE + 0x80 + i)
#else
/* Length of bram panic data */
#define BRAM_PANIC_LEN (BRAM_PANIC_DATA_END - BRAM_PANIC_DATA_START + 1)
#endif

/*
 * Enhanced SMBus/I2C Interface
 * Ch_D: 0x00F03680 , Ch_E: 0x00F03500 , Ch_F: 0x00F03580
 * Ch_D: ch = 0x03  , Ch_E: ch = 0x00  , Ch_F: ch = 0x01
 */
#define IT83XX_I2C_BASE   0x00F03500

#define IT83XX_I2C_DRR(ch)           REG8(IT83XX_I2C_BASE+0x00+(ch << 7))
#define IT83XX_I2C_PSR(ch)           REG8(IT83XX_I2C_BASE+0x01+(ch << 7))
#define IT83XX_I2C_HSPR(ch)          REG8(IT83XX_I2C_BASE+0x02+(ch << 7))
#define IT83XX_I2C_STR(ch)           REG8(IT83XX_I2C_BASE+0x03+(ch << 7))
#define IT83XX_I2C_BB                (1 << 5)
#define IT83XX_I2C_TIME_OUT          (1 << 3)
#define IT83XX_I2C_RW                (1 << 2)
#define IT83XX_I2C_INTPEND           (1 << 1)
#define IT83XX_I2C_DHTR(ch)          REG8(IT83XX_I2C_BASE+0x04+(ch << 7))
#define IT83XX_I2C_TOR(ch)           REG8(IT83XX_I2C_BASE+0x05+(ch << 7))
#define IT83XX_I2C_DTR(ch)           REG8(IT83XX_I2C_BASE+0x08+(ch << 7))
#define IT83XX_I2C_CTR(ch)           REG8(IT83XX_I2C_BASE+0x09+(ch << 7))
#define IT83XX_I2C_INTEN             (1 << 6)
#define IT83XX_I2C_MODE              (1 << 5)
#define IT83XX_I2C_STARST            (1 << 4)
#define IT83XX_I2C_ACK               (1 << 3)
#define IT83XX_I2C_HALT              (1 << 0)
#define IT83XX_I2C_CTR1(ch)          REG8(IT83XX_I2C_BASE+0x0A+(ch << 7))
#define IT83XX_I2C_COMQ_EN           (1 << 7)
#define IT83XX_I2C_MDL_EN            (1 << 1)
#define IT83XX_I2C_BYTE_CNT_L(ch)    REG8(IT83XX_I2C_BASE+0x0C+(ch << 7))
#define IT83XX_I2C_IRQ_ST(ch)        REG8(IT83XX_I2C_BASE+0x0D+(ch << 7))
#define IT83XX_I2C_IDW_CLR           (1 << 3)
#define IT83XX_I2C_IDR_CLR           (1 << 2)
#define IT83XX_I2C_SLVDATAFLG        (1 << 1)
#define IT83XX_I2C_P_CLR             (1 << 0)
#define IT83XX_I2C_IDR(ch)           REG8(IT83XX_I2C_BASE+0x06+(ch << 7))
#define IT83XX_I2C_TOS(ch)           REG8(IT83XX_I2C_BASE+0x07+(ch << 7))
#define IT83XX_I2C_CLK_STR           (1 << 7)
#define IT83XX_I2C_IDR2(ch)          REG8(IT83XX_I2C_BASE+0x1F+(ch << 7))
#define IT83XX_I2C_RAMHA(ch)         REG8(IT83XX_I2C_BASE+0x23+(ch << 7))
#define IT83XX_I2C_RAMLA(ch)         REG8(IT83XX_I2C_BASE+0x24+(ch << 7))
#define IT83XX_I2C_RAMHA2(ch)        REG8(IT83XX_I2C_BASE+0x2B+(ch << 7))
#define IT83XX_I2C_RAMLA2(ch)        REG8(IT83XX_I2C_BASE+0x2C+(ch << 7))
#define IT83XX_I2C_CMD_ADDH(ch)      REG8(IT83XX_I2C_BASE+0x25+(ch << 7))
#define IT83XX_I2C_CMD_ADDL(ch)      REG8(IT83XX_I2C_BASE+0x26+(ch << 7))
#define IT83XX_I2C_RAMH2A(ch)        REG8(IT83XX_I2C_BASE+0x50+(ch << 7))
#define IT83XX_I2C_CMD_ADDH2(ch)     REG8(IT83XX_I2C_BASE+0x52+(ch << 7))

enum i2c_channels {
	IT83XX_I2C_CH_A,  /* GPIO.B3/B4 */
	IT83XX_I2C_CH_B,  /* GPIO.C1/C2 */
	IT83XX_I2C_CH_C,  /* GPIO.F6/F7 or GPIO.C7/F7 */
	IT83XX_I2C_CH_D,  /* GPIO.H1/H2 */
	IT83XX_I2C_CH_E,  /* GPIO.E0/E7 */
	IT83XX_I2C_CH_F,  /* GPIO.A4/A5 (for util/iteflash) */
	IT83XX_I2C_PORT_COUNT,
};

#define USB_VID_ITE 0x048d

#define IT83XX_ESPI_BASE  0x00F03100

#define IT83XX_ESPI_GCAC1      REG8(IT83XX_ESPI_BASE+0x05)
#define IT83XX_ESPI_ESPCTRL0   REG8(IT83XX_ESPI_BASE+0x90)
#define IT83XX_ESPI_ESGCTRL0   REG8(IT83XX_ESPI_BASE+0xA0)
#define IT83XX_ESPI_ESGCTRL1   REG8(IT83XX_ESPI_BASE+0xA1)
#define IT83XX_ESPI_ESGCTRL2   REG8(IT83XX_ESPI_BASE+0xA2)

/* eSPI VW */
#define IT83XX_ESPI_VW_BASE  0x00F03200
#define IT83XX_ESPI_VWIDX(i)   REG8(IT83XX_ESPI_VW_BASE+(i))

#define VW_LEVEL_FIELD(f)          ((f) << 0)
#define VW_VALID_FIELD(f)          ((f) << 4)

#define ESPI_SYSTEM_EVENT_VW_IDX_2 0x2
#define VW_IDX_2_SLP_S3            BIT(0)
#define VW_IDX_2_SLP_S4            BIT(1)
#define VW_IDX_2_SLP_S5            BIT(2)

#define ESPI_SYSTEM_EVENT_VW_IDX_3 0x3
#define VW_IDX_3_SUS_STAT          BIT(0)
#define VW_IDX_3_PLTRST            BIT(1)
#define VW_IDX_3_OOB_RST_WARN      BIT(2)

#define ESPI_SYSTEM_EVENT_VW_IDX_4 0x4
#define VW_IDX_4_OOB_RST_ACK       BIT(0)
#define VW_IDX_4_WAKE              BIT(2)
#define VW_IDX_4_PME               BIT(3)

#define ESPI_SYSTEM_EVENT_VW_IDX_5 0x5
#define VW_IDX_5_SLAVE_BTLD_DONE   BIT(0)
#define VW_IDX_5_FATAL             BIT(1)
#define VW_IDX_5_NON_FATAL         BIT(2)
#define VW_IDX_5_SLAVE_BTLD_STATUS BIT(3)
#define VW_IDX_5_BTLD_STATUS_DONE  (VW_IDX_5_SLAVE_BTLD_DONE | \
					VW_IDX_5_SLAVE_BTLD_STATUS)

#define ESPI_SYSTEM_EVENT_VW_IDX_6 0x6
#define VW_IDX_6_SCI               BIT(0)
#define VW_IDX_6_SMI               BIT(1)
#define VW_IDX_6_RCIN              BIT(2)
#define VW_IDX_6_HOST_RST_ACK      BIT(3)

#define ESPI_SYSTEM_EVENT_VW_IDX_7 0x7
#define VW_IDX_7_HOST_RST_WARN     BIT(0)

#define ESPI_SYSTEM_EVENT_VW_IDX_40 0x40
#define VW_IDX_40_SUS_ACK           BIT(0)

#define ESPI_SYSTEM_EVENT_VW_IDX_41 0x41
#define VW_IDX_41_SUS_WARN          BIT(0)
#define VW_IDX_41_SUS_PWRDN_ACK     BIT(1)
#define VW_IDX_41_SLP_A             BIT(3)

#define ESPI_SYSTEM_EVENT_VW_IDX_42 0x42
#define VW_IDX_42_SLP_LAN           BIT(0)
#define VW_IDX_42_SLP_WLAN          BIT(1)

#define ESPI_SYSTEM_EVENT_VW_IDX_43 0x43
#define ESPI_SYSTEM_EVENT_VW_IDX_44 0x44
#define ESPI_SYSTEM_EVENT_VW_IDX_47 0x47

#define IT83XX_ESPI_VWCTRL0    REG8(IT83XX_ESPI_VW_BASE+0x90)
#define ESPI_INTERRUPT_EVENT_PUT_PC BIT(7)

#define IT83XX_ESPI_VWCTRL1    REG8(IT83XX_ESPI_VW_BASE+0x91)
#define IT83XX_ESPI_VWCTRL2    REG8(IT83XX_ESPI_VW_BASE+0x92)
#define IT83XX_ESPI_VWCTRL3    REG8(IT83XX_ESPI_VW_BASE+0x93)

/* eSPI Queue 0 */
#define IT83XX_ESPI_QUEUE_BASE  0x00F03300
/* PUT_PC data byte 0 - 63 */
#define IT83XX_ESPI_QUEUE_PUT_PC(i)  REG8(IT83XX_ESPI_QUEUE_BASE+(i))
/* PUT_OOB data byte 0 - 79 */
#define IT83XX_ESPI_QUEUE_PUT_OOB(i) REG8(IT83XX_ESPI_QUEUE_BASE+0x80+(i))

/* USB Controller */
#define IT83XX_USB_BASE   0x00F02F00

#define IT83XX_USB_P0MCR  REG8(IT83XX_USB_BASE+0xE4)
#define USB_DP_DM_PULL_DOWN_EN BIT(4)

/* Wake pin definitions, defined at board-level */
#ifndef CONFIG_HIBERNATE_WAKE_PINS_DYNAMIC
extern const enum gpio_signal hibernate_wake_pins[];
extern const int hibernate_wake_pins_used;
#else
extern enum gpio_signal hibernate_wake_pins[];
extern int hibernate_wake_pins_used;
#endif

/* --- MISC (not implemented yet) --- */

#define IT83XX_PS2_BASE   0x00F01700
#define IT83XX_EGPIO_BASE 0x00F02100
#define IT83XX_CIR_BASE   0x00F02300
#define IT83XX_DBGR_BASE  0x00F02500
#define IT83XX_OW_BASE    0x00F02A00
#define IT83XX_CEC_BASE   0x00F02E00

#endif /* __CROS_EC_REGISTERS_H */
