/* -*- Mode: c++ -*-
*******************************************************************************
*
* File:         elenco_fm_demo.cc
* Description:  
*
*******************************************************************************
*/

/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

//
// This is a demo of a complete FM reception path.
//
// We start with a really cheap AM/FM radio kit, the Elenco Radio Kit
// AM-FM 108 (It's all discrete transistors, no ICs).  We suck the FM
// IF signal out at test point 9 (output of second IF amp), and feed
// it through a buffer board and from there into a 50 Ohm terminated
// input on the Measurement Computing PCI DAS4020/12 A/D board.
//
// Think of the buffer board as a low capacitance input, low output impedance 
// converter that can successfully drive IF signals through 2 meters of coax.
// [I ought to scan in the schematic]


#include <make_GrMC4020Source.h>
#include <GrFFTSink.h>
#include <VrFixOffset.h>
#include <VrComplexFIRfilter.h>
#include <VrQuadratureDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>
#include <VrFileSink.h>
#include <VrFileSource.h>
#include <VrAudioSink.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <VrGUI.h>


const int inputRate = 20000000;		// input sample rate from PCI-DAS4020/12
const float FM_IF_Freq = 8.41797e6;	// this is the aliased FM IF from the ELENCO kit

const float IF_Freq = FM_IF_Freq;

// const int chanTaps = 150;
const int chanTaps = 75;
const int CFIRdecimate = 125;
const float chanGain = 2.0;


//const float FMdemodGain = 8800;
//const float FMdemodGain = 4400;	// works well
const float FMdemodGain = 2200;
//const float FMdemodGain = 1100;

const int RFIRdecimate = 5;
//const int RFIRdecimate = 1;	// for now
//const int ifTaps = 22;
const int ifTaps = 50;
const float ifGain = 1.0;

const int quadRate = inputRate / CFIRdecimate;
const int audioRate = quadRate / RFIRdecimate;


#define	ENABLE_CHANNEL_FILTER_FFT	1


int main(int argc, char **argv)
{
  float volume = 1.0;

  VrGUI *guimain = new VrGUI(argc, argv);
  VrGUILayout *horiz = guimain->top->horizontal();
  VrGUILayout *vert = horiz->vertical();


  cerr << "Input Sampling Rate: " << inputRate << endl;
  cerr << "Complex FIR decimation factor: " << CFIRdecimate << endl;
  cerr << "QuadDemod Sampling Rate: " << quadRate << endl;
  cerr << "Real FIR decimation factor: " << RFIRdecimate << endl;
  cerr << "Audio Sampling Rate: " << audioRate << endl;


  // --> short
  VrSource<short> *source = 
    make_GrMC4020SourceS(inputRate, MCC_CH3_EN | MCC_ALL_1V);

  // short --> short 
  VrFixOffset<short,short> *offset_fixer =
    new VrFixOffset<short,short>();

  // short --> VrComplex
  VrComplexFIRfilter<short>* chan_filter = 
    new VrComplexFIRfilter<short>(CFIRdecimate, chanTaps, IF_Freq, chanGain);

  // VrComplex --> float
  VrQuadratureDemod<float> *fm_demod =
    new VrQuadratureDemod<float>(volume * FMdemodGain);

  // float --> short
  VrRealFIRfilter<float,short> *audio_filter = 
    new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
				     ifTaps, ifGain);

  // order on the display depends on order of initialization, not construction

  // sink1 is channel filter output
  VrSink<VrComplex> *fft_sink1 = new GrFFTSink<VrComplex>(vert, 0, 200, 1024);

  // sink2 is fm demod output
  VrSink<float> *fft_sink2 = new GrFFTSink<float>(vert, 0, 200, 512);

  // sink3 is audio output
  VrSink<short> *fft_sink3 = new GrFFTSink<short>(horiz, 0, 200, 512);

  //connect the modules together

#if 1
  VrSink<short> *audio_sink = new VrAudioSink<short>();
#else
  VrSink<short> *audio_sink = new VrFileSink<short>("elenco_audio.sw");
#endif

  NWO_CONNECT (source, offset_fixer);
  NWO_CONNECT (offset_fixer, chan_filter);
  NWO_CONNECT (chan_filter, fm_demod);
  NWO_CONNECT (chan_filter, fft_sink1);
  NWO_CONNECT (fm_demod, audio_filter);
  NWO_CONNECT (fm_demod, fft_sink2);
  NWO_CONNECT (audio_filter, audio_sink);
  NWO_CONNECT (audio_filter, fft_sink3);

  VrMultiTask *m = new VrMultiTask ();
  m->add (fft_sink1);
  m->add (fft_sink3);
  m->add (fft_sink2);
  m->add (audio_sink);

  m->start ();
  guimain->start ();

  while (1){
    guimain->processEvents(10 /*ms*/);
    m->process();
  }  
}
