/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef _GRWEAVERMODHEAD_H_
#define _GRWEAVERMODHEAD_H_

#include <VrInterpolatingSigProc.h>

/*!
 * \brief Head block of a Weaver SSB/VSB modulator
 *
 * Let the input rate be Fsym.  Complex mix with cos/sin of Fsym/4 & interpolate
 * by 2.  This is typically used like this:
 *
 *                      |---> matched filter --->|
 * ---> GrWeaverModHead |                        | GrWeaverTail --->
 *	  	        |---> matched filter --->|
 */

template<class iType, class oType>
class GrWeaverModHead : public VrInterpolatingSigProc<iType,oType> {
public:
  GrWeaverModHead (int interp_factor) :
    VrInterpolatingSigProc<iType,oType>(2, interp_factor){}

  virtual const char *name () { return "GrWeaverModHead"; }

  virtual int work (VrSampleRange output, void *o[],
		    VrSampleRange inputs[], void *i[]);

  void initialize () { setOutputSize (interp * 4); }
};


template<class iType, class oType> int
GrWeaverModHead<iType,oType>::work (VrSampleRange output, void *ao[],
				    VrSampleRange inputs[], void *ai[])
{
  iType *i = ((iType **) ai)[0];
  oType *oI = ((oType **) ao)[0];
  oType *oQ = ((oType **) ao)[1];
  int size = output.size;
  iType input_symbol;

  static const iType nco[4][2] = {
    // cos	sin
    {    1,	0 },
    {	 0,	1 },
    {   -1,	0 },
    {    0,    -1 }
  };

  assert ((inputs[0].index & 0x3) == 0);

  for (int insamp = 0; insamp < size/interp; insamp++){
    input_symbol = i[insamp];
    *oI++ = input_symbol * nco[insamp & 0x3][0];
    *oQ++ = input_symbol * nco[insamp & 0x3][1];
    for (int outsamp = 1; outsamp < interp; outsamp++){
      *oI++ = 0;
      *oQ++ = 0;
    }
  }
  return output.size;
}

#endif /* _GRWEAVERMODHEAD_H_ */
