/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef _GR_FIR_H_
#define _GR_FIR_H_

#warning "gr_fir.h is deprecated.  Please transition to gr_fir_???.h"

#include <vector>
using std::vector;

/*!
 * \brief base class template for Finite Impulse Response filter (FIR)
 */
template<class o_type, class i_type, class tap_type> 
class gr_fir {
public:
  /*!
   * \brief construct new FIR with given taps.
   *
   * Note that taps must be in backwards order, e.g., coefficient 0 is
   * stored in new_taps[N-1], coefficient 1 is stored in
   * new_taps[N-2], etc.
   */
  gr_fir (const vector<tap_type> &new_taps) : taps (new_taps) {}
  gr_fir () {}

  virtual ~gr_fir () {}

  /*!
   * \brief compute a single output value.
   *
   * \p input must have ntaps() valid entries.
   * input[0] .. input[ntaps() - 1] are referenced to compute the output value.
   *
   * \returns the filtered input value.
   */
  virtual o_type filter (const i_type input[]);

  /*!
   * \brief compute an array of N output values.
   *
   * \p input must have (n - 1 + ntaps()) valid entries.
   * input[0] .. input[n - 1 + ntaps() - 1] are referenced to compute the output values.
   */
  virtual void filterN (o_type output[], const i_type input[],
			unsigned long n);

  /*!
   * \brief compute an array of N output values, decimating the input
   *
   * \p input must have (decimate * (n - 1) + ntaps()) valid entries.
   * input[0] .. input[decimate * (n - 1) + ntaps() - 1] are referenced to compute the output values.
   */
  virtual void filterNdec (o_type output[], const i_type input[],
			   unsigned long n, unsigned decimate);

  /*!
   * \return number of taps in filter.
   */
  unsigned ntaps () const { return taps.size (); }

  /*!
   * \brief install \p new_taps as the current taps.
   */
  virtual void set_taps (const vector<tap_type> &new_taps)
  {
    taps = new_taps;
  }

protected:
  vector<tap_type>	taps;
};


//
// general case.  We may want to specialize this
//
template<class o_type, class i_type, class tap_type> 
o_type
gr_fir<o_type, i_type, tap_type>::filter (const i_type input[])
{
  tap_type	acc = 0;	// FIXME.  Is this the best acc type?
  unsigned	i = 0;
  unsigned	n = (ntaps () / 4) * 4;

  for (i = 0; i < n; i += 4){
    acc += taps[i + 0] * input[i + 0];
    acc += taps[i + 1] * input[i + 1];
    acc += taps[i + 2] * input[i + 2];
    acc += taps[i + 3] * input[i + 3];
  }

  for (; i < ntaps (); i++)
    acc += taps[i] * input[i];

  return (o_type) acc;
}


template<class o_type, class i_type, class tap_type> 
void 
gr_fir<o_type, i_type, tap_type>::filterN (o_type output[],
					   const i_type input[],
					   unsigned long n)
{
  for (unsigned i = 0; i < n; i++)
    output[i] = filter (&input[i]);
}

template<class o_type, class i_type, class tap_type> 
void 
gr_fir<o_type, i_type, tap_type>::filterNdec (o_type output[],
					      const i_type input[],
					      unsigned long n,
					      unsigned decimate)
{
  unsigned j = 0;
  for (unsigned i = 0; i < n; i++){
    output[i] = filter (&input[j]);
    j += decimate;
  }
}


#endif /* _FIR_H_ */
