/* -*- c++ -*- */
/*
 * Copyright 2005 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef INCLUDED_GR_MESSAGE_H
#define INCLUDED_GR_MESSAGE_H

#include <gr_types.h>

class gr_message;
typedef boost::shared_ptr<gr_message> gr_message_sptr;

/*!
 * \brief public constructor for gr_message
 */
gr_message_sptr 
gr_make_message (long type = 0, double arg1 = 0, double arg2 = 0, size_t length = 0);

/*!
 * \brief Message.
 *
 * The ideas and method names for adjustable message length were
 * lifted from the click modular router "Packet" class.
 */
class gr_message {
  gr_message_sptr d_next;	// link field for msg queue
  long		  d_type;	// type of the message
  double	  d_arg1;	// optional arg1
  double 	  d_arg2;	// optional arg2

  // mimic Linux sk_buff
  unsigned char	 *d_head;	// start of allocated buffer
  unsigned char  *d_data;	// where the packet starts
  unsigned char  *d_tail;	// one beyond end of packet
  unsigned char  *d_end;	// one beyond end of allocated buffer
  
  gr_message (long type, double arg1, double arg2, size_t length);

  friend gr_message_sptr 
  gr_make_message (long type, double arg1, double arg2, size_t length);
  friend class gr_msg_queue;

public:
  ~gr_message ();

  long type () const   { return d_type; }
  double arg1 () const { return d_arg1; }
  double arg2 () const { return d_arg2; }

  void set_type (long type)   { d_type = type; }
  void set_arg1 (double arg1) { d_arg1 = arg1; }
  void set_arg2 (double arg2) { d_arg2 = arg2; }

  unsigned char *data () const		{ return d_data; }
  unsigned char *end_data () const 	{ return d_tail; }
  size_t length () const 		{ return d_tail - d_data; }
  size_t headroom () const 		{ return d_data - d_head; }
  size_t tailroom () const 		{ return d_end - d_tail; }
  unsigned char *buffer_data() const 	{ return d_head; }
  size_t buffer_length() const 		{ return d_end - d_head; }

  // void push (size_t nb);	// Add more space before message.
  void pull (size_t nb);	// Get rid of initial bytes.
  // void put (size_t nb);	// Add bytes to end of message.
  void take (size_t nb);	// Delete bytes from end of message.
};

long gr_message_ncurrently_allocated ();

#endif /* INCLUDED_GR_MESSAGE_H */
