;; Copyright (C) 2006 Free Software Foundation, Inc.
;;
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2 of
;; the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public
;; License along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(load "tests.scm")
(or (getenv "DISPLAY")
    (begin
      (display "Cannot run gtk tests without an X display\n")
      (exit 77)))

(use-modules (gtk-1.2 gtk)
	     (gtk-1.2 gdk))

;; the first-ever load of a given font name ties up some memory in a hash
;; table, or something, so make that happen here
(define sample-font     (gdk-font-load "fixed"))

;; don't put "(ash 1 256)" inside a `malloced-steady' because there was a
;; bug in guile 1.6.7 and earlier where the `bytes-malloced' wasn't
;; correctly maintained in bignum calculations
;;
(define sample-bignum (ash 1 256))

(define sample-unspecified (if #f #f))

(define sample-toplevel (gtk-window-new 'toplevel))
(gtk-widget-realize sample-toplevel) ;; realized but not mapped

;; guile 1.6 style make-shared-substring made available in guile 1.8
(if (not (defined? 'make-shared-substring))
    (define make-shared-substring substring/shared))

(define (exact-integer? x)
  (and (number? x)
       (integer? x)
       (exact? x)))

(define (exact-nonnegative-integer? x)
  (and (exact-integer? x)
       (>= x 0)))

;;
;; gtk-adjustment-set-value
;;

(malloced-steady  ;; inum
 (lambda ()
   (let* ((adj   (gtk-adjustment-new 0 0 10 1 1 1))
	  (val   5)
	  (want  #t)
	  (got   (begin
		   (gtk-adjustment-set-value adj val)
		   #t)))
     (test "gtk-adjustment-set-value" val want got))))

(malloced-steady  ;; bignum
 (lambda ()
   (let* ((adj   (gtk-adjustment-new 0 0 10 1 1 1))
	  (val   sample-bignum)
	  (want  #t)
	  (got   (begin
		   (gtk-adjustment-set-value adj val)
		   #t)))
     (test "gtk-adjustment-set-value" val want got))))

(malloced-steady  ;; real
 (lambda ()
   (let* ((adj   (gtk-adjustment-new 0 0 10 1 1 1))
	  (val   5.0)
	  (want  #t)
	  (got   (begin
		   (gtk-adjustment-set-value adj val)
		   #t)))
     (test "gtk-adjustment-set-value" val want got))))

(malloced-steady  ;; complex
 (lambda ()
   (let* ((adj   (gtk-adjustment-new 0 0 10 1 1 1))
	  (val   5.0+5.0i)
	  (want  #f)
	  (got   (false-if-exception
		  (begin
		    (gtk-adjustment-set-value adj val)
		    #t))))
     (test "gtk-adjustment-set-value" val want got))))

;;
;; gtk-bin-child
;;

(malloced-steady
 (lambda ()
   (let* ((child (gtk-label-new "foo"))
	  (want  child)
	  (bin   (gtk-widget-new 'GtkBin))
	  (got   (begin
		   (gtk-container-add bin child)
		   (gtk-bin-child bin))))
     (test "gtk-bin-child" #f want got))))

;;
;; gtk-box-query-child-packing
;;

(malloced-steady
 (lambda ()
   (let* ((pack  '(#t #f 2 end))
	  (box   (gtk-hbox-new #f 0))
	  (child (gtk-label-new "foo"))
	  (want  pack)
	  (got   (begin
		   (gtk-container-add box child)
		   (apply gtk-box-set-child-packing box child pack)
		   (gtk-box-query-child-packing box child))))
     (test "gtk-box-query-child-packing" pack want got))))

;;
;; gtk-check-version
;;

(malloced-steady
 (lambda ()
   (let* ((args '(0 0 sample-bignum))
	  (want #f)
	  (got  (false-if-exception (apply gtk-check-version args))))
     (test "gtk-check-version" args want got))))

(malloced-steady
 (lambda ()
   (let* ((args '(1 0 0))
	  (want #t)
	  (got  (string? (apply gtk-check-version args))))
     (test "gtk-check-version" args want got))))

(malloced-steady
 (lambda ()
   (let* ((args (list gtk-major-version gtk-minor-version gtk-micro-version))
	  (want #f)
	  (got  (apply gtk-check-version args)))
     (test "gtk-check-version" args want got))))

(malloced-steady
 (lambda ()
   (let* ((args '(999 2 0))
	  (want #t)
	  (got  (string? (apply gtk-check-version args))))
     (test "gtk-check-version" args want got))))

;;
;; gtk-clist-new-with-titles
;;

(malloced-steady  ;; list
 (lambda ()
   (let* ((title-list '("abc" "def"))
	  (clist      (gtk-clist-new-with-titles title-list))
	  (want       title-list)
	  (got        (list (gtk-clist-get-column-title clist 0)
			    (gtk-clist-get-column-title clist 1))))
     (test "gtk-clist-new-with-titles" title-list want got))))

(malloced-steady  ;; vector
 (lambda ()
   (let* ((title-vec (vector "abc" "def"))
	  (clist     (gtk-clist-new-with-titles title-vec))
	  (want      title-vec)
	  (got       (vector (gtk-clist-get-column-title clist 0)
			     (gtk-clist-get-column-title clist 1))))
     (test "gtk-clist-new-with-titles" title-vec want got))))

(malloced-steady  ;; number (bad)
 (lambda ()
   (let* ((title-list 999)
	  (want       #f)
	  (got        (false-if-exception
		       (gtk-clist-new-with-titles title-list))))
     (test "gtk-clist-new-with-titles" title-list want got))))

;;
;; gtk-combo-set-popdown-strings
;;

(define (my-gtk-list-item-string item)
  (gtk-label-get (gtk-bin-child item)))

(define (my-gtk-combo-get-popdown-strings combo)
  (map my-gtk-list-item-string
       (gtk-container-children (gtk-combo-list combo))))

(malloced-steady
 (lambda ()
   (let* ((strings '("foo"))
	  (want    strings)
	  (combo   (gtk-combo-new))
	  (got     (begin
		     (gtk-combo-set-popdown-strings combo strings)
		     (my-gtk-combo-get-popdown-strings combo))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

(malloced-steady
 (lambda ()
   (let* ((strings '("foo" "bar" "quux"))
	  (want    strings)
	  (combo   (gtk-combo-new))
	  (got     (begin
		     (gtk-combo-set-popdown-strings combo strings)
		     (my-gtk-combo-get-popdown-strings combo))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

(malloced-steady   ;; empty list
 (lambda ()
   (let* ((strings '())
	  (want    #f)
	  (combo   (gtk-combo-new))
	  (got     (false-if-exception
		    (begin
		      (gtk-combo-set-popdown-strings combo strings)
		      (my-gtk-combo-get-popdown-strings combo)))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

(malloced-steady   ;; empty vector
 (lambda ()
   (let* ((strings (vector))
	  (want    #f)
	  (combo   (gtk-combo-new))
	  (got     (false-if-exception
		    (begin
		      (gtk-combo-set-popdown-strings combo strings)
		      (my-gtk-combo-get-popdown-strings combo)))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

(malloced-steady   ;; bad strings
 (lambda ()
   (let* ((strings 12345)
	  (want    #f)
	  (combo   (gtk-combo-new))
	  (got     (false-if-exception
		    (begin
		      (gtk-combo-set-popdown-strings combo strings)
		      (my-gtk-combo-get-popdown-strings combo)))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

(malloced-steady   ;; non-string in list
 (lambda ()
   (let* ((strings '("foo" "bar" 12345))
	  (want    #f)
	  (combo   (gtk-combo-new))
	  (got     (false-if-exception
		    (begin
		      (gtk-combo-set-popdown-strings combo strings)
		      (my-gtk-combo-get-popdown-strings combo)))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

(malloced-steady   ;; non-string in vector
 (lambda ()
   (let* ((strings (vector "foo" "bar" 12345))
	  (want    #f)
	  (combo   (gtk-combo-new))
	  (got     (false-if-exception
		    (begin
		      (gtk-combo-set-popdown-strings combo strings)
		      (my-gtk-combo-get-popdown-strings combo)))))
     ;; final free()s only take place under the main loop, so need this to
     ;; satisfy `malloced-steady
     (gtk-main-iteration-do #f)
     (test "gtk-combo-set-popdown-strings" strings want got))))

;;
;; gtk-container-children
;;

(malloced-steady
 (lambda ()
   (let ((hbox (gtk-hbox-new #f 0)))
     (gtk-container-add hbox (gtk-label-new "one"))
     (gtk-container-add hbox (gtk-label-new "two"))
     (let* ((want 2)
	    (got  (length (gtk-container-children hbox))))
       (test "gtk-container-children" hbox want got)))))

;;
;; gtk-fixed-put
;;

(malloced-steady     ;; y too big for 16 bits
 (lambda ()
   (let* ((x     123)
	  (y     32768)
	  (fixed (gtk-fixed-new))
	  (child (gtk-label-new "foo"))
	  (want  #f)
	  (got   (false-if-exception
		  (begin
		    (gtk-fixed-put fixed child x y)
		    #t))))
     (test "gtk-fixed-put" (list x y) want got))))

(malloced-steady
 (lambda ()
   (let* ((x     123)
	  (y     456)
	  (fixed (gtk-fixed-new))
	  (child (gtk-label-new "foo"))
	  (want  #t)
	  (got   (false-if-exception
		  (begin
		    (gtk-fixed-put fixed child x y)
		    #t))))
     (test "gtk-fixed-put" (list x y) want got))))

;;
;; gtk-rc-get-default-files
;;

(malloced-steady
 (lambda ()
   (let* ((want #t)
	  (lst  (gtk-rc-get-default-files))
	  (got  (and (>= (length lst) 1)
		     (every string? (gtk-rc-get-default-files)))))
     (test "gtk-rc-get-default-files" #f want got))))

;;
;; gtk-object-get
;;

(malloced-steady
 (lambda ()
   (let* ((label (gtk-label-new "some text")))

     (let* ((prop #:label)
	    (want "some text")
	    (got  (gtk-object-get label prop)))
       (test "gtk-object-get label" prop want got)))))

(malloced-steady
 (lambda ()
   (let* ((label (gtk-label-new "some text")))

     (let* ((prop 'label)
	    (want "some text")
	    (got  (gtk-object-get label prop)))
       (test "gtk-object-get label" prop want got)))))

(malloced-steady
 (lambda ()
   (let* ((label (gtk-label-new "some text")))

     (let* ((prop 12345)
	    (want #f)
	    (got  (false-if-exception (gtk-object-get label prop))))
       (test "gtk-object-get label" prop want got)))))

;;
;; gtk-label-new
;;

(malloced-steady  ;; string
 (lambda ()
   (let* ((str  "foo")
	  (want #t)
	  (got  (->bool (gtk-label-new str))))
     (test "gtk-label-new" str want got))))

(malloced-steady  ;; empty string
 (lambda ()
   (let* ((str  "")
	  (want #t)
	  (got  (->bool (gtk-label-new str))))
     (test "gtk-label-new" str want got))))

(malloced-steady  ;; #f
 (lambda ()
   (let* ((str  #f)
	  (want #t)
	  (got  (->bool (gtk-label-new str))))
     (test "gtk-label-new" str want got))))

(malloced-steady  ;; bad
 (lambda ()
   (let* ((str  1234)
	  (want #f)
	  (got  (false-if-exception (->bool (gtk-label-new str)))))
     (test "gtk-label-new" str want got))))

;;
;; gtk-label-set-text
;;

(malloced-steady  ;; string
 (lambda ()
   (let* ((label (gtk-label-new #f))
	  (str   "foo")
	  (want  #t)
	  (got   (->bool (gtk-label-set-text label str))))
     (test "gtk-label-set-text" str want got))))

(malloced-steady  ;; empty string
 (lambda ()
   (let* ((label (gtk-label-new #f))
	  (str   "")
	  (want  #t)
	  (got   (->bool (gtk-label-set-text label str))))
     (test "gtk-label-set-text" str want got))))

(malloced-steady  ;; #f is bad
 (lambda ()
   (let* ((label (gtk-label-new #f))
	  (str   #f)
	  (want  #f)
	  (got   (false-if-exception (->bool (gtk-label-set-text label str)))))
     (test "gtk-label-set-text" str want got))))

(malloced-steady  ;; bad
 (lambda ()
   (let* ((label (gtk-label-new #f))
	  (str   1234)
	  (want  #f)
	  (got   (false-if-exception (->bool (gtk-label-set-text label str)))))
     (test "gtk-label-set-text" str want got))))

;;
;; gtk-major-version
;;

(malloced-steady
 (lambda ()
   (let* ((want #t)
	  (got  (exact-nonnegative-integer? gtk-major-version)))
     (test "gtk-major-version" #f want got))))

;;
;; gtk-minor-version
;;

(malloced-steady
 (lambda ()
   (let* ((want #t)
	  (got  (exact-nonnegative-integer? gtk-minor-version)))
     (test "gtk-minor-version" #f want got))))

;;
;; gtk-micro-version
;;

(malloced-steady
 (lambda ()
   (let* ((want #t)
	  (got  (exact-nonnegative-integer? gtk-micro-version)))
     (test "gtk-micro-version" #f want got))))

;;
;; gtk-binary-age
;;

(malloced-steady
 (lambda ()
   (let* ((want #t)
	  (got  (exact-nonnegative-integer? gtk-binary-age)))
     (test "gtk-binary-age" #f want got))))

;;
;; gtk-interface-age
;;

(malloced-steady
 (lambda ()
   (let* ((want #t)
	  (got  (exact-nonnegative-integer? gtk-interface-age)))
     (test "gtk-interface-age" #f want got))))

;;
;; gtk-object-new
;;

(malloced-steady
 (lambda ()
   (let* ((args  (list 'NoSuchType))
	  (want  #f)
	  (got   (false-if-exception (apply gtk-object-new args))))
     (test "gtk-object-new" args want got))))

;;
;; gtk-object-new label
;;

(malloced-steady
 (lambda ()
   (let* ((args  (list 'GtkLabel #:label "foo"))
	  (want  "foo")
	  (label (apply gtk-object-new args))
	  (got   (gtk-object-get label #:label)))
     (test "gtk-object-new label" args want got))))

;;
;; gtk-container-query-child-args
;;
;; Would like to look at the actual return value contents, but there's no
;; easy way at the moment to get GtkType objects to compare with what's
;; returned.
;;

(malloced-steady
 (lambda ()
   (let* ((type 'GtkTable)
	  (want #t)
	  (got  (list? (gtk-container-query-child-args type))))
     (test "gtk-container-query-child-args" type want got))))

;;
;; gtk-object-set string
;;

(malloced-steady
 (lambda ()
   (let* ((label (gtk-label-new #f))
	  (str   "foo")
	  (want  str)
	  (got   (begin
		   (gtk-object-set label #:label str)
		   (gtk-object-get label #:label))))
     (test "gtk-object-set string" str want got))))

;;
;; gtk-paint-hline
;;


(malloced-steady
 (lambda ()
   (let* ((args (list (gtk-widget-get-style sample-toplevel)
		      (gtk-widget-window sample-toplevel)
		      'normal
		      #f
		      sample-toplevel
		      #f
		      10 20 10))
	  (want #t)
	  (got  (begin
		  (apply gtk-paint-hline args)
		  #t)))
     (test "gtk-paint-hline" args want got))))

(malloced-steady
 (lambda ()
   (let* ((args (list (gtk-widget-get-style sample-toplevel)
		      (gtk-widget-window sample-toplevel)
		      'normal
		      '((0 . 0) . (100 . 200))
		      sample-toplevel
		      #f
		      10 20 10))
	  (want #t)
	  (got  (begin
		  (apply gtk-paint-hline args)
		  #t)))
     (test "gtk-paint-hline" args want got))))

(malloced-steady
 (lambda ()
   (let* ((args (list (gtk-widget-get-style sample-toplevel)
		      (gtk-widget-window sample-toplevel)
		      'normal
		      '((0 . 0) . (100 . 200))
		      sample-toplevel
		      "label"
		      10 20 10))
	  (want #t)
	  (got  (begin
		  (apply gtk-paint-hline args)
		  #t)))
     (test "gtk-paint-hline" args want got))))

;;
;; gtk-radio-button-group
;;

(malloced-steady
 (lambda ()
   (let* ((rbutton (gtk-radio-button-new-from-widget #f))
	  (want    (list rbutton))
	  (got     (gtk-radio-button-group rbutton)))
     (test "gtk-radio-button-group" rbutton want got))))


;;
;; gtk-scrolled-window-new
;;

(malloced-steady
 (lambda ()
   (let* ((args       '())
	  (scrolled   (apply gtk-scrolled-window-new args))
	  (want       #t)
	  (got        (and (gtk-adjustment?
			    (gtk-scrolled-window-get-hadjustment scrolled))
			   (gtk-adjustment?
			    (gtk-scrolled-window-get-vadjustment scrolled)))))
     (test "gtk-scrolled-window-new" args want got))))

(malloced-steady
 (lambda ()
   (let* ((hadj       (gtk-adjustment-new 0 0 1 0.5 0.5 0.5))
	  (args       (list hadj))
	  (scrolled   (apply gtk-scrolled-window-new args))
	  (want       #t)
	  (got        (and (eq? hadj
				(gtk-scrolled-window-get-hadjustment scrolled))
			   (gtk-adjustment?
			    (gtk-scrolled-window-get-vadjustment scrolled)))))
     (test "gtk-scrolled-window-new" args want got))))

(malloced-steady
 (lambda ()
   (let* ((hadj       (gtk-adjustment-new 0 0 1 0.5 0.5 0.5))
	  (vadj       (gtk-adjustment-new 0 0 1 0.5 0.5 0.5))
	  (args       (list hadj vadj))
	  (scrolled   (apply gtk-scrolled-window-new args))
	  (want       args)
	  (got        (list (gtk-scrolled-window-get-hadjustment scrolled)
			    (gtk-scrolled-window-get-vadjustment scrolled))))
     (test "gtk-scrolled-window-new" args want got))))

(malloced-steady
 (lambda ()
   (let* ((vadj       (gtk-adjustment-new 0 0 1 0.5 0.5 0.5))
	  (args       (list #f vadj))
	  (scrolled   (apply gtk-scrolled-window-new args))
	  (want       #t)
	  (got        (and (gtk-adjustment?
			    (gtk-scrolled-window-get-hadjustment scrolled))
			   (eq? vadj
				(gtk-scrolled-window-get-vadjustment scrolled)))))
     (test "gtk-scrolled-window-new" args want got))))

;;
;; signal call font
;;

(gtk-signal-new-generic "test-call-font" '() 'GtkObject 'void '(GdkFont))
(let ((obj (gtk-object-new 'GtkObject))
      (ret #f))
  (gtk-signal-connect obj "test-call-font" noop)

  (malloced-steady  ;; font
   (lambda ()
     (let* ((font  sample-font)
	    (want  #t)
	    (got   (begin
		     (gtk-signal-emit obj "test-call-font" font)
		     #t)))
       (test "gtk-signal-emit call font" font want got))))

  (malloced-steady  ;; string
   (lambda ()
     (let* ((font  "fixed")
	    (want  #t)
	    (got   (begin
		     (gtk-signal-emit obj "test-call-font" font)
		     #t)))
       (test "gtk-signal-emit call font" font want got)))))

;;
;; signal call string+int
;;

(gtk-signal-new-generic "test-call-string+int" '() 'GtkObject
			'void '(GtkString gint))
(let ((obj (gtk-object-new 'GtkObject))
      (ret #f))
  (gtk-signal-connect obj "test-call-string+int" noop)

  (malloced-steady  ;; good
   (lambda ()
     (let* ((args  (list "foo" 123))
	    (want  #t)
	    (got   (begin
		     (apply gtk-signal-emit obj "test-call-string+int" args)
		     #t)))
       (test "gtk-signal-emit call string+int" args want got))))

  (malloced-steady  ;; int too big
   (lambda ()
     (let* ((args  (list "foo" sample-bignum))
	    (want  #f)
	    (got   (false-if-exception
		    (begin
		      (apply gtk-signal-emit obj "test-call-string+int" args)
		      #t))))
       (test "gtk-signal-emit call string+int" args want got)))))

;;
;; signal call return string
;;

;; Somehow the return is always GTK_TYPE_NONE so you don't get back a
;; string.  The tests here are in readiness if that's fixed.

(gtk-signal-new-generic "test-call-return-string" '() 'GtkObject
			'GtkString '())
(let ((obj (gtk-object-new 'GtkObject))
      (ret #f))
  (gtk-signal-connect obj "test-call-return-string"
    (lambda ()
      ret))

  (malloced-steady
   (lambda ()
     (let* ((str  "")
	    (want #t)
	    (got  (begin
		    (set! ret str)
		    (gtk-signal-emit obj "test-call-return-string")
		    #t)))
       (test "gtk-signal-emit call return string" str want got))))

  (malloced-steady
   (lambda ()
     (let* ((str  "foo")
	    (want #t)
	    (got  (begin
		    (set! ret str)
		    (gtk-signal-emit obj "test-call-return-string")
		    #t)))
       (test "gtk-signal-emit call return string" str want got))))

  (malloced-steady
   (lambda ()
     (let* ((str  (make-shared-substring (string #\nul #\x #\nul) 1 2))
	    (want #t)
	    (got  (begin
		    (set! ret str)
		    (gtk-signal-emit obj "test-call-return-string")
		    #t)))
       (test "gtk-signal-emit call return string" str want got))))

  (malloced-steady  ;; bad type
   (lambda ()
     (let* ((str  999)
	    (want #t)
	    (got  (begin
		    (set! ret str)
		    (gtk-signal-emit obj "test-call-return-string")
		    #t)))
       (test "gtk-signal-emit call return string" str want got))))

  (malloced-steady  ;; bomb with \0 in string
   (lambda ()
     (let* ((str  (string #\nul))
	    (want #t)
	    (got  (begin
		    (set! ret str)
		    (gtk-signal-emit obj "test-call-return-string")
		    #t)))
       (test "gtk-signal-emit call return string" str want got)))))


(tests-end)
