# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000, 2001, 2002  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, gnome, soundcard
import GDK
from i18n import _
import gu, os
import cfg

class MidiInstrumentMenu(gtk.GtkMenu):
    def __init__(self, callback):
        gtk.GtkMenu.__init__(self)
        self.m_callback = callback
        for x in range(len(soundcard.instrument_names)):
            if x % 8 == 0:
                menuitem = gtk.GtkMenuItem(soundcard.instrument_sections[x/8])
                submenu = gtk.GtkMenu()
                self.append(menuitem)
                menuitem.set_submenu(submenu)
                menuitem.show()
            item = gtk.GtkMenuItem(soundcard.instrument_names[x])
            item.connect('event', self.menuevent)
            item.set_data('instrument', x)
            submenu.append(item)
            item.show()
        self.show()
    def menuevent(self, _o=None, event=None):
        if event.type == GDK.BUTTON_RELEASE:
            self.m_callback(_o.get_data('instrument'))


class ReadOnlyInstrumentSelector(gtk.GtkFrame, cfg.ConfigUtils):
    def __init__(self, txt, section, name, on_change_callback=None):
        """
        ReadOnlyInstrumentSelector is used by ConfigWindow, because
        the config data should not be changed before the user press
        'apply' or 'ok'. The exercises' config page should use
        the InstrumentSelector class
        
        section: what section in config file to use, typically an exercise name
        name: name and name_velocity are used in configfile
        """
        gtk.GtkFrame.__init__(self, txt)
        #cfg.ConfigUtils.__init__(self, section)
        cfg.ConfigUtils.__dict__['__init__'](self, section)
        self.m_name = name
        self.m_on_change_callback = on_change_callback
        vbox = gtk.GtkVBox()
        vbox.set_border_width(gnome.uiconsts.PAD_SMALL)
        self.add(vbox)
        self.g_button = gu.bButton(vbox,
              soundcard.instrument_names[self.get_int(self.m_name)],
              self.on_btnclick)
        self.g_menu = MidiInstrumentMenu(self.on_instrument_selected)
        self.m_instrument = self.get_int('config/preferred_instrument')
        if not self.get_int(self.m_name+'_velocity'):
            self.set_int(self.m_name+'_velocity',
                      self.get_int('config/preferred_instrument_velocity'))
        hbox = gu.bHBox(vbox)
        hbox.set_spacing(gnome.uiconsts.PAD_SMALL)
        label = gtk.GtkLabel(_("Velocity:"))
        label.set_alignment(1.0, 0.5)
        hbox.pack_start(label)
        self.g_velocity = gtk.GtkSpinButton(gtk.GtkAdjustment(self.get_int(self.m_name+'_velocity'), 0, 127, 1, 10), digits=0)
        self.g_velocity.connect('changed', self.spin_callback)
        ihbox = gu.bHBox(hbox, gtk.FALSE)
        ihbox.pack_start(self.g_velocity, gtk.FALSE)
        btn = gu.bButton(ihbox, None, self.play_selected_instrument, gtk.FALSE)
        pix, mask = gtk.create_pixmap_from_xpm(btn, None,
                                       os.path.join("xpm", "test-sound.xpm"))
        p = gtk.GtkPixmap(pix, mask)

        btn.add(p)
        self.show_all()
    def on_btnclick(self, *argv):
        self.g_menu.popup(None, None, None, 1, 0)
    def on_instrument_selected(self, instrument=None):
        self.g_button.children()[0].set_text(soundcard.instrument_names[instrument])
        self.m_instrument = instrument
        self.play_selected_instrument()
        if self.m_on_change_callback:
            self.m_on_change_callback()
    def spin_callback(self, spin):
        if self.m_on_change_callback:
            self.m_on_change_callback()
    def play_selected_instrument(self, _o=None):
        soundcard.play_note(self.m_instrument, 4, 0, 60,
                            self.g_velocity.get_value_as_int())

class InstrumentSelector(ReadOnlyInstrumentSelector):
    def __init__(self, txt, section, name, on_change_callback=None):
        ReadOnlyInstrumentSelector.__init__(self, txt, section, name, on_change_callback)
        self.add_watch(self.m_name, self._instr_cb)
        self.add_watch(self.m_name+'_velocity', self._velocity_cb)
    def _instr_cb(self, name):
        self.g_button.children()[0].set_text(
              soundcard.instrument_names[self.get_int(name)])
    def _velocity_cb(self, name):
        self.g_velocity.set_value(self.get_float(name))
    def on_instrument_selected(self, instrument=None):
        ReadOnlyInstrumentSelector.on_instrument_selected(self, instrument)
        self.set_int(self.m_name, instrument)        
    def spin_callback(self, p=None):
        ReadOnlyInstrumentSelector.spin_callback(self, p)
        self.set_int(self.m_name+'_velocity', p.get_value_as_int())
                                        
class InstrumentConfigurator(gtk.GtkVBox, cfg.ConfigUtils):
    def __init__(self, exname, num):
        gtk.GtkVBox.__init__(self)
        #cfg.ConfigUtils.__init__(self, exname)
        cfg.ConfigUtils.__dict__['__init__'](self, exname)
        assert num in (2, 3)
        self.m_num = num
        self.g_override_default_instrument_checkbutton \
            = gu.nCheckButton(exname, 'override_default_instrument',
                 _("Change midi instrument for this exercise"),
                 callback=self.update_instrument_override)
        self.pack_start(self.g_override_default_instrument_checkbutton,
                        gtk.FALSE)
        hbox = gu.bHBox(self)
        hbox.set_spacing(gnome.uiconsts.PAD_SMALL)

        self.g_instrsel_high = InstrumentSelector(_("Highest instrument"),
                                                 exname, 'highest_instrument')
        hbox.pack_start(self.g_instrsel_high, gtk.FALSE)
        if num == 3:
            self.g_instrsel_middle = InstrumentSelector(_("Middle instrument"),
                                            exname, 'middle_instrument')
            hbox.pack_start(self.g_instrsel_middle, gtk.FALSE)
        else:
            self.g_instrsel_middle = None
        self.g_instrsel_low = InstrumentSelector(_("Lowest instrument"),
                                                exname, 'lowest_instrument')
        hbox.pack_start(self.g_instrsel_low, gtk.FALSE)
        self.update_instrument_override()
    def update(self):
        self.update_instrument_override()
        self.g_instrsel_high.update()
        if self.m_num == 3:
            self.g_instrsel_middle.update()
        self.g_instrsel_low.update()
    def update_instrument_override(self, _o=None):
        self.g_override_default_instrument_checkbutton.set_active(
                self.get_bool('override_default_instrument'))
        self.g_instrsel_high.set_sensitive(
               self.g_override_default_instrument_checkbutton.get_active())
        self.g_instrsel_low.set_sensitive(
               self.g_override_default_instrument_checkbutton.get_active())
        if self.g_instrsel_middle:
            self.g_instrsel_middle.set_sensitive(
               self.g_override_default_instrument_checkbutton.get_active())
    def show(self):
        self.show_all()

