// Cover page for AML files.
// Renders all account properties, current rules, etc.

#let form(data) = {
  set page(
    paper: "a4",
    margin: (left: 2cm, right: 2cm, top: 2cm, bottom: 2.5cm),
    footer: context [
      #grid(
        columns: (1fr, 1fr),
        align: (left, right),
        text(size: 8pt)[
        ],
        text(size: 8pt)[
          Page #here().page() of #counter(page).final().first()
        ]
      )
    ]
  )

  set text(font: "Liberation Sans", size: 10pt)
  set par(justify: false, leading: 0.65em)

  // Helper function to get value or empty string
  let get(key, default: "") = {
    data.at(key, default: default)
  }

  // Helper function for checkbox
  let checkbox(checked) = {
    box(
      width: 3mm,
      height: 3mm,
      stroke: 0.5pt + black,
      inset: 0.3mm,
      if checked == true or checked == "true" {
        place(center + horizon, text(size: 8pt, sym.checkmark))
      }
    )
  }

  // Helper function to get nice labels for standard properties
  let get_property_label(key) = {
    if key == "pep" { "Politically exposed person (PEP)" }
    else if key == "sanctioned" { "Sanctioned account" }
    else if key == "high_risk" { "High risk account" }
    else if key == "business_domain" { "Business domain" }
    else if key == "is_frozen" { "Account frozen" }
    else if key == "was_reported" { "Reported to authorities" }
    else { key }
  }

  // Helper function to format timeframe
  let format_timeframe(d_us) = {
    if d_us == "forever" {
      "forever"
    } else {
      let us = int(d_us)
      let s = calc.quo(us, 1000000)
      let m = calc.quo(s, 60)
      let h = calc.quo(m, 60)
      let d = calc.quo(h, 24)

      if calc.rem(us, 1000000) == 0 {
        if calc.rem(s, 60) == 0 {
          if calc.rem(m, 60) == 0 {
            if calc.rem(h, 24) == 0 {
              str(d) + " day" + if d != 1 { "s" } else { "" }
            } else {
              str(h) + " hour" + if h != 1 { "s" } else { "" }
            }
          } else {
            str(m) + " minute" + if m != 1 { "s" } else { "" }
          }
        } else {
          str(s) + " s"
        }
      } else {
        str(us) + " μs"
      }
    }
  }


  // Helper function to format timestamp; ignores leap seconds (too variable)
  let format_timestamp(ts) = {
    if type(ts) == dictionary and "t_s" in ts {
      let t_s = ts.t_s
      if t_s == "never" {
        "never"
      } else {
        // Convert Unix timestamp to human-readable format
        let seconds = int(t_s)
        let days_since_epoch = calc.quo(seconds, 86400)
        let remaining_seconds = calc.rem(seconds, 86400)
        let hours = calc.quo(remaining_seconds, 3600)
        let minutes = calc.quo(calc.rem(remaining_seconds, 3600), 60)
        let secs = calc.rem(remaining_seconds, 60)

        // Helper to check if year is leap year
        let is_leap(y) = {
          calc.rem(y, 4) == 0 and (calc.rem(y, 100) != 0 or calc.rem(y, 400) == 0)
        }

        // Calculate year, month, day
        let year = 1970
        let days_left = days_since_epoch

        // Find the year
        let done = false
        while not done {
          let days_in_year = if is_leap(year) { 366 } else { 365 }
          if days_left >= days_in_year {
            days_left = days_left - days_in_year
            year = year + 1
          } else {
            done = true
          }
        }

        // Days in each month
        let days_in_months = if is_leap(year) {
          (31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)
        } else {
          (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)
        }

        // Find month and day
        let month = 1
        for days_in_month in days_in_months {
          if days_left >= days_in_month {
            days_left = days_left - days_in_month
            month = month + 1
          } else {
            break
          }
        }
        let day = days_left + 1

        // Format with leading zeros
        let m_str = if month < 10 { "0" + str(month) } else { str(month) }
        let d_str = if day < 10 { "0" + str(day) } else { str(day) }
        let h_str = if hours < 10 { "0" + str(hours) } else { str(hours) }
        let min_str = if minutes < 10 { "0" + str(minutes) } else { str(minutes) }
        let s_str = if secs < 10 { "0" + str(secs) } else { str(secs) }

        str(year) + "-" + m_str + "-" + d_str + " " + h_str + ":" + min_str + ":" + s_str + " UTC"
      }
    } else {
      str(ts)
    }
  }



  // Header
  align(center, text(size: 11pt, weight: "bold")[CONFIDENTIAL])

  v(0.5em)

  grid(
    columns: (50%, 50%),
    gutter: 1em,
    image("taler-logo.svg", width: 80%),
    align(right)[
      #table(
        columns: (1fr, 1fr),
        stroke: 0.5pt + black,
        inset: 5pt,
        align: (left, left),
        [AMLA File No.], [#get("FILE_NUMBER")],
        [Account open?], [#checkbox(get("is_active"))],
        [Active investigation?], [#checkbox(get("to_investigate"))],
      )
    ]
  )

  v(1em)

  // Section 1: Properties
  text(size: 11pt, weight: "bold")[Properties:]

  v(0.5em)

  block(breakable: false)[
    #v(0.5em)
    #let props = get("properties", default: (:))
    #let standard_props = ("pep", "sanctioned", "high_risk", "business_domain", "is_frozen", "was_reported")
    #let all_keys = props.keys()

    #table(
      columns: (35%, 65%),
      stroke: 0.5pt + black,
      inset: 5pt,
      align: (left, left),
      ..for key in all_keys {
        let value = props.at(key)
        let label = get_property_label(key)

        // Render based on value type
        if type(value) == bool {
          ([#label], [#checkbox(value)])
        } else {
          ([#label], [#value])
        }
      }
    )
    #v(0.5em)
  ]

  // Section 2: Rules
  let rules_data = get("rules", default: none)

  if rules_data != none {
    text(size: 11pt, weight: "bold")[
      Rules
      #if "expiration_time" in rules_data {
        [ (expires: #format_timestamp(rules_data.expiration_time))]
      }
      :
    ]

    v(0.5em)

    let rules = rules_data.at("rules", default: ())

    if rules.len() > 0 {
      block(breakable: true)[
        #table(
          columns: (17%, 13%, 13%, 20%, 17%, 10%, 10%),
          stroke: 0.5pt + black,
          inset: 4pt,
          align: (left, left, left, left, left, center, center),
          table.header(
            [*Operation*],
            [*Threshold*],
            [*Timeframe*],
            [*Rule Name*],
            [*Measures*],
            [*Exposed*],
            [*Verboten*]
          ),
          ..for rule in rules {
            let op_type = rule.at("operation_type", default: "")
            let threshold = rule.at("threshold", default: "")
            let timeframe_raw = rule.at("timeframe", default: (:))
            let timeframe = if "d_us" in timeframe_raw {
              format_timeframe(timeframe_raw.d_us)
            } else { "" }
            let rule_name = rule.at("rule_name", default: "")
            let measures = rule.at("measures", default: ())
            let exposed = rule.at("exposed", default: false)
            let is_verboten = if type(measures) == array { "verboten" in measures } else { "verboten" == measures }
            let measures_text = if type(measures) == array {
              measures.filter(m => m != "verboten").map(m => str(m)).join(", ")
            } else if measures != "verboten" {
              str(measures)
            } else {
              ""
            }

            (
              [#op_type],
              [#threshold],
              [#timeframe],
              [#rule_name],
              [#measures_text],
              [#checkbox(exposed)],
              [#checkbox(is_verboten)]
            )
          }
        )
      ]
    } else {
      text(style: "italic")[No rules defined.]
    }
  }
}

// Example usage:
#form((
  "FILE_NUMBER": "42",
  "is_active": true,
  "to_investigate": false,
  "properties": (
    "pep": false,
    "sanctioned": false,
    "high_risk": true,
    "business_domain": "Financial services",
    "is_frozen": false,
    "was_reported": false,
    "custom_field": "Custom value"
  ),
  "rules": (
    "expiration_time": ("t_s": 1764967786),  // Fri Dec 5 20:49:46 UTC 2025
    "rules": (
      (
        "operation_type": "WITHDRAW",
        "rule_name": "large_withdrawal",
        "threshold": "EUR:10000",
        "timeframe": ("d_us": 86400000000),
        "measures": ("kyc_review"),
        "display_priority": 10,
        "exposed": true
      ),
      (
        "operation_type": "DEPOSIT",
        "rule_name": "suspicious_deposit",
        "threshold": "EUR:50000",
        "timeframe": ("d_us": 604800000000),
        "measures": ("verboten"),
        "display_priority": 20,
        "exposed": false
      ),
      (
        "operation_type": "BALANCE",
        "threshold": "EUR:5000",
        "timeframe": ("d_us": 3600000000),
        "measures": ("aml_check", "manager_approval"),
        "display_priority": 5,
        "exposed": true
      )
    )
  )
))