/*
 This file is part of GNU Taler
 (C) 2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  Duration,
  MerchantApiClient,
  PreparePayResultType,
  TransactionType,
  j2s,
  narrowOpSuccessOrThrow,
  randomRfc3548Base32Key,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runOtpTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV3(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());
  const createOtpRes = await merchantClient.createOtpDevice({
    otp_algorithm: 1,
    otp_device_description: "Hello",
    otp_device_id: "mydevice",
    otp_key: randomRfc3548Base32Key(),
  });
  narrowOpSuccessOrThrow("createOtpDevice", createOtpRes);

  const createTemplateRes = await merchantClient.createTemplate({
    template_description: "my template",
    template_id: "tpl1",
    otp_id: "mydevice",
    template_contract: {
      summary: "test",
      amount: "TESTKUDOS:1",
      minimum_age: 0,
      pay_duration: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ hours: 1 }),
      ),
    },
  });
  narrowOpSuccessOrThrow("createTemplate", createTemplateRes);

  const getTemplateResp = await merchantClient.getTemplate("tpl1");
  narrowOpSuccessOrThrow("getTemplate", getTemplateResp);

  console.log(`template: ${j2s(getTemplateResp.body)}`);

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });
  await wres.withdrawalFinishedCond;

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForTemplate,
    {
      talerPayTemplateUri: `taler+http://pay-template/localhost:${merchant.port}/tpl1`,
      templateParams: {},
    },
  );

  console.log(preparePayResult);

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  // Pay for it

  const r2 = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  t.assertTrue(r2.type === ConfirmPayResultType.Done);

  const transaction = await walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: preparePayResult.transactionId,
    },
  );

  console.log(j2s(transaction));

  t.assertTrue(transaction.type === TransactionType.Payment);
  t.assertTrue(transaction.posConfirmation != null);
  t.assertTrue(transaction.posConfirmation.length > 10);
}

runOtpTest.suites = ["wallet"];
