/*
 This file is part of GNU Taler
 (C) 2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  j2s,
  NotificationType,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
  WalletNotification,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  BankServiceHandle,
  ExchangeService,
  GlobalTestState,
  WalletClient,
} from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  createWalletDaemonWithClient,
  withdrawViaBankV2,
} from "../harness/helpers.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

const coinConfigList: CoinConfig[] = [
  {
    ...coinCommon,
    name: "n1",
    value: "TESTKUDOS:1",
  },
];

/**
 * Test peer pull payments with a large number of coins.
 *
 * Since we use an artificallly large number of coins, this
 * test is a bit slower than other tests.
 */
export async function runPeerPullLargeTest(t: GlobalTestState) {
  // Set up test environment

  const { bank, exchange } = await createSimpleTestkudosEnvironmentV2(
    t,
    coinConfigList,
  );

  let allW1Notifications: WalletNotification[] = [];
  let allW2Notifications: WalletNotification[] = [];

  const w1 = await createWalletDaemonWithClient(t, {
    name: "w1",
    persistent: true,
    handleNotification(wn) {
      allW1Notifications.push(wn);
    },
  });
  const w2 = await createWalletDaemonWithClient(t, {
    name: "w2",
    persistent: true,
    handleNotification(wn) {
      allW2Notifications.push(wn);
    },
  });

  // Withdraw digital cash into the wallet.
  const wallet1 = w1.walletClient;
  const wallet2 = w2.walletClient;

  await checkNormalPeerPull(t, bank, exchange, wallet1, wallet2);
}

async function checkNormalPeerPull(
  t: GlobalTestState,
  bank: BankServiceHandle,
  exchange: ExchangeService,
  wallet1: WalletClient,
  wallet2: WalletClient,
): Promise<void> {
  t.logStep("starting withdrawal");
  const withdrawRes = await withdrawViaBankV2(t, {
    walletClient: wallet2,
    bank,
    exchange,
    amount: "TESTKUDOS:500",
  });

  await withdrawRes.withdrawalFinishedCond;

  t.logStep("finished withdrawal");

  const purseExpiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );

  const resp = await wallet1.client.call(
    WalletApiOperation.InitiatePeerPullCredit,
    {
      exchangeBaseUrl: exchange.baseUrl,
      partialContractTerms: {
        summary: "Hello World",
        amount: "TESTKUDOS:200" as AmountString,
        purse_expiration: purseExpiration,
      },
    },
  );

  const peerPullCreditReadyCond = wallet1.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === resp.transactionId &&
      x.newTxState.major === TransactionMajorState.Pending &&
      x.newTxState.minor === TransactionMinorState.Ready,
  );

  await peerPullCreditReadyCond;

  const creditTx = await wallet1.call(WalletApiOperation.GetTransactionById, {
    transactionId: resp.transactionId,
  });

  t.assertDeepEqual(creditTx.type, TransactionType.PeerPullCredit);
  t.assertTrue(!!creditTx.talerUri);

  const checkResp = await wallet2.client.call(
    WalletApiOperation.PreparePeerPullDebit,
    {
      talerUri: creditTx.talerUri,
    },
  );

  console.log(`checkResp: ${j2s(checkResp)}`);

  const peerPullCreditDoneCond = wallet1.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === resp.transactionId &&
      x.newTxState.major === TransactionMajorState.Done,
  );

  const peerPullDebitDoneCond = wallet2.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === checkResp.transactionId &&
      x.newTxState.major === TransactionMajorState.Done,
  );

  await wallet2.client.call(WalletApiOperation.ConfirmPeerPullDebit, {
    transactionId: checkResp.transactionId,
  });

  await peerPullCreditDoneCond;
  await peerPullDebitDoneCond;

  const txn1 = await wallet1.client.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  const txn2 = await wallet2.client.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  console.log(`txn1: ${j2s(txn1)}`);
  console.log(`txn2: ${j2s(txn2)}`);
}

runPeerPullLargeTest.suites = ["wallet", "slow"];
