/*
 This file is part of GNU Taler
 (C) 2019 GNUnet e.V.
 (C) 2024 Taler Systems SA

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { userIdentifierDerive } from "@gnu-taler/anastasis-core/lib/crypto.js";
import { AmountString, j2s } from "@gnu-taler/taler-util";
import {
  WalletApiOperation,
  createNativeWalletHost2,
} from "@gnu-taler/taler-wallet-core";

export async function testWithGv() {
  const w = await createNativeWalletHost2({});
  await w.wallet.client.call(WalletApiOperation.InitWallet, {
    config: {
      features: {
        allowHttp: true,
      },
    },
  });
  await w.wallet.client.call(WalletApiOperation.RunIntegrationTest, {
    amountToSpend: "KUDOS:1" as AmountString,
    amountToWithdraw: "KUDOS:3" as AmountString,
    corebankApiBaseUrl: "https://bank.demo.taler.net/",
    exchangeBaseUrl: "https://exchange.demo.taler.net/",
    merchantBaseUrl: "https://backend.demo.taler.net/",
    merchantAuthToken: "secret-token:sandbox",
  });
  await w.wallet.client.call(WalletApiOperation.TestingWaitTasksDone, {});
  await w.wallet.client.call(WalletApiOperation.Shutdown, {});
}

export async function testWithFdold() {
  const w = await createNativeWalletHost2({});
  await w.wallet.client.call(WalletApiOperation.InitWallet, {
    config: {
      features: {
        allowHttp: true,
      },
    },
  });
  await w.wallet.client.call(WalletApiOperation.RunIntegrationTest, {
    amountToSpend: "TESTKUDOS:1" as AmountString,
    amountToWithdraw: "TESTKUDOS:3" as AmountString,
    corebankApiBaseUrl: "https://bank.taler.fdold.eu/",
    exchangeBaseUrl: "https://exchange.taler.fdold.eu/",
    merchantBaseUrl: "https://merchant.taler.fdold.eu/",
  });
  await w.wallet.client.call(WalletApiOperation.TestingWaitTasksDone, {});
  await w.wallet.client.call(WalletApiOperation.Shutdown, {});
}

export async function testWithLocal(path: string) {
  console.log("running local test");
  const w = await createNativeWalletHost2({
    persistentStoragePath: path ?? "walletdb.json",
  });
  console.log("created wallet");
  await w.wallet.client.call(WalletApiOperation.InitWallet, {
    config: {
      features: {
        allowHttp: true,
      },
      testing: {
        skipDefaults: true,
      },
    },
  });
  console.log("initialized wallet");
  await w.wallet.client.call(WalletApiOperation.RunIntegrationTest, {
    amountToSpend: "TESTKUDOS:1" as AmountString,
    amountToWithdraw: "TESTKUDOS:3" as AmountString,
    corebankApiBaseUrl: "http://localhost:8082/taler-bank-access/",
    exchangeBaseUrl: "http://localhost:8081/",
    merchantBaseUrl: "http://localhost:8083/",
  });
  console.log("started integration test");
  await w.wallet.client.call(WalletApiOperation.TestingWaitTasksDone, {});
  console.log("done with task loop");
  await w.wallet.client.call(WalletApiOperation.Shutdown, {});
  console.log("DB stats:", j2s(w.getDbStats()));
}

export async function testArgon2id() {
  const userIdVector = {
    input_id_data: {
      name: "Fleabag",
      ssn: "AB123",
    },
    input_server_salt: "FZ48EFS7WS3R2ZR4V53A3GFFY4",
    output_id:
      "YS45R6CGJV84K1NN7T14ZBCPVTZ6H15XJSM1FV0R748MHPV82SM0126EBZKBAAGCR34Q9AFKPEW1HRT2Q9GQ5JRA3642AB571DKZS18",
  };

  if (
    (await userIdentifierDerive(
      userIdVector.input_id_data,
      userIdVector.input_server_salt,
    )) != userIdVector.output_id
  ) {
    throw Error("argon2id is not working!");
  }

  console.log("argon2id is working!");
}
