/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { AmountString, Amounts, TalerMerchantApi } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { ComponentChildren, Fragment, VNode, h } from "preact";
import { StateUpdater, useState } from "preact/hooks";
import emptyImage from "../../../../assets/empty.png";
import {
  FormErrors,
  FormProvider,
  TalerForm,
} from "../../../../components/form/FormProvider.js";
import { InputCurrency } from "../../../../components/form/InputCurrency.js";
import { InputNumber } from "../../../../components/form/InputNumber.js";
import {
  dateFormatForSettings,
  usePreference,
} from "../../../../hooks/preference.js";
import { WithId } from "../../../../declaration.js";

type Entity = TalerMerchantApi.ProductDetail & WithId;

interface Props {
  instances: Entity[];
  onDelete?: (id: Entity) => void;
  onSelect: (product: Entity) => void;
  onUpdate?: (
    id: string,
    data: TalerMerchantApi.ProductPatchDetail,
  ) => Promise<void>;
  onCreate?: () => void;
  selected?: boolean;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

export function CardTable({
  instances,
  onCreate,
  onSelect,
  onUpdate,
  onDelete,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: Props): VNode {
  const [rowSelection, rowSelectionHandler] = useState<string | undefined>(
    undefined,
  );
  const { i18n } = useTranslationContext();
  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-shopping" />
          </span>
          <i18n.Translate>Inventory</i18n.Translate>
        </p>
        <div class="card-header-icon" aria-label="more options">
          {!onCreate ? undefined : (
            <span
              class="has-tooltip-left"
              data-tooltip={i18n.str`Add product to inventory`}
            >
              <button class="button is-info" type="button" onClick={onCreate}>
                <span class="icon is-small">
                  <i class="mdi mdi-plus mdi-36px" />
                </span>
              </button>
            </span>
          )}
        </div>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {instances.length > 0 ? (
              <Table
                instances={instances}
                onSelect={onSelect}
                onDelete={onDelete}
                onUpdate={onUpdate}
                onLoadMoreAfter={onLoadMoreAfter}
                onLoadMoreBefore={onLoadMoreBefore}
                rowSelection={rowSelection}
                rowSelectionHandler={rowSelectionHandler}
              />
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
interface TableProps {
  rowSelection: string | undefined;
  instances: Entity[];
  onSelect: (id: Entity) => void;
  onUpdate:
    | ((id: string, data: TalerMerchantApi.ProductPatchDetail) => Promise<void>)
    | undefined;
  onDelete: ((id: Entity) => void) | undefined;
  rowSelectionHandler: StateUpdater<string | undefined>;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

function Table({
  rowSelection,
  rowSelectionHandler,
  instances,
  onSelect,
  onUpdate,
  onDelete,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: TableProps): VNode {
  const { i18n } = useTranslationContext();
  const [preference] = usePreference();
  return (
    <div class="table-container">
      {onLoadMoreBefore && (
        <button class="button is-fullwidth" onClick={onLoadMoreBefore}>
          <i18n.Translate>Load first page</i18n.Translate>
        </button>
      )}
      <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
        <thead>
          <tr>
            <th>
              <i18n.Translate>Image</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Name</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Description</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Price per unit</i18n.Translate>
            </th>
            {preference.developerMode ? (
              <Fragment>
                <th>
                  <i18n.Translate>Taxes</i18n.Translate>
                </th>
                <th>
                  <i18n.Translate>Sales</i18n.Translate>
                </th>
              </Fragment>
            ) : (
              <Fragment />
            )}
            <th>
              <i18n.Translate>Stock</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Sold</i18n.Translate>
            </th>
            <th />
          </tr>
        </thead>
        <tbody>
          {instances.map((i) => {
            const restStockInfo = !i.next_restock
              ? ""
              : i.next_restock.t_s === "never"
                ? i18n.str`Never`
                : `restock at ${format(
                    new Date(i.next_restock.t_s * 1000),
                    dateFormatForSettings(preference),
                  )}`;
            let stockInfo: ComponentChildren = "";
            if (i.total_stock < 0) {
              stockInfo = "infinite";
            } else {
              const totalStock = i.total_stock - i.total_lost - i.total_sold;
              stockInfo = (
                <label title={restStockInfo}>
                  {totalStock} {i.unit}
                </label>
              );
            }

            const isFree = Amounts.isZero(Amounts.parseOrThrow(i.price));

            return (
              <Fragment key={i.id}>
                <tr key="info">
                  <td
                    onClick={() =>
                      onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                    <img
                      src={i.image ? i.image : emptyImage}
                      style={{
                        border: "solid black 1px",
                        maxHeight: "2em",
                        width: "auto",
                        height: "auto",
                      }}
                    />
                  </td>
                  <td
                    class="has-tooltip-right"
                    data-tooltip={i.product_name}
                    onClick={() =>
                      onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                    {i.product_name}
                  </td>
                  <td
                    class="has-tooltip-right"
                    data-tooltip={i.description}
                    onClick={() =>
                      onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                    {i.description.length > 30
                      ? i.description.substring(0, 30) + "..."
                      : i.description}
                  </td>
                  <td
                    onClick={() =>
                      onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                    {isFree ? i18n.str`Free` : `${i.price} / ${i.unit}`}
                  </td>
                  {preference.developerMode ? (
                    <Fragment>
                      <td
                        onClick={() =>
                          onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                        }
                        style={{ cursor: "pointer" }}
                      >
                        {sum(i.taxes)}
                      </td>
                      <td
                        onClick={() =>
                          onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                        }
                        style={{ cursor: "pointer" }}
                      >
                        {difference(i.price, sum(i.taxes))}
                      </td>
                    </Fragment>
                  ) : (
                    <Fragment />
                  )}
                  <td
                    onClick={() =>
                      onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                    {stockInfo}
                  </td>
                  <td
                    onClick={() =>
                      onUpdate && rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                    <span style={{ whiteSpace: "nowrap" }}>
                      {i.total_sold} {i.unit}
                    </span>
                  </td>
                  <td class="is-actions-cell right-sticky">
                    <div class="buttons is-right">
                      <span
                        class="has-tooltip-bottom"
                        data-tooltip={i18n.str`Go to product update page`}
                      >
                        <button
                          class="button is-small is-success "
                          type="button"
                          onClick={(): void => onSelect(i)}
                        >
                          <i18n.Translate>Update</i18n.Translate>
                        </button>
                      </span>
                      {!onDelete ? undefined : (
                        <span
                          class="has-tooltip-left"
                          data-tooltip={i18n.str`Remove this product from the database`}
                        >
                          <button
                            class="button is-small is-danger"
                            type="button"
                            onClick={(): void => onDelete(i)}
                          >
                            <i18n.Translate>Delete</i18n.Translate>
                          </button>
                        </span>
                      )}
                    </div>
                  </td>
                </tr>
                {rowSelection === i.id && (
                  <tr key="form">
                    <td colSpan={10}>
                      <FastProductUpdateForm
                        product={i}
                        onUpdate={async (prod) => {
                          if (!onUpdate) return;
                          return onUpdate(i.id, prod).then(() =>
                            rowSelectionHandler(undefined),
                          );
                        }}
                        onCancel={() => rowSelectionHandler(undefined)}
                      />
                    </td>
                  </tr>
                )}
              </Fragment>
            );
          })}
        </tbody>
      </table>
      {onLoadMoreAfter && (
        <button
          class="button is-fullwidth"
          data-tooltip={i18n.str`Load more products after the last one`}
          onClick={onLoadMoreAfter}
        >
          <i18n.Translate>Load next page</i18n.Translate>
        </button>
      )}
    </div>
  );
}

interface FastProductUpdateFormProps {
  product: Entity;
  onUpdate: (data: TalerMerchantApi.ProductPatchDetail) => Promise<void>;
  onCancel: () => void;
}
interface FastProductUpdate extends TalerForm {
  incoming: number;
  lost: number;
  price: string;
}
interface UpdatePrice {
  price: string;
}

function FastProductWithInfiniteStockUpdateForm({
  product,
  onUpdate,
  onCancel,
}: FastProductUpdateFormProps) {
  const [value, valueHandler] = useState<UpdatePrice>({ price: product.price });
  const { i18n } = useTranslationContext();

  return (
    <Fragment>
      <FormProvider<FastProductUpdate>
        name="added"
        object={value}
        valueHandler={valueHandler as any}
      >
        <InputCurrency<FastProductUpdate>
          name="price"
          label={i18n.str`Price per unit`}
          tooltip={i18n.str`Update the product with new price`}
        />
      </FormProvider>

      <div class="buttons is-expanded">
        <div class="buttons is-right mt-5">
          <button class="button" onClick={onCancel}>
            <i18n.Translate>Cancel</i18n.Translate>
          </button>
          <span
            class="has-tooltip-left"
            data-tooltip={i18n.str`Update product with new price`}
          >
            <button
              class="button is-info"
              onClick={() =>
                onUpdate({
                  ...product,
                  price: value.price as AmountString,
                })
              }
            >
              <i18n.Translate>Confirm update</i18n.Translate>
            </button>
          </span>
        </div>
      </div>
    </Fragment>
  );
}

function FastProductWithManagedStockUpdateForm({
  product,
  onUpdate,
  onCancel,
}: FastProductUpdateFormProps) {
  const [value, valueHandler] = useState<FastProductUpdate>({
    incoming: 0,
    lost: 0,
    price: product.price,
  });
  const { i18n } = useTranslationContext();

  const currentStock =
    product.total_stock - product.total_sold - product.total_lost;

  const errors: FormErrors<FastProductUpdate> = {
    lost:
      currentStock + value.incoming < value.lost
        ? i18n.str`lost can't be greater that current + incoming (max ${
            currentStock + value.incoming
          })`
        : undefined,
  };

  const hasErrors = Object.keys(errors).some(
    (k) => (errors as Record<string, unknown>)[k] !== undefined,
  );

  return (
    <Fragment>
      <FormProvider<FastProductUpdate>
        name="added"
        errors={errors}
        object={value}
        valueHandler={valueHandler as any}
      >
        <InputNumber<FastProductUpdate>
          name="incoming"
          label={i18n.str`Incoming`}
          tooltip={i18n.str`Add more elements to the inventory`}
        />
        <InputNumber<FastProductUpdate>
          name="lost"
          label={i18n.str`Lost`}
          tooltip={i18n.str`Report elements lost in the inventory`}
        />
        <InputCurrency<FastProductUpdate>
          name="price"
          label={i18n.str`Price per unit`}
          tooltip={i18n.str`New price for the product`}
        />
      </FormProvider>

      <div class="buttons is-right mt-5">
        <button class="button" onClick={onCancel}>
          <i18n.Translate>Cancel</i18n.Translate>
        </button>
        <span
          class="has-tooltip-left"
          data-tooltip={
            hasErrors
              ? i18n.str`There are values with errors.`
              : i18n.str`Update product with new stock and price`
          }
        >
          <button
            class="button is-info"
            disabled={hasErrors}
            onClick={() =>
              onUpdate({
                ...product,
                total_stock: product.total_stock + value.incoming,
                total_lost: product.total_lost + value.lost,
                price: value.price as AmountString,
              })
            }
          >
            <i18n.Translate>Confirm</i18n.Translate>
          </button>
        </span>
      </div>
    </Fragment>
  );
}

function FastProductUpdateForm(props: FastProductUpdateFormProps) {
  return props.product.total_stock === -1 ? (
    <FastProductWithInfiniteStockUpdateForm {...props} />
  ) : (
    <FastProductWithManagedStockUpdateForm {...props} />
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-magnify mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There are no products to list yet, add more by pressing the + sign
        </i18n.Translate>
      </p>
    </div>
  );
}

function difference(price: string, tax: number) {
  if (!tax) return price;
  const ps = price.split(":");
  const p = parseInt(ps[1], 10);
  ps[1] = `${p - tax}`;
  return ps.join(":");
}
function sum(taxes: TalerMerchantApi.Tax[] | undefined) {
  if (taxes === undefined) return 0;
  return taxes.reduce((p, c) => p + parseInt(c.tax.split(":")[1], 10), 0);
}
