/* $Id: cipher-gost.c,v 1.3 2002/07/09 21:22:09 kyle Exp $
 *
 * The GOST 28147-89 cipher
 *
 * This is based on the 25 November 1993 draft translation
 * by Aleksandr Malchik, with Whitfield Diffie, of the Government
 * Standard of the U.S.S.R. GOST 28149-89, "Cryptographic Transformation
 * Algorithm", effective 1 July 1990.  (Whitfield.Diffie@eng.sun.com)
 *
 * That is a draft, and may contain errors, which will be faithfully
 * reflected here, along with possible exciting new bugs.
 *
 * Some details have been cleared up by the paper "Soviet Encryption
 * Algorithm" by Josef Pieprzyk and Leonid Tombak of the University
 * of Wollongong, New South Wales.  (josef/leo@cs.adfa.oz.au)
 *
 * The standard is written by A. Zabotin (project leader), G.P. Glazkov,
 * and V.B. Isaeva.  It was accepted and introduced into use by the
 * action of the State Standards Committee of the USSR on 2 June 89 as
 * No. 1409.  It was to be reviewed in 1993, but whether anyone wishes
 * to take on this obligation from the USSR is questionable.
 *
 * This code is placed in the public domain.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/init.h>

#include <linux/crypto.h>
#include <linux/wordops.h>

#include <asm/byteorder.h>

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

#define PRECOMPUTE 1
#define FAST_SUBSTITUTION 1

/*
 * If you read the standard, it belabors the point of copying corresponding
 * bits from point A to point B quite a bit.  It helps to understand that
 * the standard is uniformly little-endian, although it numbers bits from
 * 1 rather than 0, so bit n has value 2^(n-1).  The least significant bit
 * of the 32-bit words that are manipulated in the algorithm is the first,
 * lowest-numbered, in the bit string.
 */

/*
 * The standard does not specify the contents of the 8 4 bit->4 bit
 * substitution boxes, saying they're a parameter of the network
 * being set up.  For illustration purposes here, I have used
 * the first rows of the 8 S-boxes from the DES.  (Note that the
 * DES S-boxes are numbered starting from 1 at the msb.  In keeping
 * with the rest of the GOST, I have used little-endian numbering.
 * Thus, k8 is S-box 1.
 *
 * Obviously, a careful look at the cryptographic properties of the cipher
 * must be undertaken before "production" substitution boxes are defined.
 *
 * The standard also does not specify a standard bit-string representation
 * for the contents of these blocks.
 */

#if !PRECOMPUTE
static u8 k8[16] = {
	14,  4, 13,  1,  2, 15, 11,  8,  3, 10,  6, 12,  5,  9,  0,  7 }; 
static u8 const k7[16] = {
	15,  1,  8, 14,  6, 11,  3,  4,  9,  7,  2, 13, 12,  0,  5, 10 };
static u8 const k6[16] = {
	10,  0,  9, 14,  6,  3, 15,  5,  1, 13, 12,  7, 11,  4,  2,  8 };
static u8 const k5[16] = {
	 7, 13, 14,  3,  0,  6,  9, 10,  1,  2,  8,  5, 11, 12,  4, 15 };
static u8 const k4[16] = {
	 2, 12,  4,  1,  7, 10, 11,  6,  8,  5,  3, 15, 13,  0, 14,  9 };
static u8 const k3[16] = {
	12,  1, 10, 15,  9,  2,  6,  8,  0, 13,  3,  4, 14,  7,  5, 11 };
static u8 const k2[16] = {
	 4, 11,  2, 14, 15,  0,  8, 13,  3, 12,  9,  7,  5, 10,  6,  1 };
static u8 const k1[16] = {
	13,  2,  8,  4,  6, 15, 11,  1, 10,  9,  3, 14,  5,  0, 12,  7 };

static u8 k87[256];
static u8 k65[256];
static u8 k43[256];
static u8 k21[256];

/*
 * Build byte-at-a-time subtitution tables.
 * This must be called once for global setup.
 */
static int
kboxinit(void)
{
	int i;
	for (i = 0; i < 256; i++) {
		k87[i] = k8[i >> 4] << 4 | k7[i & 15];
		k65[i] = k6[i >> 4] << 4 | k5[i & 15];
		k43[i] = k4[i >> 4] << 4 | k3[i & 15];
		k21[i] = k2[i >> 4] << 4 | k1[i & 15];
	}
}
#else /* #if PRECOMPUTE */
static u8 const k21[] = {
	0x4d, 0x42, 0x48, 0x44, 0x46, 0x4f, 0x4b, 0x41, 0x4a, 0x49, 0x43, 0x4e, 0x45, 0x40, 0x4c, 0x47,
	0xbd, 0xb2, 0xb8, 0xb4, 0xb6, 0xbf, 0xbb, 0xb1, 0xba, 0xb9, 0xb3, 0xbe, 0xb5, 0xb0, 0xbc, 0xb7,
	0x2d, 0x22, 0x28, 0x24, 0x26, 0x2f, 0x2b, 0x21, 0x2a, 0x29, 0x23, 0x2e, 0x25, 0x20, 0x2c, 0x27,
	0xed, 0xe2, 0xe8, 0xe4, 0xe6, 0xef, 0xeb, 0xe1, 0xea, 0xe9, 0xe3, 0xee, 0xe5, 0xe0, 0xec, 0xe7,
	0xfd, 0xf2, 0xf8, 0xf4, 0xf6, 0xff, 0xfb, 0xf1, 0xfa, 0xf9, 0xf3, 0xfe, 0xf5, 0xf0, 0xfc, 0xf7,
	0x0d, 0x02, 0x08, 0x04, 0x06, 0x0f, 0x0b, 0x01, 0x0a, 0x09, 0x03, 0x0e, 0x05, 0x00, 0x0c, 0x07,
	0x8d, 0x82, 0x88, 0x84, 0x86, 0x8f, 0x8b, 0x81, 0x8a, 0x89, 0x83, 0x8e, 0x85, 0x80, 0x8c, 0x87,
	0xdd, 0xd2, 0xd8, 0xd4, 0xd6, 0xdf, 0xdb, 0xd1, 0xda, 0xd9, 0xd3, 0xde, 0xd5, 0xd0, 0xdc, 0xd7,
	0x3d, 0x32, 0x38, 0x34, 0x36, 0x3f, 0x3b, 0x31, 0x3a, 0x39, 0x33, 0x3e, 0x35, 0x30, 0x3c, 0x37,
	0xcd, 0xc2, 0xc8, 0xc4, 0xc6, 0xcf, 0xcb, 0xc1, 0xca, 0xc9, 0xc3, 0xce, 0xc5, 0xc0, 0xcc, 0xc7,
	0x9d, 0x92, 0x98, 0x94, 0x96, 0x9f, 0x9b, 0x91, 0x9a, 0x99, 0x93, 0x9e, 0x95, 0x90, 0x9c, 0x97,
	0x7d, 0x72, 0x78, 0x74, 0x76, 0x7f, 0x7b, 0x71, 0x7a, 0x79, 0x73, 0x7e, 0x75, 0x70, 0x7c, 0x77,
	0x5d, 0x52, 0x58, 0x54, 0x56, 0x5f, 0x5b, 0x51, 0x5a, 0x59, 0x53, 0x5e, 0x55, 0x50, 0x5c, 0x57,
	0xad, 0xa2, 0xa8, 0xa4, 0xa6, 0xaf, 0xab, 0xa1, 0xaa, 0xa9, 0xa3, 0xae, 0xa5, 0xa0, 0xac, 0xa7,
	0x6d, 0x62, 0x68, 0x64, 0x66, 0x6f, 0x6b, 0x61, 0x6a, 0x69, 0x63, 0x6e, 0x65, 0x60, 0x6c, 0x67,
	0x1d, 0x12, 0x18, 0x14, 0x16, 0x1f, 0x1b, 0x11, 0x1a, 0x19, 0x13, 0x1e, 0x15, 0x10, 0x1c, 0x17,
};
static u8 const k43[] = {
	0x2c, 0x21, 0x2a, 0x2f, 0x29, 0x22, 0x26, 0x28, 0x20, 0x2d, 0x23, 0x24, 0x2e, 0x27, 0x25, 0x2b,
	0xcc, 0xc1, 0xca, 0xcf, 0xc9, 0xc2, 0xc6, 0xc8, 0xc0, 0xcd, 0xc3, 0xc4, 0xce, 0xc7, 0xc5, 0xcb,
	0x4c, 0x41, 0x4a, 0x4f, 0x49, 0x42, 0x46, 0x48, 0x40, 0x4d, 0x43, 0x44, 0x4e, 0x47, 0x45, 0x4b,
	0x1c, 0x11, 0x1a, 0x1f, 0x19, 0x12, 0x16, 0x18, 0x10, 0x1d, 0x13, 0x14, 0x1e, 0x17, 0x15, 0x1b,
	0x7c, 0x71, 0x7a, 0x7f, 0x79, 0x72, 0x76, 0x78, 0x70, 0x7d, 0x73, 0x74, 0x7e, 0x77, 0x75, 0x7b,
	0xac, 0xa1, 0xaa, 0xaf, 0xa9, 0xa2, 0xa6, 0xa8, 0xa0, 0xad, 0xa3, 0xa4, 0xae, 0xa7, 0xa5, 0xab,
	0xbc, 0xb1, 0xba, 0xbf, 0xb9, 0xb2, 0xb6, 0xb8, 0xb0, 0xbd, 0xb3, 0xb4, 0xbe, 0xb7, 0xb5, 0xbb,
	0x6c, 0x61, 0x6a, 0x6f, 0x69, 0x62, 0x66, 0x68, 0x60, 0x6d, 0x63, 0x64, 0x6e, 0x67, 0x65, 0x6b,
	0x8c, 0x81, 0x8a, 0x8f, 0x89, 0x82, 0x86, 0x88, 0x80, 0x8d, 0x83, 0x84, 0x8e, 0x87, 0x85, 0x8b,
	0x5c, 0x51, 0x5a, 0x5f, 0x59, 0x52, 0x56, 0x58, 0x50, 0x5d, 0x53, 0x54, 0x5e, 0x57, 0x55, 0x5b,
	0x3c, 0x31, 0x3a, 0x3f, 0x39, 0x32, 0x36, 0x38, 0x30, 0x3d, 0x33, 0x34, 0x3e, 0x37, 0x35, 0x3b,
	0xfc, 0xf1, 0xfa, 0xff, 0xf9, 0xf2, 0xf6, 0xf8, 0xf0, 0xfd, 0xf3, 0xf4, 0xfe, 0xf7, 0xf5, 0xfb,
	0xdc, 0xd1, 0xda, 0xdf, 0xd9, 0xd2, 0xd6, 0xd8, 0xd0, 0xdd, 0xd3, 0xd4, 0xde, 0xd7, 0xd5, 0xdb,
	0x0c, 0x01, 0x0a, 0x0f, 0x09, 0x02, 0x06, 0x08, 0x00, 0x0d, 0x03, 0x04, 0x0e, 0x07, 0x05, 0x0b,
	0xec, 0xe1, 0xea, 0xef, 0xe9, 0xe2, 0xe6, 0xe8, 0xe0, 0xed, 0xe3, 0xe4, 0xee, 0xe7, 0xe5, 0xeb,
	0x9c, 0x91, 0x9a, 0x9f, 0x99, 0x92, 0x96, 0x98, 0x90, 0x9d, 0x93, 0x94, 0x9e, 0x97, 0x95, 0x9b,
};
static u8 const k65[] = {
	0xa7, 0xad, 0xae, 0xa3, 0xa0, 0xa6, 0xa9, 0xaa, 0xa1, 0xa2, 0xa8, 0xa5, 0xab, 0xac, 0xa4, 0xaf,
	0x07, 0x0d, 0x0e, 0x03, 0x00, 0x06, 0x09, 0x0a, 0x01, 0x02, 0x08, 0x05, 0x0b, 0x0c, 0x04, 0x0f,
	0x97, 0x9d, 0x9e, 0x93, 0x90, 0x96, 0x99, 0x9a, 0x91, 0x92, 0x98, 0x95, 0x9b, 0x9c, 0x94, 0x9f,
	0xe7, 0xed, 0xee, 0xe3, 0xe0, 0xe6, 0xe9, 0xea, 0xe1, 0xe2, 0xe8, 0xe5, 0xeb, 0xec, 0xe4, 0xef,
	0x67, 0x6d, 0x6e, 0x63, 0x60, 0x66, 0x69, 0x6a, 0x61, 0x62, 0x68, 0x65, 0x6b, 0x6c, 0x64, 0x6f,
	0x37, 0x3d, 0x3e, 0x33, 0x30, 0x36, 0x39, 0x3a, 0x31, 0x32, 0x38, 0x35, 0x3b, 0x3c, 0x34, 0x3f,
	0xf7, 0xfd, 0xfe, 0xf3, 0xf0, 0xf6, 0xf9, 0xfa, 0xf1, 0xf2, 0xf8, 0xf5, 0xfb, 0xfc, 0xf4, 0xff,
	0x57, 0x5d, 0x5e, 0x53, 0x50, 0x56, 0x59, 0x5a, 0x51, 0x52, 0x58, 0x55, 0x5b, 0x5c, 0x54, 0x5f,
	0x17, 0x1d, 0x1e, 0x13, 0x10, 0x16, 0x19, 0x1a, 0x11, 0x12, 0x18, 0x15, 0x1b, 0x1c, 0x14, 0x1f,
	0xd7, 0xdd, 0xde, 0xd3, 0xd0, 0xd6, 0xd9, 0xda, 0xd1, 0xd2, 0xd8, 0xd5, 0xdb, 0xdc, 0xd4, 0xdf,
	0xc7, 0xcd, 0xce, 0xc3, 0xc0, 0xc6, 0xc9, 0xca, 0xc1, 0xc2, 0xc8, 0xc5, 0xcb, 0xcc, 0xc4, 0xcf,
	0x77, 0x7d, 0x7e, 0x73, 0x70, 0x76, 0x79, 0x7a, 0x71, 0x72, 0x78, 0x75, 0x7b, 0x7c, 0x74, 0x7f,
	0xb7, 0xbd, 0xbe, 0xb3, 0xb0, 0xb6, 0xb9, 0xba, 0xb1, 0xb2, 0xb8, 0xb5, 0xbb, 0xbc, 0xb4, 0xbf,
	0x47, 0x4d, 0x4e, 0x43, 0x40, 0x46, 0x49, 0x4a, 0x41, 0x42, 0x48, 0x45, 0x4b, 0x4c, 0x44, 0x4f,
	0x27, 0x2d, 0x2e, 0x23, 0x20, 0x26, 0x29, 0x2a, 0x21, 0x22, 0x28, 0x25, 0x2b, 0x2c, 0x24, 0x2f,
	0x87, 0x8d, 0x8e, 0x83, 0x80, 0x86, 0x89, 0x8a, 0x81, 0x82, 0x88, 0x85, 0x8b, 0x8c, 0x84, 0x8f,
};
static u8 const k87[] = {
	0xef, 0xe1, 0xe8, 0xee, 0xe6, 0xeb, 0xe3, 0xe4, 0xe9, 0xe7, 0xe2, 0xed, 0xec, 0xe0, 0xe5, 0xea,
	0x4f, 0x41, 0x48, 0x4e, 0x46, 0x4b, 0x43, 0x44, 0x49, 0x47, 0x42, 0x4d, 0x4c, 0x40, 0x45, 0x4a,
	0xdf, 0xd1, 0xd8, 0xde, 0xd6, 0xdb, 0xd3, 0xd4, 0xd9, 0xd7, 0xd2, 0xdd, 0xdc, 0xd0, 0xd5, 0xda,
	0x1f, 0x11, 0x18, 0x1e, 0x16, 0x1b, 0x13, 0x14, 0x19, 0x17, 0x12, 0x1d, 0x1c, 0x10, 0x15, 0x1a,
	0x2f, 0x21, 0x28, 0x2e, 0x26, 0x2b, 0x23, 0x24, 0x29, 0x27, 0x22, 0x2d, 0x2c, 0x20, 0x25, 0x2a,
	0xff, 0xf1, 0xf8, 0xfe, 0xf6, 0xfb, 0xf3, 0xf4, 0xf9, 0xf7, 0xf2, 0xfd, 0xfc, 0xf0, 0xf5, 0xfa,
	0xbf, 0xb1, 0xb8, 0xbe, 0xb6, 0xbb, 0xb3, 0xb4, 0xb9, 0xb7, 0xb2, 0xbd, 0xbc, 0xb0, 0xb5, 0xba,
	0x8f, 0x81, 0x88, 0x8e, 0x86, 0x8b, 0x83, 0x84, 0x89, 0x87, 0x82, 0x8d, 0x8c, 0x80, 0x85, 0x8a,
	0x3f, 0x31, 0x38, 0x3e, 0x36, 0x3b, 0x33, 0x34, 0x39, 0x37, 0x32, 0x3d, 0x3c, 0x30, 0x35, 0x3a,
	0xaf, 0xa1, 0xa8, 0xae, 0xa6, 0xab, 0xa3, 0xa4, 0xa9, 0xa7, 0xa2, 0xad, 0xac, 0xa0, 0xa5, 0xaa,
	0x6f, 0x61, 0x68, 0x6e, 0x66, 0x6b, 0x63, 0x64, 0x69, 0x67, 0x62, 0x6d, 0x6c, 0x60, 0x65, 0x6a,
	0xcf, 0xc1, 0xc8, 0xce, 0xc6, 0xcb, 0xc3, 0xc4, 0xc9, 0xc7, 0xc2, 0xcd, 0xcc, 0xc0, 0xc5, 0xca,
	0x5f, 0x51, 0x58, 0x5e, 0x56, 0x5b, 0x53, 0x54, 0x59, 0x57, 0x52, 0x5d, 0x5c, 0x50, 0x55, 0x5a,
	0x9f, 0x91, 0x98, 0x9e, 0x96, 0x9b, 0x93, 0x94, 0x99, 0x97, 0x92, 0x9d, 0x9c, 0x90, 0x95, 0x9a,
	0x0f, 0x01, 0x08, 0x0e, 0x06, 0x0b, 0x03, 0x04, 0x09, 0x07, 0x02, 0x0d, 0x0c, 0x00, 0x05, 0x0a,
	0x7f, 0x71, 0x78, 0x7e, 0x76, 0x7b, 0x73, 0x74, 0x79, 0x77, 0x72, 0x7d, 0x7c, 0x70, 0x75, 0x7a,
};
#endif /* #if !PRECOMPUTE */

/*
 * Do the substitution and rotation that are the core of the operation,
 * like the expansion, substitution and permutation of the DES.
 * It would be possible to perform DES-like optimisations and store
 * the table entries as 32-bit words, already rotated, but the
 * efficiency gain is questionable.
 *
 * This should be inlined for maximum speed
 */
static inline u32
f(u32 x)
{
	/* Do substitutions */
#if !FAST_SUBSTITUTION
	/* This is annoyingly slow */
	x = k8[x>>28 & 15] << 28 | k7[x>>24 & 15] << 24 |
	    k6[x>>20 & 15] << 20 | k5[x>>16 & 15] << 16 |
	    k4[x>>12 & 15] << 12 | k3[x>> 8 & 15] <<  8 |
	    k2[x>> 4 & 15] <<  4 | k1[x     & 15];
#else /* FAST_SUBSTITUTION */
	/* This is faster */
	x = k87[x>>24 & 255] << 24 | k65[x>>16 & 255] << 16 |
	    k43[x>> 8 & 255] <<  8 | k21[x & 255];
#endif /* !FAST_SUBSTITUTION */

	/* Rotate left 11 bits */
	return x<<11 | x>>(32-11);
}

/*
 * The GOST standard defines the input in terms of bits 1..64, with
 * bit 1 being the lsb of in[0] and bit 64 being the msb of in[1].
 *
 * The keys are defined similarly, with bit 256 being the msb of key[7].
 */
static void
GostEncrypt(u32 in[2], u32 out[2], u32 key[8])
{
	register u32 n1, n2; /* As named in the GOST */

	n1 = in[0];
	n2 = in[1];

	/* Instead of swapping halves, swap names each round */
	n2 ^= f(n1+key[0]);
	n1 ^= f(n2+key[1]);
	n2 ^= f(n1+key[2]);
	n1 ^= f(n2+key[3]);
	n2 ^= f(n1+key[4]);
	n1 ^= f(n2+key[5]);
	n2 ^= f(n1+key[6]);
	n1 ^= f(n2+key[7]);

	n2 ^= f(n1+key[0]);
	n1 ^= f(n2+key[1]);
	n2 ^= f(n1+key[2]);
	n1 ^= f(n2+key[3]);
	n2 ^= f(n1+key[4]);
	n1 ^= f(n2+key[5]);
	n2 ^= f(n1+key[6]);
	n1 ^= f(n2+key[7]);

	n2 ^= f(n1+key[0]);
	n1 ^= f(n2+key[1]);
	n2 ^= f(n1+key[2]);
	n1 ^= f(n2+key[3]);
	n2 ^= f(n1+key[4]);
	n1 ^= f(n2+key[5]);
	n2 ^= f(n1+key[6]);
	n1 ^= f(n2+key[7]);

	n2 ^= f(n1+key[7]);
	n1 ^= f(n2+key[6]);
	n2 ^= f(n1+key[5]);
	n1 ^= f(n2+key[4]);
	n2 ^= f(n1+key[3]);
	n1 ^= f(n2+key[2]);
	n2 ^= f(n1+key[1]);
	n1 ^= f(n2+key[0]);

	/* There is no swap after the last round */
	out[0] = n2;
	out[1] = n1;
}
	

/*
 * The key schedule is somewhat different for decryption.
 * (The key table is used once forward and three times backward.)
 * You could define an expanded key, or just write the code twice,
 * as done here.
 */
static void
GostDecrypt(u32 in[2], u32 out[2], u32 key[8])
{
	register u32 n1, n2; /* As named in the GOST */

	n1 = in[0];
	n2 = in[1];

	n2 ^= f(n1+key[0]);
	n1 ^= f(n2+key[1]);
	n2 ^= f(n1+key[2]);
	n1 ^= f(n2+key[3]);
	n2 ^= f(n1+key[4]);
	n1 ^= f(n2+key[5]);
	n2 ^= f(n1+key[6]);
	n1 ^= f(n2+key[7]);

	n2 ^= f(n1+key[7]);
	n1 ^= f(n2+key[6]);
	n2 ^= f(n1+key[5]);
	n1 ^= f(n2+key[4]);
	n2 ^= f(n1+key[3]);
	n1 ^= f(n2+key[2]);
	n2 ^= f(n1+key[1]);
	n1 ^= f(n2+key[0]);

	n2 ^= f(n1+key[7]);
	n1 ^= f(n2+key[6]);
	n2 ^= f(n1+key[5]);
	n1 ^= f(n2+key[4]);
	n2 ^= f(n1+key[3]);
	n1 ^= f(n2+key[2]);
	n2 ^= f(n1+key[1]);
	n1 ^= f(n2+key[0]);

	n2 ^= f(n1+key[7]);
	n1 ^= f(n2+key[6]);
	n2 ^= f(n1+key[5]);
	n1 ^= f(n2+key[4]);
	n2 ^= f(n1+key[3]);
	n1 ^= f(n2+key[2]);
	n2 ^= f(n1+key[1]);
	n1 ^= f(n2+key[0]);

	out[0] = n2;
	out[1] = n1;
}

static int gost_set_key(struct cipher_context *cx,
			const u8 *key, int key_len, int atomic)
{

	if (key_len != 32)
		return -EINVAL; /* unsupported key length */

	cx->key_length = key_len;
	memcpy(cx->keyinfo, key, key_len);

	return 0;
}

static int gost_encrypt(struct cipher_context *cx,
			const u8 *in, u8 *out, int size, int atomic)
{
	u8 input_buffer[8];
	u8 output_buffer[8];

	int sz;

	for (sz = size; sz >= 8; sz -= 8) {
		memcpy(&input_buffer, in, 8);
		in += 8;
		GostEncrypt((u32*)input_buffer,
			    (u32*)output_buffer, cx->keyinfo);
		memcpy(out, &output_buffer, 8);
		out += 8;
	}
	return 0;
}

static int gost_decrypt(struct cipher_context *cx,
			const u8 *in, u8 *out, int size, int atomic)
{
	u8 input_buffer[8];
	u8 output_buffer[8];

	int sz;

	for (sz = size; sz >= 8; sz -= 8) {
		memcpy(&input_buffer, in, 8);
		in += 8;
		GostDecrypt((u32*)input_buffer, 
			    (u32*)output_buffer, cx->keyinfo);
		memcpy(out, &output_buffer, 8);
		out += 8;
	}
	return 0;
}

#define CIPHER_ID                gost
#define CIPHER_BLOCKSIZE         64
#define CIPHER_KEY_SIZE_MASK     CIPHER_KEYSIZE_256
#define CIPHER_KEY_SCHEDULE_SIZE 32

#include "gen-cipher.h"

EXPORT_NO_SYMBOLS;
