// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * omap_cf.c -- OMAP 16xx CompactFlash controller driver
 *
 * Copyright (c) 2005 David Brownell
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/slab.h>

#include <pcmcia/ss.h>

#include <asm/io.h>
#include <linux/sizes.h>

#include <linux/soc/ti/omap1-io.h>
#include <linux/soc/ti/omap1-soc.h>
#include <linux/soc/ti/omap1-mux.h>

/* NOTE:  don't expect this to support many I/O cards.  The 16xx chips have
 * hard-wired timings to support Compact Flash memory cards; they won't work
 * with various other devices (like WLAN adapters) without some external
 * logic to help out.
 *
 * NOTE:  CF controller docs disagree with address space docs as to where
 * CF_BASE really lives; this is a doc erratum.
 */
#define	CF_BASE	0xfffe2800

/* status; read after IRQ */
#define CF_STATUS			(CF_BASE + 0x00)
#	define	CF_STATUS_BAD_READ	(1 << 2)
#	define	CF_STATUS_BAD_WRITE	(1 << 1)
#	define	CF_STATUS_CARD_DETECT	(1 << 0)

/* which chipselect (CS0..CS3) is used for CF (active low) */
#define CF_CFG				(CF_BASE + 0x02)

/* card reset */
#define CF_CONTROL			(CF_BASE + 0x04)
#	define	CF_CONTROL_RESET	(1 << 0)

#define omap_cf_present() (!(omap_readw(CF_STATUS) & CF_STATUS_CARD_DETECT))

/*--------------------------------------------------------------------------*/

static const char driver_name[] = "omap_cf";

struct omap_cf_socket {
	struct pcmcia_socket	socket;

	struct timer_list	timer;
	unsigned		present:1;
	unsigned		active:1;

	struct platform_device	*pdev;
	unsigned long		phys_cf;
	u_int			irq;
	struct resource		iomem;
};

#define	POLL_INTERVAL		(2 * HZ)

/*--------------------------------------------------------------------------*/

static int omap_cf_ss_init(struct pcmcia_socket *s)
{
	return 0;
}

/* the timer is primarily to kick this socket's pccardd */
static void omap_cf_timer(struct timer_list *t)
{
	struct omap_cf_socket	*cf = from_timer(cf, t, timer);
	unsigned		present = omap_cf_present();

	if (present != cf->present) {
		cf->present = present;
		pr_debug("%s: card %s\n", driver_name,
			present ? "present" : "gone");
		pcmcia_parse_events(&cf->socket, SS_DETECT);
	}

	if (cf->active)
		mod_timer(&cf->timer, jiffies + POLL_INTERVAL);
}

/* This irq handler prevents "irqNNN: nobody cared" messages as drivers
 * claim the card's IRQ.  It may also detect some card insertions, but
 * not removals; it can't always eliminate timer irqs.
 */
static irqreturn_t omap_cf_irq(int irq, void *_cf)
{
	struct omap_cf_socket *cf = (struct omap_cf_socket *)_cf;

	omap_cf_timer(&cf->timer);
	return IRQ_HANDLED;
}

static int omap_cf_get_status(struct pcmcia_socket *s, u_int *sp)
{
	if (!sp)
		return -EINVAL;

	/* NOTE CF is always 3VCARD */
	if (omap_cf_present()) {
		struct omap_cf_socket	*cf;

		*sp = SS_READY | SS_DETECT | SS_POWERON | SS_3VCARD;
		cf = container_of(s, struct omap_cf_socket, socket);
		s->pcmcia_irq = 0;
		s->pci_irq = cf->irq;
	} else
		*sp = 0;
	return 0;
}

static int
omap_cf_set_socket(struct pcmcia_socket *sock, struct socket_state_t *s)
{
	/* REVISIT some non-OSK boards may support power switching */
	switch (s->Vcc) {
	case 0:
	case 33:
		break;
	default:
		return -EINVAL;
	}

	omap_readw(CF_CONTROL);
	if (s->flags & SS_RESET)
		omap_writew(CF_CONTROL_RESET, CF_CONTROL);
	else
		omap_writew(0, CF_CONTROL);

	pr_debug("%s: Vcc %d, io_irq %d, flags %04x csc %04x\n",
		driver_name, s->Vcc, s->io_irq, s->flags, s->csc_mask);

	return 0;
}

static int omap_cf_ss_suspend(struct pcmcia_socket *s)
{
	pr_debug("%s: %s\n", driver_name, __func__);
	return omap_cf_set_socket(s, &dead_socket);
}

/* regions are 2K each:  mem, attrib, io (and reserved-for-ide) */

static int
omap_cf_set_io_map(struct pcmcia_socket *s, struct pccard_io_map *io)
{
	struct omap_cf_socket	*cf;

	cf = container_of(s, struct omap_cf_socket, socket);
	io->flags &= MAP_ACTIVE|MAP_ATTRIB|MAP_16BIT;
	io->start = cf->phys_cf + SZ_4K;
	io->stop = io->start + SZ_2K - 1;
	return 0;
}

static int
omap_cf_set_mem_map(struct pcmcia_socket *s, struct pccard_mem_map *map)
{
	struct omap_cf_socket	*cf;

	if (map->card_start)
		return -EINVAL;
	cf = container_of(s, struct omap_cf_socket, socket);
	map->static_start = cf->phys_cf;
	map->flags &= MAP_ACTIVE|MAP_ATTRIB|MAP_16BIT;
	if (map->flags & MAP_ATTRIB)
		map->static_start += SZ_2K;
	return 0;
}

static struct pccard_operations omap_cf_ops = {
	.init			= omap_cf_ss_init,
	.suspend		= omap_cf_ss_suspend,
	.get_status		= omap_cf_get_status,
	.set_socket		= omap_cf_set_socket,
	.set_io_map		= omap_cf_set_io_map,
	.set_mem_map		= omap_cf_set_mem_map,
};

/*--------------------------------------------------------------------------*/

/*
 * NOTE:  right now the only board-specific platform_data is
 * "what chipselect is used".  Boards could want more.
 */

static int __init omap_cf_probe(struct platform_device *pdev)
{
	unsigned		seg;
	struct omap_cf_socket	*cf;
	int			irq;
	int			status;
	struct resource		*res;
	struct resource		iospace = DEFINE_RES_IO(SZ_64, SZ_4K);

	seg = (int) pdev->dev.platform_data;
	if (seg == 0 || seg > 3)
		return -ENODEV;

	/* either CFLASH.IREQ (INT_1610_CF) or some GPIO */
	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return -EINVAL;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -EINVAL;

	cf = kzalloc(sizeof *cf, GFP_KERNEL);
	if (!cf)
		return -ENOMEM;
	timer_setup(&cf->timer, omap_cf_timer, 0);

	cf->pdev = pdev;
	platform_set_drvdata(pdev, cf);

	/* this primarily just shuts up irq handling noise */
	status = request_irq(irq, omap_cf_irq, IRQF_SHARED,
			driver_name, cf);
	if (status < 0)
		goto fail0;
	cf->irq = irq;
	cf->socket.pci_irq = irq;
	cf->phys_cf = res->start;

	/* pcmcia layer only remaps "real" memory */
	cf->socket.io_offset = iospace.start;
	status = pci_remap_iospace(&iospace, cf->phys_cf + SZ_4K);
	if (status) {
		status = -ENOMEM;
		goto fail1;
	}

	if (!request_mem_region(cf->phys_cf, SZ_8K, driver_name)) {
		status = -ENXIO;
		goto fail1;
	}

	/* NOTE:  CF conflicts with MMC1 */
	omap_cfg_reg(W11_1610_CF_CD1);
	omap_cfg_reg(P11_1610_CF_CD2);
	omap_cfg_reg(R11_1610_CF_IOIS16);
	omap_cfg_reg(V10_1610_CF_IREQ);
	omap_cfg_reg(W10_1610_CF_RESET);

	omap_writew(~(1 << seg), CF_CFG);

	pr_info("%s: cs%d on irq %d\n", driver_name, seg, irq);

	/* CF uses armxor_ck, which is "always" available */

	pr_debug("%s: sts %04x cfg %04x control %04x %s\n", driver_name,
		omap_readw(CF_STATUS), omap_readw(CF_CFG),
		omap_readw(CF_CONTROL),
		omap_cf_present() ? "present" : "(not present)");

	cf->socket.owner = THIS_MODULE;
	cf->socket.dev.parent = &pdev->dev;
	cf->socket.ops = &omap_cf_ops;
	cf->socket.resource_ops = &pccard_static_ops;
	cf->socket.features = SS_CAP_PCCARD | SS_CAP_STATIC_MAP
				| SS_CAP_MEM_ALIGN;
	cf->socket.map_size = SZ_2K;
	cf->socket.io[0].res = &cf->iomem;

	status = pcmcia_register_socket(&cf->socket);
	if (status < 0)
		goto fail2;

	cf->active = 1;
	mod_timer(&cf->timer, jiffies + POLL_INTERVAL);
	return 0;

fail2:
	release_mem_region(cf->phys_cf, SZ_8K);
fail1:
	free_irq(irq, cf);
fail0:
	kfree(cf);
	return status;
}

static int __exit omap_cf_remove(struct platform_device *pdev)
{
	struct omap_cf_socket *cf = platform_get_drvdata(pdev);

	cf->active = 0;
	pcmcia_unregister_socket(&cf->socket);
	del_timer_sync(&cf->timer);
	release_mem_region(cf->phys_cf, SZ_8K);
	free_irq(cf->irq, cf);
	kfree(cf);
	return 0;
}

/*
 * omap_cf_remove() lives in .exit.text. For drivers registered via
 * platform_driver_probe() this is ok because they cannot get unbound at
 * runtime. So mark the driver struct with __refdata to prevent modpost
 * triggering a section mismatch warning.
 */
static struct platform_driver omap_cf_driver __refdata = {
	.driver = {
		.name	= driver_name,
	},
	.remove		= __exit_p(omap_cf_remove),
};

static int __init omap_cf_init(void)
{
	if (cpu_is_omap16xx())
		return platform_driver_probe(&omap_cf_driver, omap_cf_probe);
	return -ENODEV;
}

static void __exit omap_cf_exit(void)
{
	if (cpu_is_omap16xx())
		platform_driver_unregister(&omap_cf_driver);
}

module_init(omap_cf_init);
module_exit(omap_cf_exit);

MODULE_DESCRIPTION("OMAP CF Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:omap_cf");
