/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Cedrus VPU driver
 *
 * Copyright (c) 2013-2016 Jens Kuske <jenskuske@gmail.com>
 * Copyright (C) 2016 Florent Revest <florent.revest@free-electrons.com>
 * Copyright (C) 2018 Paul Kocialkowski <paul.kocialkowski@bootlin.com>
 */

#ifndef _CEDRUS_REGS_H_
#define _CEDRUS_REGS_H_

#define SHIFT_AND_MASK_BITS(v, h, l) \
	(((unsigned long)(v) << (l)) & GENMASK(h, l))

/*
 * Common acronyms and contractions used in register descriptions:
 * * VLD : Variable-Length Decoder
 * * IQ: Inverse Quantization
 * * IDCT: Inverse Discrete Cosine Transform
 * * MC: Motion Compensation
 * * STCD: Start Code Detect
 * * SDRT: Scale Down and Rotate
 * * WB: Writeback
 * * BITS/BS: Bitstream
 * * MB: Macroblock
 * * CTU: Coding Tree Unit
 * * CTB: Coding Tree Block
 * * IDX: Index
 */

#define VE_ENGINE_DEC_MPEG			0x100
#define VE_ENGINE_DEC_H264			0x200
#define VE_ENGINE_DEC_H265			0x500

#define VE_MODE					0x00

#define VE_MODE_PIC_WIDTH_IS_4096		BIT(22)
#define VE_MODE_PIC_WIDTH_MORE_2048		BIT(21)
#define VE_MODE_REC_WR_MODE_2MB			(0x01 << 20)
#define VE_MODE_REC_WR_MODE_1MB			(0x00 << 20)
#define VE_MODE_DDR_MODE_BW_128			(0x03 << 16)
#define VE_MODE_DDR_MODE_BW_256			(0x02 << 16)
#define VE_MODE_DISABLED			(0x07 << 0)
#define VE_MODE_DEC_H265			(0x04 << 0)
#define VE_MODE_DEC_H264			(0x01 << 0)
#define VE_MODE_DEC_MPEG			(0x00 << 0)

#define VE_BUF_CTRL				0x50

#define VE_BUF_CTRL_INTRAPRED_EXT_RAM		(0x02 << 2)
#define VE_BUF_CTRL_INTRAPRED_MIXED_RAM		(0x01 << 2)
#define VE_BUF_CTRL_INTRAPRED_INT_SRAM		(0x00 << 2)
#define VE_BUF_CTRL_DBLK_EXT_RAM		(0x02 << 0)
#define VE_BUF_CTRL_DBLK_MIXED_RAM		(0x01 << 0)
#define VE_BUF_CTRL_DBLK_INT_SRAM		(0x00 << 0)

#define VE_DBLK_DRAM_BUF_ADDR			0x54
#define VE_INTRAPRED_DRAM_BUF_ADDR		0x58
#define VE_PRIMARY_CHROMA_BUF_LEN		0xc4
#define VE_PRIMARY_FB_LINE_STRIDE		0xc8

#define VE_PRIMARY_FB_LINE_STRIDE_CHROMA(s)	SHIFT_AND_MASK_BITS(s, 31, 16)
#define VE_PRIMARY_FB_LINE_STRIDE_LUMA(s)	SHIFT_AND_MASK_BITS(s, 15, 0)

#define VE_CHROMA_BUF_LEN			0xe8

#define VE_SECONDARY_OUT_FMT_TILED_32_NV12	(0x00 << 30)
#define VE_SECONDARY_OUT_FMT_EXT		(0x01 << 30)
#define VE_SECONDARY_OUT_FMT_YU12		(0x02 << 30)
#define VE_SECONDARY_OUT_FMT_YV12		(0x03 << 30)
#define VE_CHROMA_BUF_LEN_SDRT(l)		SHIFT_AND_MASK_BITS(l, 27, 0)

#define VE_PRIMARY_OUT_FMT			0xec

#define VE_PRIMARY_OUT_FMT_TILED_32_NV12	(0x00 << 4)
#define VE_PRIMARY_OUT_FMT_TILED_128_NV12	(0x01 << 4)
#define VE_PRIMARY_OUT_FMT_YU12			(0x02 << 4)
#define VE_PRIMARY_OUT_FMT_YV12			(0x03 << 4)
#define VE_PRIMARY_OUT_FMT_NV12			(0x04 << 4)
#define VE_PRIMARY_OUT_FMT_NV21			(0x05 << 4)
#define VE_SECONDARY_OUT_FMT_EXT_TILED_32_NV12	(0x00 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_TILED_128_NV12	(0x01 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_YU12		(0x02 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_YV12		(0x03 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_NV12		(0x04 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_NV21		(0x05 << 0)

#define VE_VERSION				0xf0

#define VE_VERSION_SHIFT			16

#define VE_DEC_MPEG_MP12HDR			(VE_ENGINE_DEC_MPEG + 0x00)

#define VE_DEC_MPEG_MP12HDR_SLICE_TYPE(t)	SHIFT_AND_MASK_BITS(t, 30, 28)
#define VE_DEC_MPEG_MP12HDR_F_CODE_SHIFT(x, y)	(24 - 4 * (y) - 8 * (x))
#define VE_DEC_MPEG_MP12HDR_F_CODE(__x, __y, __v) \
	(((unsigned long)(__v) & GENMASK(3, 0)) << VE_DEC_MPEG_MP12HDR_F_CODE_SHIFT(__x, __y))

#define VE_DEC_MPEG_MP12HDR_INTRA_DC_PRECISION(p) \
	SHIFT_AND_MASK_BITS(p, 11, 10)
#define VE_DEC_MPEG_MP12HDR_INTRA_PICTURE_STRUCTURE(s) \
	SHIFT_AND_MASK_BITS(s, 9, 8)
#define VE_DEC_MPEG_MP12HDR_TOP_FIELD_FIRST(v) \
	((v) ? BIT(7) : 0)
#define VE_DEC_MPEG_MP12HDR_FRAME_PRED_FRAME_DCT(v) \
	((v) ? BIT(6) : 0)
#define VE_DEC_MPEG_MP12HDR_CONCEALMENT_MOTION_VECTORS(v) \
	((v) ? BIT(5) : 0)
#define VE_DEC_MPEG_MP12HDR_Q_SCALE_TYPE(v) \
	((v) ? BIT(4) : 0)
#define VE_DEC_MPEG_MP12HDR_INTRA_VLC_FORMAT(v) \
	((v) ? BIT(3) : 0)
#define VE_DEC_MPEG_MP12HDR_ALTERNATE_SCAN(v) \
	((v) ? BIT(2) : 0)
#define VE_DEC_MPEG_MP12HDR_FULL_PEL_FORWARD_VECTOR(v) \
	((v) ? BIT(1) : 0)
#define VE_DEC_MPEG_MP12HDR_FULL_PEL_BACKWARD_VECTOR(v) \
	((v) ? BIT(0) : 0)

#define VE_DEC_MPEG_PICCODEDSIZE		(VE_ENGINE_DEC_MPEG + 0x08)

#define VE_DEC_MPEG_PICCODEDSIZE_WIDTH(w) \
	SHIFT_AND_MASK_BITS(DIV_ROUND_UP(w, 16), 15, 8)
#define VE_DEC_MPEG_PICCODEDSIZE_HEIGHT(h) \
	SHIFT_AND_MASK_BITS(DIV_ROUND_UP(h, 16), 7, 0)

#define VE_DEC_MPEG_PICBOUNDSIZE		(VE_ENGINE_DEC_MPEG + 0x0c)

#define VE_DEC_MPEG_PICBOUNDSIZE_WIDTH(w)	SHIFT_AND_MASK_BITS(w, 27, 16)
#define VE_DEC_MPEG_PICBOUNDSIZE_HEIGHT(h)	SHIFT_AND_MASK_BITS(h, 11, 0)

#define VE_DEC_MPEG_MBADDR			(VE_ENGINE_DEC_MPEG + 0x10)

#define VE_DEC_MPEG_MBADDR_X(w)			SHIFT_AND_MASK_BITS(w, 15, 8)
#define VE_DEC_MPEG_MBADDR_Y(h)			SHIFT_AND_MASK_BITS(h, 7, 0)

#define VE_DEC_MPEG_CTRL			(VE_ENGINE_DEC_MPEG + 0x14)

#define VE_DEC_MPEG_CTRL_MC_CACHE_EN		BIT(31)
#define VE_DEC_MPEG_CTRL_SW_VLD			BIT(27)
#define VE_DEC_MPEG_CTRL_SW_IQ_IS		BIT(17)
#define VE_DEC_MPEG_CTRL_QP_AC_DC_OUT_EN	BIT(14)
#define VE_DEC_MPEG_CTRL_ROTATE_SCALE_OUT_EN	BIT(8)
#define VE_DEC_MPEG_CTRL_MC_NO_WRITEBACK	BIT(7)
#define VE_DEC_MPEG_CTRL_ROTATE_IRQ_EN		BIT(6)
#define VE_DEC_MPEG_CTRL_VLD_DATA_REQ_IRQ_EN	BIT(5)
#define VE_DEC_MPEG_CTRL_ERROR_IRQ_EN		BIT(4)
#define VE_DEC_MPEG_CTRL_FINISH_IRQ_EN		BIT(3)
#define VE_DEC_MPEG_CTRL_IRQ_MASK \
	(VE_DEC_MPEG_CTRL_FINISH_IRQ_EN | VE_DEC_MPEG_CTRL_ERROR_IRQ_EN | \
	 VE_DEC_MPEG_CTRL_VLD_DATA_REQ_IRQ_EN)

#define VE_DEC_MPEG_TRIGGER			(VE_ENGINE_DEC_MPEG + 0x18)

#define VE_DEC_MPEG_TRIGGER_MB_BOUNDARY		BIT(31)

#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_420	(0x00 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_411	(0x01 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_422	(0x02 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_444	(0x03 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_422T	(0x04 << 27)

#define VE_DEC_MPEG_TRIGGER_MPEG1		(0x01 << 24)
#define VE_DEC_MPEG_TRIGGER_MPEG2		(0x02 << 24)
#define VE_DEC_MPEG_TRIGGER_JPEG		(0x03 << 24)
#define VE_DEC_MPEG_TRIGGER_MPEG4		(0x04 << 24)
#define VE_DEC_MPEG_TRIGGER_VP62		(0x05 << 24)

#define VE_DEC_MPEG_TRIGGER_VP62_AC_GET_BITS	BIT(7)

#define VE_DEC_MPEG_TRIGGER_STCD_VC1		(0x02 << 4)
#define VE_DEC_MPEG_TRIGGER_STCD_MPEG2		(0x01 << 4)
#define VE_DEC_MPEG_TRIGGER_STCD_AVC		(0x00 << 4)

#define VE_DEC_MPEG_TRIGGER_HW_MPEG_VLD		(0x0f << 0)
#define VE_DEC_MPEG_TRIGGER_HW_JPEG_VLD		(0x0e << 0)
#define VE_DEC_MPEG_TRIGGER_HW_MB		(0x0d << 0)
#define VE_DEC_MPEG_TRIGGER_HW_ROTATE		(0x0c << 0)
#define VE_DEC_MPEG_TRIGGER_HW_VP6_VLD		(0x0b << 0)
#define VE_DEC_MPEG_TRIGGER_HW_MAF		(0x0a << 0)
#define VE_DEC_MPEG_TRIGGER_HW_STCD_END		(0x09 << 0)
#define VE_DEC_MPEG_TRIGGER_HW_STCD_BEGIN	(0x08 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_MC		(0x07 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_IQ		(0x06 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_IDCT		(0x05 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_SCALE		(0x04 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_VP6		(0x03 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_VP62_AC_GET_BITS	(0x02 << 0)

#define VE_DEC_MPEG_STATUS			(VE_ENGINE_DEC_MPEG + 0x1c)

#define VE_DEC_MPEG_STATUS_START_DETECT_BUSY	BIT(27)
#define VE_DEC_MPEG_STATUS_VP6_BIT		BIT(26)
#define VE_DEC_MPEG_STATUS_VP6_BIT_BUSY		BIT(25)
#define VE_DEC_MPEG_STATUS_MAF_BUSY		BIT(23)
#define VE_DEC_MPEG_STATUS_VP6_MVP_BUSY		BIT(22)
#define VE_DEC_MPEG_STATUS_JPEG_BIT_END		BIT(21)
#define VE_DEC_MPEG_STATUS_JPEG_RESTART_ERROR	BIT(20)
#define VE_DEC_MPEG_STATUS_JPEG_MARKER		BIT(19)
#define VE_DEC_MPEG_STATUS_ROTATE_BUSY		BIT(18)
#define VE_DEC_MPEG_STATUS_DEBLOCKING_BUSY	BIT(17)
#define VE_DEC_MPEG_STATUS_SCALE_DOWN_BUSY	BIT(16)
#define VE_DEC_MPEG_STATUS_IQIS_BUF_EMPTY	BIT(15)
#define VE_DEC_MPEG_STATUS_IDCT_BUF_EMPTY	BIT(14)
#define VE_DEC_MPEG_STATUS_VE_BUSY		BIT(13)
#define VE_DEC_MPEG_STATUS_MC_BUSY		BIT(12)
#define VE_DEC_MPEG_STATUS_IDCT_BUSY		BIT(11)
#define VE_DEC_MPEG_STATUS_IQIS_BUSY		BIT(10)
#define VE_DEC_MPEG_STATUS_DCAC_BUSY		BIT(9)
#define VE_DEC_MPEG_STATUS_VLD_BUSY		BIT(8)
#define VE_DEC_MPEG_STATUS_ROTATE_SUCCESS	BIT(3)
#define VE_DEC_MPEG_STATUS_VLD_DATA_REQ		BIT(2)
#define VE_DEC_MPEG_STATUS_ERROR		BIT(1)
#define VE_DEC_MPEG_STATUS_SUCCESS		BIT(0)
#define VE_DEC_MPEG_STATUS_CHECK_MASK \
	(VE_DEC_MPEG_STATUS_SUCCESS | VE_DEC_MPEG_STATUS_ERROR | \
	 VE_DEC_MPEG_STATUS_VLD_DATA_REQ)
#define VE_DEC_MPEG_STATUS_CHECK_ERROR \
	(VE_DEC_MPEG_STATUS_ERROR | VE_DEC_MPEG_STATUS_VLD_DATA_REQ)

#define VE_DEC_MPEG_VLD_ADDR			(VE_ENGINE_DEC_MPEG + 0x28)

#define VE_DEC_MPEG_VLD_ADDR_FIRST_PIC_DATA	BIT(30)
#define VE_DEC_MPEG_VLD_ADDR_LAST_PIC_DATA	BIT(29)
#define VE_DEC_MPEG_VLD_ADDR_VALID_PIC_DATA	BIT(28)
#define VE_DEC_MPEG_VLD_ADDR_BASE(a)					\
	({								\
		u32 _tmp = (a);						\
		u32 _lo = _tmp & GENMASK(27, 4);			\
		u32 _hi = (_tmp >> 28) & GENMASK(3, 0);			\
		(_lo | _hi);						\
	})

#define VE_DEC_MPEG_VLD_OFFSET			(VE_ENGINE_DEC_MPEG + 0x2c)
#define VE_DEC_MPEG_VLD_LEN			(VE_ENGINE_DEC_MPEG + 0x30)
#define VE_DEC_MPEG_VLD_END_ADDR		(VE_ENGINE_DEC_MPEG + 0x34)

#define VE_DEC_MPEG_REC_LUMA			(VE_ENGINE_DEC_MPEG + 0x48)
#define VE_DEC_MPEG_REC_CHROMA			(VE_ENGINE_DEC_MPEG + 0x4c)
#define VE_DEC_MPEG_FWD_REF_LUMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x50)
#define VE_DEC_MPEG_FWD_REF_CHROMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x54)
#define VE_DEC_MPEG_BWD_REF_LUMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x58)
#define VE_DEC_MPEG_BWD_REF_CHROMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x5c)

#define VE_DEC_MPEG_IQMINPUT			(VE_ENGINE_DEC_MPEG + 0x80)

#define VE_DEC_MPEG_IQMINPUT_FLAG_INTRA		(0x01 << 14)
#define VE_DEC_MPEG_IQMINPUT_FLAG_NON_INTRA	(0x00 << 14)
#define VE_DEC_MPEG_IQMINPUT_WEIGHT(i, v) \
	(SHIFT_AND_MASK_BITS(i, 13, 8) | SHIFT_AND_MASK_BITS(v, 7, 0))

#define VE_DEC_MPEG_ERROR			(VE_ENGINE_DEC_MPEG + 0xc4)
#define VE_DEC_MPEG_CRTMBADDR			(VE_ENGINE_DEC_MPEG + 0xc8)
#define VE_DEC_MPEG_ROT_LUMA			(VE_ENGINE_DEC_MPEG + 0xcc)
#define VE_DEC_MPEG_ROT_CHROMA			(VE_ENGINE_DEC_MPEG + 0xd0)

#define VE_DEC_H265_DEC_NAL_HDR			(VE_ENGINE_DEC_H265 + 0x00)

#define VE_DEC_H265_DEC_NAL_HDR_NUH_TEMPORAL_ID_PLUS1(v) \
	SHIFT_AND_MASK_BITS(v, 8, 6)
#define VE_DEC_H265_DEC_NAL_HDR_NAL_UNIT_TYPE(v) \
	SHIFT_AND_MASK_BITS(v, 5, 0)

#define VE_DEC_H265_FLAG(reg_flag, ctrl_flag, flags) \
	(((flags) & (ctrl_flag)) ? reg_flag : 0)

#define VE_DEC_H265_DEC_SPS_HDR			(VE_ENGINE_DEC_H265 + 0x04)

#define VE_DEC_H265_DEC_SPS_HDR_FLAG_STRONG_INTRA_SMOOTHING_ENABLE	BIT(26)
#define VE_DEC_H265_DEC_SPS_HDR_FLAG_SPS_TEMPORAL_MVP_ENABLED		BIT(25)
#define VE_DEC_H265_DEC_SPS_HDR_FLAG_SAMPLE_ADAPTIVE_OFFSET_ENABLED	BIT(24)
#define VE_DEC_H265_DEC_SPS_HDR_FLAG_AMP_ENABLED			BIT(23)
#define VE_DEC_H265_DEC_SPS_HDR_FLAG_SEPARATE_COLOUR_PLANE		BIT(2)

#define VE_DEC_H265_DEC_SPS_HDR_MAX_TRANSFORM_HIERARCHY_DEPTH_INTRA(v) \
	SHIFT_AND_MASK_BITS(v, 22, 20)
#define VE_DEC_H265_DEC_SPS_HDR_MAX_TRANSFORM_HIERARCHY_DEPTH_INTER(v) \
	SHIFT_AND_MASK_BITS(v, 19, 17)
#define VE_DEC_H265_DEC_SPS_HDR_LOG2_DIFF_MAX_MIN_TRANSFORM_BLOCK_SIZE(v) \
	SHIFT_AND_MASK_BITS(v, 16, 15)
#define VE_DEC_H265_DEC_SPS_HDR_LOG2_MIN_TRANSFORM_BLOCK_SIZE_MINUS2(v) \
	SHIFT_AND_MASK_BITS(v, 14, 13)
#define VE_DEC_H265_DEC_SPS_HDR_LOG2_DIFF_MAX_MIN_LUMA_CODING_BLOCK_SIZE(v) \
	SHIFT_AND_MASK_BITS(v, 12, 11)
#define VE_DEC_H265_DEC_SPS_HDR_LOG2_MIN_LUMA_CODING_BLOCK_SIZE_MINUS3(v) \
	SHIFT_AND_MASK_BITS(v, 10, 9)
#define VE_DEC_H265_DEC_SPS_HDR_BIT_DEPTH_CHROMA_MINUS8(v) \
	SHIFT_AND_MASK_BITS(v, 8, 6)
#define VE_DEC_H265_DEC_SPS_HDR_BIT_DEPTH_LUMA_MINUS8(v) \
	SHIFT_AND_MASK_BITS(v, 5, 3)
#define VE_DEC_H265_DEC_SPS_HDR_CHROMA_FORMAT_IDC(v) \
	SHIFT_AND_MASK_BITS(v, 1, 0)

#define VE_DEC_H265_DEC_PIC_SIZE		(VE_ENGINE_DEC_H265 + 0x08)

#define VE_DEC_H265_DEC_PIC_SIZE_WIDTH(w)	(((w) << 0) & GENMASK(13, 0))
#define VE_DEC_H265_DEC_PIC_SIZE_HEIGHT(h)	(((h) << 16) & GENMASK(29, 16))

#define VE_DEC_H265_DEC_PCM_CTRL		(VE_ENGINE_DEC_H265 + 0x0c)

#define VE_DEC_H265_DEC_PCM_CTRL_FLAG_PCM_ENABLED		BIT(15)
#define VE_DEC_H265_DEC_PCM_CTRL_FLAG_PCM_LOOP_FILTER_DISABLED	BIT(14)

#define VE_DEC_H265_DEC_PCM_CTRL_LOG2_DIFF_MAX_MIN_PCM_LUMA_CODING_BLOCK_SIZE(v) \
	SHIFT_AND_MASK_BITS(v, 11, 10)
#define VE_DEC_H265_DEC_PCM_CTRL_LOG2_MIN_PCM_LUMA_CODING_BLOCK_SIZE_MINUS3(v) \
	SHIFT_AND_MASK_BITS(v, 9, 8)
#define VE_DEC_H265_DEC_PCM_CTRL_PCM_SAMPLE_BIT_DEPTH_CHROMA_MINUS1(v) \
	SHIFT_AND_MASK_BITS(v, 7, 4)
#define VE_DEC_H265_DEC_PCM_CTRL_PCM_SAMPLE_BIT_DEPTH_LUMA_MINUS1(v) \
	SHIFT_AND_MASK_BITS(v, 3, 0)

#define VE_DEC_H265_DEC_PPS_CTRL0		(VE_ENGINE_DEC_H265 + 0x10)

#define VE_DEC_H265_DEC_PPS_CTRL0_FLAG_CU_QP_DELTA_ENABLED	BIT(3)
#define VE_DEC_H265_DEC_PPS_CTRL0_FLAG_TRANSFORM_SKIP_ENABLED	BIT(2)
#define VE_DEC_H265_DEC_PPS_CTRL0_FLAG_CONSTRAINED_INTRA_PRED	BIT(1)
#define VE_DEC_H265_DEC_PPS_CTRL0_FLAG_SIGN_DATA_HIDING_ENABLED	BIT(0)

#define VE_DEC_H265_DEC_PPS_CTRL0_PPS_CR_QP_OFFSET(v) \
	SHIFT_AND_MASK_BITS(v, 29, 24)
#define VE_DEC_H265_DEC_PPS_CTRL0_PPS_CB_QP_OFFSET(v) \
	SHIFT_AND_MASK_BITS(v, 21, 16)
#define VE_DEC_H265_DEC_PPS_CTRL0_INIT_QP_MINUS26(v) \
	SHIFT_AND_MASK_BITS(v, 14, 8)
#define VE_DEC_H265_DEC_PPS_CTRL0_DIFF_CU_QP_DELTA_DEPTH(v) \
	SHIFT_AND_MASK_BITS(v, 5, 4)

#define VE_DEC_H265_DEC_PPS_CTRL1		(VE_ENGINE_DEC_H265 + 0x14)

#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_PPS_LOOP_FILTER_ACROSS_SLICES_ENABLED BIT(6)
#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_LOOP_FILTER_ACROSS_TILES_ENABLED	BIT(5)
#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_ENTROPY_CODING_SYNC_ENABLED	BIT(4)
#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_TILES_ENABLED			BIT(3)
#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_TRANSQUANT_BYPASS_ENABLED	BIT(2)
#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_WEIGHTED_BIPRED			BIT(1)
#define VE_DEC_H265_DEC_PPS_CTRL1_FLAG_WEIGHTED_PRED			BIT(0)

#define VE_DEC_H265_DEC_PPS_CTRL1_LOG2_PARALLEL_MERGE_LEVEL_MINUS2(v) \
	SHIFT_AND_MASK_BITS(v, 10, 8)

#define VE_DEC_H265_SCALING_LIST_CTRL0		(VE_ENGINE_DEC_H265 + 0x18)

#define VE_DEC_H265_SCALING_LIST_CTRL0_FLAG_ENABLED			BIT(31)

#define VE_DEC_H265_SCALING_LIST_CTRL0_SRAM	(0 << 30)
#define VE_DEC_H265_SCALING_LIST_CTRL0_DEFAULT	(1 << 30)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO0		(VE_ENGINE_DEC_H265 + 0x20)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_COLLOCATED_FROM_L0		BIT(11)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_CABAC_INIT			BIT(10)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_MVD_L1_ZERO		BIT(9)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_SLICE_SAO_CHROMA		BIT(8)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_SLICE_SAO_LUMA		BIT(7)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_SLICE_TEMPORAL_MVP_ENABLE	BIT(6)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_DEPENDENT_SLICE_SEGMENT	BIT(1)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FLAG_FIRST_SLICE_SEGMENT_IN_PIC	BIT(0)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_PICTURE_TYPE(v) \
	SHIFT_AND_MASK_BITS(v, 29, 28)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_FIVE_MINUS_MAX_NUM_MERGE_CAND(v) \
	SHIFT_AND_MASK_BITS(v, 26, 24)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_NUM_REF_IDX_L1_ACTIVE_MINUS1(v) \
	SHIFT_AND_MASK_BITS(v, 23, 20)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_NUM_REF_IDX_L0_ACTIVE_MINUS1(v) \
	SHIFT_AND_MASK_BITS(v, 19, 16)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_COLLOCATED_REF_IDX(v) \
	SHIFT_AND_MASK_BITS(v, 15, 12)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_COLOUR_PLANE_ID(v) \
	SHIFT_AND_MASK_BITS(v, 5, 4)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO0_SLICE_TYPE(v) \
	SHIFT_AND_MASK_BITS(v, 3, 2)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO1		(VE_ENGINE_DEC_H265 + 0x24)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_FLAG_SLICE_DEBLOCKING_FILTER_DISABLED BIT(23)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_FLAG_SLICE_LOOP_FILTER_ACROSS_SLICES_ENABLED BIT(22)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_FLAG_SLICE_NOT_LOW_DELAY BIT(21)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_SLICE_TC_OFFSET_DIV2(v) \
	SHIFT_AND_MASK_BITS(v, 31, 28)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_SLICE_BETA_OFFSET_DIV2(v) \
	SHIFT_AND_MASK_BITS(v, 27, 24)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_SLICE_CR_QP_OFFSET(v) \
	SHIFT_AND_MASK_BITS(v, 20, 16)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_SLICE_CB_QP_OFFSET(v) \
	SHIFT_AND_MASK_BITS(v, 12, 8)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO1_SLICE_QP_DELTA(v) \
	SHIFT_AND_MASK_BITS(v, 6, 0)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO2		(VE_ENGINE_DEC_H265 + 0x28)

#define VE_DEC_H265_DEC_SLICE_HDR_INFO2_NUM_ENTRY_POINT_OFFSETS(v) \
	SHIFT_AND_MASK_BITS(v, 21, 8)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO2_CHROMA_LOG2_WEIGHT_DENOM(v) \
	SHIFT_AND_MASK_BITS(v, 6, 4)
#define VE_DEC_H265_DEC_SLICE_HDR_INFO2_LUMA_LOG2_WEIGHT_DENOM(v) \
	SHIFT_AND_MASK_BITS(v, 2, 0)

#define VE_DEC_H265_DEC_CTB_ADDR		(VE_ENGINE_DEC_H265 + 0x2c)

#define VE_DEC_H265_DEC_CTB_ADDR_Y(y)		SHIFT_AND_MASK_BITS(y, 25, 16)
#define VE_DEC_H265_DEC_CTB_ADDR_X(x)		SHIFT_AND_MASK_BITS(x, 9, 0)

#define VE_DEC_H265_CTRL			(VE_ENGINE_DEC_H265 + 0x30)

#define VE_DEC_H265_CTRL_DDR_CONSISTENCY_EN	BIT(31)
#define VE_DEC_H265_CTRL_STCD_EN		BIT(25)
#define VE_DEC_H265_CTRL_EPTB_DEC_BYPASS_EN	BIT(24)
#define VE_DEC_H265_CTRL_TQ_BYPASS_EN		BIT(12)
#define VE_DEC_H265_CTRL_VLD_BYPASS_EN		BIT(11)
#define VE_DEC_H265_CTRL_NCRI_CACHE_DISABLE	BIT(10)
#define VE_DEC_H265_CTRL_ROTATE_SCALE_OUT_EN	BIT(9)
#define VE_DEC_H265_CTRL_MC_NO_WRITEBACK	BIT(8)
#define VE_DEC_H265_CTRL_VLD_DATA_REQ_IRQ_EN	BIT(2)
#define VE_DEC_H265_CTRL_ERROR_IRQ_EN		BIT(1)
#define VE_DEC_H265_CTRL_FINISH_IRQ_EN		BIT(0)
#define VE_DEC_H265_CTRL_IRQ_MASK \
	(VE_DEC_H265_CTRL_FINISH_IRQ_EN | VE_DEC_H265_CTRL_ERROR_IRQ_EN | \
	 VE_DEC_H265_CTRL_VLD_DATA_REQ_IRQ_EN)

#define VE_DEC_H265_TRIGGER			(VE_ENGINE_DEC_H265 + 0x34)

#define VE_DEC_H265_TRIGGER_TYPE_N_BITS(x)	(((x) & 0x3f) << 8)
#define VE_DEC_H265_TRIGGER_STCD_VC1		(0x02 << 4)
#define VE_DEC_H265_TRIGGER_STCD_AVS		(0x01 << 4)
#define VE_DEC_H265_TRIGGER_STCD_HEVC		(0x00 << 4)
#define VE_DEC_H265_TRIGGER_DEC_SLICE		(0x08 << 0)
#define VE_DEC_H265_TRIGGER_INIT_SWDEC		(0x07 << 0)
#define VE_DEC_H265_TRIGGER_BYTE_ALIGN		(0x06 << 0)
#define VE_DEC_H265_TRIGGER_GET_VLCUE		(0x05 << 0)
#define VE_DEC_H265_TRIGGER_GET_VLCSE		(0x04 << 0)
#define VE_DEC_H265_TRIGGER_FLUSH_BITS		(0x03 << 0)
#define VE_DEC_H265_TRIGGER_GET_BITS		(0x02 << 0)
#define VE_DEC_H265_TRIGGER_SHOW_BITS		(0x01 << 0)

#define VE_DEC_H265_STATUS			(VE_ENGINE_DEC_H265 + 0x38)

#define VE_DEC_H265_STATUS_STCD			BIT(24)
#define VE_DEC_H265_STATUS_STCD_BUSY		BIT(21)
#define VE_DEC_H265_STATUS_WB_BUSY		BIT(20)
#define VE_DEC_H265_STATUS_BS_DMA_BUSY		BIT(19)
#define VE_DEC_H265_STATUS_IT_BUSY		BIT(18)
#define VE_DEC_H265_STATUS_INTER_BUSY		BIT(17)
#define VE_DEC_H265_STATUS_MORE_DATA		BIT(16)
#define VE_DEC_H265_STATUS_DBLK_BUSY		BIT(15)
#define VE_DEC_H265_STATUS_IREC_BUSY		BIT(14)
#define VE_DEC_H265_STATUS_INTRA_BUSY		BIT(13)
#define VE_DEC_H265_STATUS_MCRI_BUSY		BIT(12)
#define VE_DEC_H265_STATUS_IQIT_BUSY		BIT(11)
#define VE_DEC_H265_STATUS_MVP_BUSY		BIT(10)
#define VE_DEC_H265_STATUS_IS_BUSY		BIT(9)
#define VE_DEC_H265_STATUS_VLD_BUSY		BIT(8)
#define VE_DEC_H265_STATUS_OVER_TIME		BIT(3)
#define VE_DEC_H265_STATUS_VLD_DATA_REQ		BIT(2)
#define VE_DEC_H265_STATUS_ERROR		BIT(1)
#define VE_DEC_H265_STATUS_SUCCESS		BIT(0)
#define VE_DEC_H265_STATUS_STCD_TYPE_MASK	GENMASK(23, 22)
#define VE_DEC_H265_STATUS_CHECK_MASK \
	(VE_DEC_H265_STATUS_SUCCESS | VE_DEC_H265_STATUS_ERROR | \
	 VE_DEC_H265_STATUS_VLD_DATA_REQ)
#define VE_DEC_H265_STATUS_CHECK_ERROR \
	(VE_DEC_H265_STATUS_ERROR | VE_DEC_H265_STATUS_VLD_DATA_REQ)

#define VE_DEC_H265_DEC_CTB_NUM			(VE_ENGINE_DEC_H265 + 0x3c)

#define VE_DEC_H265_BITS_ADDR			(VE_ENGINE_DEC_H265 + 0x40)

#define VE_DEC_H265_BITS_ADDR_FIRST_SLICE_DATA	BIT(30)
#define VE_DEC_H265_BITS_ADDR_LAST_SLICE_DATA	BIT(29)
#define VE_DEC_H265_BITS_ADDR_VALID_SLICE_DATA	BIT(28)
#define VE_DEC_H265_BITS_ADDR_BASE(a)		(((a) >> 8) & GENMASK(27, 0))

#define VE_DEC_H265_BITS_OFFSET			(VE_ENGINE_DEC_H265 + 0x44)
#define VE_DEC_H265_BITS_LEN			(VE_ENGINE_DEC_H265 + 0x48)

#define VE_DEC_H265_BITS_END_ADDR		(VE_ENGINE_DEC_H265 + 0x4c)

#define VE_DEC_H265_BITS_END_ADDR_BASE(a)	((a) >> 8)

#define VE_DEC_H265_SDRT_CTRL			(VE_ENGINE_DEC_H265 + 0x50)
#define VE_DEC_H265_SDRT_LUMA_ADDR		(VE_ENGINE_DEC_H265 + 0x54)
#define VE_DEC_H265_SDRT_CHROMA_ADDR		(VE_ENGINE_DEC_H265 + 0x58)

#define VE_DEC_H265_OUTPUT_FRAME_IDX		(VE_ENGINE_DEC_H265 + 0x5c)

#define VE_DEC_H265_NEIGHBOR_INFO_ADDR		(VE_ENGINE_DEC_H265 + 0x60)

#define VE_DEC_H265_NEIGHBOR_INFO_ADDR_BASE(a)	((a) >> 8)

#define VE_DEC_H265_ENTRY_POINT_OFFSET_ADDR	(VE_ENGINE_DEC_H265 + 0x64)
#define VE_DEC_H265_TILE_START_CTB		(VE_ENGINE_DEC_H265 + 0x68)
#define VE_DEC_H265_TILE_END_CTB		(VE_ENGINE_DEC_H265 + 0x6c)
#define VE_DEC_H265_SCALING_LIST_DC_COEF0	(VE_ENGINE_DEC_H265 + 0x78)
#define VE_DEC_H265_SCALING_LIST_DC_COEF1	(VE_ENGINE_DEC_H265 + 0x7c)

#define VE_DEC_H265_LOW_ADDR			(VE_ENGINE_DEC_H265 + 0x80)

#define VE_DEC_H265_LOW_ADDR_PRIMARY_CHROMA(a) \
	SHIFT_AND_MASK_BITS(a, 31, 24)
#define VE_DEC_H265_LOW_ADDR_SECONDARY_CHROMA(a) \
	SHIFT_AND_MASK_BITS(a, 23, 16)
#define VE_DEC_H265_LOW_ADDR_ENTRY_POINTS_BUF(a) \
	SHIFT_AND_MASK_BITS(a, 7, 0)

#define VE_DEC_H265_BITS_READ			(VE_ENGINE_DEC_H265 + 0xdc)

#define VE_DEC_H265_SRAM_OFFSET			(VE_ENGINE_DEC_H265 + 0xe0)

#define VE_DEC_H265_SRAM_OFFSET_PRED_WEIGHT_LUMA_L0	0x00
#define VE_DEC_H265_SRAM_OFFSET_PRED_WEIGHT_CHROMA_L0	0x20
#define VE_DEC_H265_SRAM_OFFSET_PRED_WEIGHT_LUMA_L1	0x60
#define VE_DEC_H265_SRAM_OFFSET_PRED_WEIGHT_CHROMA_L1	0x80
#define VE_DEC_H265_SRAM_OFFSET_FRAME_INFO		0x400
#define VE_DEC_H265_SRAM_OFFSET_FRAME_INFO_UNIT		0x20
#define VE_DEC_H265_SRAM_OFFSET_SCALING_LISTS		0x800
#define VE_DEC_H265_SRAM_OFFSET_REF_PIC_LIST0		0xc00
#define VE_DEC_H265_SRAM_OFFSET_REF_PIC_LIST1		0xc10

#define VE_DEC_H265_SRAM_DATA			(VE_ENGINE_DEC_H265 + 0xe4)

#define VE_DEC_H265_SRAM_DATA_ADDR_BASE(a)	((a) >> 8)
#define VE_DEC_H265_SRAM_REF_PIC_LIST_LT_REF	BIT(7)

#define VE_H264_SPS			0x200
#define VE_H264_SPS_MBS_ONLY			BIT(18)
#define VE_H264_SPS_MB_ADAPTIVE_FRAME_FIELD	BIT(17)
#define VE_H264_SPS_DIRECT_8X8_INFERENCE	BIT(16)

#define VE_H264_PPS			0x204
#define VE_H264_PPS_ENTROPY_CODING_MODE		BIT(15)
#define VE_H264_PPS_WEIGHTED_PRED		BIT(4)
#define VE_H264_PPS_CONSTRAINED_INTRA_PRED	BIT(1)
#define VE_H264_PPS_TRANSFORM_8X8_MODE		BIT(0)

#define VE_H264_SHS			0x208
#define VE_H264_SHS_FIRST_SLICE_IN_PIC		BIT(5)
#define VE_H264_SHS_FIELD_PIC			BIT(4)
#define VE_H264_SHS_BOTTOM_FIELD		BIT(3)
#define VE_H264_SHS_DIRECT_SPATIAL_MV_PRED	BIT(2)

#define VE_H264_SHS2			0x20c
#define VE_H264_SHS2_NUM_REF_IDX_ACTIVE_OVRD	BIT(12)

#define VE_H264_SHS_WP			0x210

#define VE_H264_SHS_QP			0x21c
#define VE_H264_SHS_QP_SCALING_MATRIX_DEFAULT	BIT(24)

#define VE_H264_CTRL			0x220
#define VE_H264_CTRL_VP8			BIT(29)
#define VE_H264_CTRL_VLD_DATA_REQ_INT		BIT(2)
#define VE_H264_CTRL_DECODE_ERR_INT		BIT(1)
#define VE_H264_CTRL_SLICE_DECODE_INT		BIT(0)

#define VE_H264_CTRL_INT_MASK		(VE_H264_CTRL_VLD_DATA_REQ_INT | \
					 VE_H264_CTRL_DECODE_ERR_INT | \
					 VE_H264_CTRL_SLICE_DECODE_INT)

#define VE_H264_TRIGGER_TYPE		0x224
#define VE_H264_TRIGGER_TYPE_PROBABILITY(x)	SHIFT_AND_MASK_BITS(x, 31, 24)
#define VE_H264_TRIGGER_TYPE_BIN_LENS(x)	SHIFT_AND_MASK_BITS((x) - 1, 18, 16)
#define VE_H264_TRIGGER_TYPE_N_BITS(x)		(((x) & 0x3f) << 8)
#define VE_H264_TRIGGER_TYPE_VP8_GET_BITS	(15 << 0)
#define VE_H264_TRIGGER_TYPE_VP8_UPDATE_COEF	(14 << 0)
#define VE_H264_TRIGGER_TYPE_VP8_SLICE_DECODE	(10 << 0)
#define VE_H264_TRIGGER_TYPE_AVC_SLICE_DECODE	(8 << 0)
#define VE_H264_TRIGGER_TYPE_INIT_SWDEC		(7 << 0)
#define VE_H264_TRIGGER_TYPE_FLUSH_BITS		(3 << 0)

#define VE_H264_STATUS			0x228
#define VE_H264_STATUS_VLD_DATA_REQ_INT		VE_H264_CTRL_VLD_DATA_REQ_INT
#define VE_H264_STATUS_DECODE_ERR_INT		VE_H264_CTRL_DECODE_ERR_INT
#define VE_H264_STATUS_SLICE_DECODE_INT		VE_H264_CTRL_SLICE_DECODE_INT
#define VE_H264_STATUS_VLD_BUSY			BIT(8)
#define VE_H264_STATUS_VP8_UPPROB_BUSY		BIT(17)

#define VE_H264_STATUS_INT_MASK			VE_H264_CTRL_INT_MASK

#define VE_H264_CUR_MB_NUM		0x22c

#define VE_H264_VLD_ADDR		0x230
#define VE_H264_VLD_ADDR_FIRST			BIT(30)
#define VE_H264_VLD_ADDR_LAST			BIT(29)
#define VE_H264_VLD_ADDR_VALID			BIT(28)
#define VE_H264_VLD_ADDR_VAL(x)			(((x) & 0x0ffffff0) | ((x) >> 28))

#define VE_H264_VLD_OFFSET		0x234
#define VE_H264_VLD_LEN			0x238
#define VE_H264_VLD_END			0x23c
#define VE_H264_SDROT_CTRL		0x240
#define VE_H264_OUTPUT_FRAME_IDX	0x24c
#define VE_H264_EXTRA_BUFFER1		0x250
#define VE_H264_EXTRA_BUFFER2		0x254
#define VE_H264_MB_ADDR			0x260
#define VE_H264_ERROR_CASE		0x2b8
#define VE_H264_BASIC_BITS		0x2dc
#define VE_AVC_SRAM_PORT_OFFSET		0x2e0
#define VE_AVC_SRAM_PORT_DATA		0x2e4

#define VE_VP8_PPS			0x214
#define VE_VP8_PPS_PIC_TYPE_P_FRAME		BIT(31)
#define VE_VP8_PPS_LAST_SHARPNESS_LEVEL(v)	SHIFT_AND_MASK_BITS(v, 30, 28)
#define VE_VP8_PPS_LAST_PIC_TYPE_P_FRAME	BIT(27)
#define VE_VP8_PPS_ALTREF_SIGN_BIAS		BIT(26)
#define VE_VP8_PPS_GOLDEN_SIGN_BIAS		BIT(25)
#define VE_VP8_PPS_RELOAD_ENTROPY_PROBS		BIT(24)
#define VE_VP8_PPS_REFRESH_ENTROPY_PROBS	BIT(23)
#define VE_VP8_PPS_MB_NO_COEFF_SKIP		BIT(22)
#define VE_VP8_PPS_TOKEN_PARTITION(v)		SHIFT_AND_MASK_BITS(v, 21, 20)
#define VE_VP8_PPS_MODE_REF_LF_DELTA_UPDATE	BIT(19)
#define VE_VP8_PPS_MODE_REF_LF_DELTA_ENABLE	BIT(18)
#define VE_VP8_PPS_LOOP_FILTER_LEVEL(v)		SHIFT_AND_MASK_BITS(v, 17, 12)
#define VE_VP8_PPS_LOOP_FILTER_SIMPLE		BIT(11)
#define VE_VP8_PPS_SHARPNESS_LEVEL(v)		SHIFT_AND_MASK_BITS(v, 10, 8)
#define VE_VP8_PPS_LAST_LOOP_FILTER_SIMPLE	BIT(7)
#define VE_VP8_PPS_SEGMENTATION_ENABLE		BIT(6)
#define VE_VP8_PPS_MB_SEGMENT_ABS_DELTA		BIT(5)
#define VE_VP8_PPS_UPDATE_MB_SEGMENTATION_MAP	BIT(4)
#define VE_VP8_PPS_FULL_PIXEL			BIT(3)
#define VE_VP8_PPS_BILINEAR_MC_FILTER		BIT(2)
#define VE_VP8_PPS_FILTER_TYPE_SIMPLE		BIT(1)
#define VE_VP8_PPS_LPF_DISABLE			BIT(0)

#define VE_VP8_QP_INDEX_DELTA		0x218
#define VE_VP8_QP_INDEX_DELTA_UVAC(v)		SHIFT_AND_MASK_BITS(v, 31, 27)
#define VE_VP8_QP_INDEX_DELTA_UVDC(v)		SHIFT_AND_MASK_BITS(v, 26, 22)
#define VE_VP8_QP_INDEX_DELTA_Y2AC(v)		SHIFT_AND_MASK_BITS(v, 21, 17)
#define VE_VP8_QP_INDEX_DELTA_Y2DC(v)		SHIFT_AND_MASK_BITS(v, 16, 12)
#define VE_VP8_QP_INDEX_DELTA_Y1DC(v)		SHIFT_AND_MASK_BITS(v, 11, 7)
#define VE_VP8_QP_INDEX_DELTA_BASE_QINDEX(v)	SHIFT_AND_MASK_BITS(v, 6, 0)

#define VE_VP8_PART_SIZE_OFFSET		0x21c
#define VE_VP8_ENTROPY_PROBS_ADDR	0x250
#define VE_VP8_FIRST_DATA_PART_LEN	0x254

#define VE_VP8_FSIZE			0x258
#define VE_VP8_FSIZE_WIDTH(w) \
	SHIFT_AND_MASK_BITS(DIV_ROUND_UP(w, 16), 15, 8)
#define VE_VP8_FSIZE_HEIGHT(h) \
	SHIFT_AND_MASK_BITS(DIV_ROUND_UP(h, 16), 7, 0)

#define VE_VP8_PICSIZE			0x25c
#define VE_VP8_PICSIZE_WIDTH(w)			SHIFT_AND_MASK_BITS(w, 27, 16)
#define VE_VP8_PICSIZE_HEIGHT(h)		SHIFT_AND_MASK_BITS(h, 11, 0)

#define VE_VP8_REC_LUMA			0x2ac
#define VE_VP8_FWD_LUMA			0x2b0
#define VE_VP8_BWD_LUMA			0x2b4
#define VE_VP8_REC_CHROMA		0x2d0
#define VE_VP8_FWD_CHROMA		0x2d4
#define VE_VP8_BWD_CHROMA		0x2d8
#define VE_VP8_ALT_LUMA			0x2e8
#define VE_VP8_ALT_CHROMA		0x2ec

#define VE_VP8_SEGMENT_FEAT_MB_LV0	0x2f0
#define VE_VP8_SEGMENT_FEAT_MB_LV1	0x2f4

#define VE_VP8_SEGMENT3(v)			SHIFT_AND_MASK_BITS(v, 31, 24)
#define VE_VP8_SEGMENT2(v)			SHIFT_AND_MASK_BITS(v, 23, 16)
#define VE_VP8_SEGMENT1(v)			SHIFT_AND_MASK_BITS(v, 15, 8)
#define VE_VP8_SEGMENT0(v)			SHIFT_AND_MASK_BITS(v, 7, 0)

#define VE_VP8_REF_LF_DELTA		0x2f8
#define VE_VP8_MODE_LF_DELTA		0x2fc

#define VE_VP8_LF_DELTA3(v)			SHIFT_AND_MASK_BITS(v, 30, 24)
#define VE_VP8_LF_DELTA2(v)			SHIFT_AND_MASK_BITS(v, 22, 16)
#define VE_VP8_LF_DELTA1(v)			SHIFT_AND_MASK_BITS(v, 14, 8)
#define VE_VP8_LF_DELTA0(v)			SHIFT_AND_MASK_BITS(v, 6, 0)

#define VE_ISP_INPUT_SIZE		0xa00
#define VE_ISP_INPUT_STRIDE		0xa04
#define VE_ISP_CTRL			0xa08
#define VE_ISP_INPUT_LUMA		0xa78
#define VE_ISP_INPUT_CHROMA		0xa7c

#define VE_AVC_PARAM			0xb04
#define VE_AVC_QP			0xb08
#define VE_AVC_MOTION_EST		0xb10
#define VE_AVC_CTRL			0xb14
#define VE_AVC_TRIGGER			0xb18
#define VE_AVC_STATUS			0xb1c
#define VE_AVC_BASIC_BITS		0xb20
#define VE_AVC_UNK_BUF			0xb60
#define VE_AVC_VLE_ADDR			0xb80
#define VE_AVC_VLE_END			0xb84
#define VE_AVC_VLE_OFFSET		0xb88
#define VE_AVC_VLE_MAX			0xb8c
#define VE_AVC_VLE_LENGTH		0xb90
#define VE_AVC_REF_LUMA			0xba0
#define VE_AVC_REF_CHROMA		0xba4
#define VE_AVC_REC_LUMA			0xbb0
#define VE_AVC_REC_CHROMA		0xbb4
#define VE_AVC_REF_SLUMA		0xbb8
#define VE_AVC_REC_SLUMA		0xbbc
#define VE_AVC_MB_INFO			0xbc0

#endif
