/*
 *	Definitions for emulating the minix OS
 */
 
#define MINIX_CS_OFFSET		0
#define MINIX_DS_OFFSET		0		/* For split I/D */

/* We have to convert from 100/ths to 60/ths 8) */
#define MINIX_HZ		60
#define HZ			100

#define MINIX_SIG_IGN		(-1)
#define MINIX_SIG_DFL		0

#define WRITE_USPACE		0
#define READ_USPACE		1


enum
{
	EXIT=1,		/* Complete */
	FORK=2,		/* Complete */
	READ=3,		/* Need to emulate directory support */
	WRITE=4,	/* Complete */
	OPEN=5,		/* Complete */
	CLOSE=6,	/* Complete */
	WAIT=7,		/* Believed complete */
	CREAT=8,	/* Complete */
	LINK=9,		/* Complete */
	UNLINK=10,	/* Complete */
	WAITPID=11,	/* Unimplemented */
	CHDIR=12,	/* Complete */
	TIME=13,	/* Complete */
	MKNOD=14,	/* Complete */
	CHMOD=15,	/* Complete */
	CHOWN=16,	/* Complete */
	BRK=17,		/* Complete */
	STAT=18,	/* Complete (Squasher may not yet be right) */
	LSEEK=19,	/* Need to emulate directory support */
	GETPID=20,	/* Complete */
	MOUNT=21,	/* Not useful */
	UMOUNT=22,	/* Not useful */
	SETUID=23,	/* Complete */
	GETUID=24,	/* Complete */
	STIME=25,	/* Semi done */
	PTRACE=26,	/* Unimplemented */
	ALARM=27,	/* Complete (Note Minix sleep() has a bug)  */
	FSTAT=28,	/* Complete (Squasher may not yet be right) */
	PAUSE=29,	/* Complete */
	UTIME=30,	/* Partial */
	STTY=31,	/* Unimplemented */
	GTTY=32,	/* Unimplemented */
	ACCESS=33,	/* Complete */
	NICE=34,	/* Unimplemented dropped from Minix */
	FTIME=35,	/* Unimplemented dropped from Minix */
	SYNC=36,	/* Complete */
	KILL=37,	/* Complete */
	RENAME=38,	/* Complete */
	MKDIR=39,	/* Complete */
	RMDIR=40,	/* Complete */
	DUP=41,		/* Believed OK */
	PIPE=42,	/* Complete */
	TIMES=43,	/* Partial */
	PROF=44,	/* Unimplemented */
	SETGID=46,	/* Complete */
	GETGID=47,	/* Complete */
	SIGNAL=48,	/* Unimplemented */
	ACCT=51,	/* Complete - but dropped from minix */
	PLOCK=53,	/* Unimplemented */
	IOCTL=54,	/* Sketched out only */
	FCNTL=55,	/* Complete except for locking */
	EXEC=59,	/* Needs work */
	UMASK=60,	/* Complete */
	CHROOT=61,	/* Complete */
	SIGACTION=71,	/* Unimplemented */
	SIGSUSPEND=72,	/* Unimplemented */
	SIGPENDING=73,	/* Unimplemented */
	SIGPROCMASK=74,	/* Unimplemented */
	SIGRETURN=75,	/* Unimplemented */
	REBOOT=76	/* Not useful */
} SysCall;


/* V6 style stty structure */

struct minix_sgttyb
{
	char sg_ispeed;
	char sg_ospeed;
	char sg_erase;
	char sg_kill;
	unsigned short sg_flag;
#define MINIX_COOKED	0x0	
#define MINIX_CBREAK	0x2
#define MINIX_ECHO	0x8
#define MINIX_CRMOD	0x10
#define MINIX_RAW	0x20
#define MINIX_XTAB	0x180
};

/*
 *	V7 tchars structure
 */
 
struct minix_tchars
{
	char t_intrc;
	char t_quitc;
	char t_startc;
	char t_stopc;
	char t_eofc;
	char t_brkc;
};

/*
 *	Minix view of stat(). We have to squash a bit here and give
 *	wrong values with inode >65535 etc
 */
 
struct minix_stat
{
	unsigned short st_dev;
	unsigned short st_inode;
	unsigned short st_mode;
	unsigned short st_nlink;
	unsigned short st_uid;
	unsigned short st_gid;
	unsigned short st_rdev;
	int st_size;
	int st_atime;
	int st_mtime;
	int st_ctime;
};


/*
 *	Minix ioctl list
 */

#define 	MINIX_TIOCGETP	(('t'<<8)|8)
#define		MINIX_TIOCSETP	(('t'<<8)|9)
#define		MINIX_TIOCGETC	(('t'<<8)|18)
#define		MINIX_TIOCSETC	(('t'<<8)|17)
#define 	MINIX_TIOCFLUSH	(('t'<<8)|16)

/*
 *	Minix fcntl list
 */
 
#define MINIX_F_DUPFD	0
#define MINIX_F_GETFD	1
#define MINIX_F_SETFD	2
#define MINIX_F_GETFL	3
#define MINIX_F_SETFL	4
#define MINIX_F_GETLK	5
#define MINIX_F_SETLK	6
#define MINIX_F_SETLKW	7

/*
 *	Minix binary formats
 */
 
#define EXEC_HEADER_SIZE	32

struct minix_exec_hdr
{
	unsigned long type;
#define MINIX_COMBID	0x04100301L
#define MINIX_SPLITID	0x04200301L	
	unsigned long hlen;
	unsigned long tseg;
	unsigned long dseg;
	unsigned long bseg;
	unsigned long unused;
	unsigned long chmem;
	unsigned long unused2; 
};

#define PARAGRAPH(x)	(((unsigned long)(x))>>4)

struct minix_msg_1
{
	short m1i1 __attribute((packed));
	short m1i2 __attribute((packed));
	short m1i3 __attribute((packed));
	unsigned short m1p1 __attribute((packed));
	unsigned short m1p2 __attribute((packed));
	unsigned short m1p3 __attribute((packed));
}; 

struct minix_msg_2
{
	short m2i1 __attribute((packed));
	short m2i2 __attribute((packed));
	short m2i3 __attribute((packed));
	long m2l1 __attribute((packed));
	long m2l2 __attribute((packed));
	unsigned short m2p1 __attribute((packed));
};

struct minix_msg_3
{
	short m3i1 __attribute((packed));
	short m3i2 __attribute((packed));
	unsigned short m3p1 __attribute((packed));
	char m3p2[14] __attribute((packed));
};

struct minix_msg_4
{
	long m4l1 __attribute((packed));
	long m4l2 __attribute((packed));
	long m4l3 __attribute((packed));
	long m4l4 __attribute((packed));
	long m4l5 __attribute((packed));
};

struct minix_msg_5
{
	unsigned char m5c1 __attribute((packed));
	unsigned char m5c2 __attribute((packed));
	short m5i1 __attribute((packed));
	short m5i2 __attribute((packed));
	long m5l1 __attribute((packed));
	long m5l2 __attribute((packed));
	long m5l3 __attribute((packed));
};

struct minix_msg_6
{
	short m6i1 __attribute((packed));
	short m6i2 __attribute((packed));
	short m6i3 __attribute((packed));
	long m6l1 __attribute((packed));
	unsigned short m6f1 __attribute((packed));
};

typedef struct
{
	union
	{
		struct minix_msg_1 m1;
		struct minix_msg_2 m2;
		struct minix_msg_3 m3;
		struct minix_msg_4 m4;
		struct minix_msg_5 m5;
		struct minix_msg_6 m6;
	} msg __attribute((packed));
} minix_msg;

#define MINIX_DSEG(x)		((unsigned char *)(((x)&0xFFFF)+(minix_cpu.regs.ds<<4)))

extern volatile struct vm86_struct minix_cpu;
