// $Id: gweMysqlStatement.java,v 1.5 1997/11/19 18:15:46 xzhu Exp $
//////////////////////////////////////////////////////////////////
// mysql JDBC Driver
//
// Version : 0.9.2
//
// JDBC : 1.22
//
// LastModified: 12.03.1997
//
// Copyright (c) 1997 GWE Technologies Ltd. All rights reserved
// See http://gwe.co.uk/mysql/jdbc/copyright.html for full details

package gwe.sql;

import java.sql.*;
import gwe.sql.gweMysql;

/**
 * This class is the implementation of the JDBC draft specification
 * Statement interface.  Statement objects are used for executing SQL
 * statements.
 */
public class gweMysqlStatement implements java.sql.Statement {
  /**
   * A connection object to direct information to.
   */
    private gweMysqlConnection connection;
  /**
   * A result set returned from this query or whatever.
   */
  private gweMysqlResultSet result;
  /**
   * The max field size for mysql
   * NOTE: This is the value of MAXBUFFER in gweMysql.packet
   */
  private int max_field_size = 65535;
  /**
   * The max rows supported by mysql - big!
   */
  private int max_rows = 50000000; 	//Because it's in the FAQ!
  /**
   * The number of seconds the driver will allow for a SQL statement to
   * execute before giving up.  The default is to wait forever (0).
   */
  private int timeout = 0;
  private SQLWarning warnings = null;	// The warningss chain.  

  /**
   * Constructs a new gweMysqlStatement object.
   * @param conn the gweMysqlConnection object
   */
  public gweMysqlStatement(gweMysqlConnection conn) {
    connection = conn;
  }
  
  /**
   * JDBC draft specification method to execute the specified SQL
   * query and gives back a result set.
   * @see java.sql.Statement#executeQuery
   * @exception SQLException raised for any SQLException
   * @param sql the SQL statement string
   * @return the result set from the query
   */
  public ResultSet executeQuery(String sql)
       throws SQLException 
  {
    this.execute(sql);
    while (result != null && !result.reallyResultSet())
      result = result.getNext();
    if (result == null)
      throw new SQLException("no results returned");
    return result;
  }

  /**
   * JDBC 0.54 specification method for executing simple UPDATE, INSERT,
   * DELETE, etc. statements which do not return ResultSet's.
   * @see java.sql.Statement#executeUpdate
   * @exception java.sql.SQLException thrown when an error occurs executing
   * the SQL
   * @return 0
   */
  public int executeUpdate(String sql) throws SQLException 
  {
    this.execute(sql);
    if (result.reallyResultSet())
      throw new SQLException("Expecting update count");
    return getUpdateCount();
  }
  
  /**
   * JDBC draft specification method to execute an SQL statement and return
   * true if a result set was gotten, false otherwise.  
   * @see java.sql.Statement#execute
   * @exception SQLException raised for any SQLException encountered
   * @param sql the SQL to be executed
   * @return true if there is a result set available
   */
  public boolean execute(String sql) throws SQLException 
  {
    gweMysql mysql = connection.executeQuery(sql);
    result = new gweMysqlResultSet(mysql);
    if (mysql.getUpdateCount() > 0 && mysql.getUpdateID() > 0)
      result.append(new gweMysqlResultSet(mysql, mysql.getUpdateID(), -1));
    return (result != null && result.reallyResultSet());
  }

  /**
   * JDBC draft specification method to close off any open result sets
   * for this Statement.  
   * @see java.sql.Statement#close
   */
  public void close() throws SQLException 
  {
    try
    {
      connection.closeResultSet();
    } catch (SQLException e)
    { throw new SQLException(e.getMessage()); }
  }
  
  /**
   * JDBC 0.54 specification method for obtaining a result set from a
   * statement object.
   * @see java.sql.Statement#getResultSet
   * @return null if no result set is available, otherwise a result set
   */
  public ResultSet getResultSet() throws SQLException {
    ResultSet r;

    r = result;    
    result = null; 
    return r;
  }

  /**
   * JDBC draft specification method for getting the max field size
   * @see java.sql.Statement#getMaxFieldSize
   * @return the value of max field size
   */
  public int getMaxFieldSize() throws SQLException {
    return max_field_size;
  }

  /**
   * JDBC draft specification method for setting the max field size.
   * @see java.sql.Statement#setMaxFieldSize
   * @param max the maximum field size
   */
  public void setMaxFieldSize(int max) throws SQLException {
    max_field_size = max;
  }

  /**
   * JDBC draft specification method for getting the max rows supported
   * @see java.sql.Statement#getMaxRows
   * @return the maximum rows supported by mysql
   */
  public int getMaxRows() throws SQLException {
    return max_rows;
  }

  /**
   * JDBC draft specification method for setting the max rows.
   * @see java.sql.Statement.setMaxRows
   * @param max the max rows
   */
  public void setMaxRows(int max) throws SQLException {
    max_rows = max;
  }

  /**
   * JDBC draft specification method for escape processing.
   * @see java.sql.Statement#setEscapeProcessing
   * @param enable this does nothing right now
   */
  public void setEscapeProcessing(boolean enable)
       throws SQLException {
    throw new SQLException("mysql does not support escape processing.");
  }

  /**
   * JDBC draft specification method for getting the query timeout, which
   * is the number of seconds the driver waits before giving up on the
   * query.
   * @see java.sql.Statement#getQueryTimeout
   * @see setQueryTimeout
   * @return the timeout value for this statement
   */
  public int getQueryTimeout() throws SQLException {
    return timeout;
  }

  /**
   * JDBC draft specification method for setting the query timeout.
   * @see java.sql.Statement#setQueryTimeout
   * @see getQueryTimeout
   * @param x the new query timeout value
   */
  public void setQueryTimeout(int x) throws SQLException {
    timeout = x;
  }

  /**
   * JDBC draft specification method for allowing one thread to cancel this
   * statement which may be running in another thread. 
   * @see java.sql.Statement#cancel
   */
  public void cancel() {
  }
  
  /**
   * JDBC draft specification for getting the chain of warnings for this
   * statement.
   * @see java.sql.Statement#getWarnings
   * @return the chain of warnings
   */
  public final SQLWarning getWarnings() throws SQLException {
    return warnings;
  }

  /**
   * JDBC draft specification for clearing the warning chain.
   * @see java.sql.Statement#clearWarnings
   */
  public void clearWarnings() throws SQLException 
  {
    warnings = null;
  }

  /**
   * JDBC 0.65 specification for setting cursor names.
   * mysql does not support cursors
   */
  public void setCursorName(String unused) throws SQLException {
    throw new SQLException("mysql does not support cursors.");
  }

  /**
   * getUpdateCount returns the current result as an update count,
   * if the result is a ResultSet or there are no more results, -1
   * is returned.  It should only be called once per result.
   *
   * @return the current result as an update count.
   * @exception SQLException if a database access error occurs
   */
  public int 
  getUpdateCount() throws SQLException
  {
    if (result == null)	
      return -1;
    if (result.reallyResultSet())
      return -1;

    setValueInWarnTop("query", ""+result.getUpdateID());
    return result.getResultCount();
  }

  /**
   * getMoreResults moves to a Statement's next result.  If it returns
   * true, this result is a ResulSet.
   *
   * @return true if the next ResultSet is valid
   * @exception SQLException if a database access error occurs
   */
  public boolean 
  getMoreResults() throws SQLException
  {
    result = result.getNext();
    return (result != null && result.reallyResultSet());
  }

  private void
  setValueInWarnTop(String command, String str)
  {
    SQLWarning nw = new SQLWarning("Command="+ command + ": " + str);
    if (warnings != null && warnings.getMessage().startsWith("Command="+command+": "))
    {
      SQLWarning w = warnings.getNextWarning();
      warnings.setNextWarning(null);
      warnings = w;
    }
    if (warnings != null)
      nw.setNextWarning(warnings);
    warnings = nw;
  }
}
