#include "QScanner.h"

/*!
  Constructeur. Prend en paramtre le nom du fichier. Il faut faire
  appel  isValid() pour savoir si le fichier a t correctement ouvert.
  Cette fonction initialise les deux chaines contenant les sparateurs
  et les caractres spciaux.
 */
QScanner::QScanner(const char *filename)
  {
  haveUngotToken = false;
  valid = false;
  line = 1;
  in.open(filename, ios::in);
  if(in)
    valid = true;
  setCommentStyle(NO_STYLE);
  setSpecialChars("");
  setSeparators(" \n\t\r");
  }


/*!
  Destructeur. Ferme le fichier si besoin est.
 */
QScanner::~QScanner(void)
  {
  close();
  }


/*!
  Extrait un token du fichier texte. Si le fichier n'est pas ouvert,
  ou si une erreur se produit, retourne "".
 */
QString QScanner::getToken(void)
  {
  // Teste si le fichier est valide
  if(!isValid())
    return "";
  // Teste si on a un token en reserve.
  // Si oui on le retourne
  if(haveUngotToken)
    return getUngotToken();
  // Situation normale
  QString res = "";
  char c = 0;
  // Passer les sparateurs avant le tocken
  while(!in.eof() && isSeparator((c=get())));
  // Est-ce un caratre spcial? Si oui le retourner
  if(isSpecialChar(c))
    {
    res += c;
    return res;
    }
  // Boucle tant que pas fin de fichier ou sparateur
  while(!in.eof() && !isSeparator(c))
    {
    res += c;
    c = get();
    // Si caratre spcial, on arrte l
    if(isSpecialChar(c))
      {
      unget(c);
      break;
      }
    }
  return res;
  }


/*!
  Permet d'annuler l'extraction d'un token. Celui-ci sera retourn
  lors du prochain appel  getToken(). Attention: un seul token peut ainsi
  etre remis en attente (pour l'instant). Elle retourne \c false si il y a
  dj un token en attente. Sinon retourne \c true.
 */
bool QScanner::ungetToken(QString token)
  {
  if(haveUngotToken)
    return false;
  ungotToken = token;
  haveUngotToken = true;
  return true;
  }

/*!
  Permet de savoir si le flux est valide, c'est  dire si le fichier
  est bien ouvert et si on n'a pas atteint la fin du fichier.
 */
bool QScanner::isValid(void)
  {
  return valid;
  }


/*!
  Permet de savoir si on a atteint la fin du fichier d'entre.
  Retourne \c true si c'est le cas, et \c false sinon.
 */
bool QScanner::eof(void)
  {
  return in.eof();
  }

/*!
  Retourne le numro de la ligne courant dans le fichier d'entre.
 */
int QScanner::currentLine(void)
  {
  return line;
  }

/*!
  Renvoie \c true si le caractre pass en paramtre est un caractre
  sparateur. Elle utilise pour cela la chaine de caractre separators()
  qui doit contenir tous les caractres sprateurs. Ex: " \n\t".
 */
bool QScanner::isSeparator(char c)
  {
  return member(c, separators);
  }

/*!
  Renvoie \c true si le caractre pass en paramtre est un caractre
  spcial. Elle utilise pour cela la chaine de caractre specialChars()
  qui doit contenir tous les caractres spciaux, c'est  dire les
  caractres qui constituent un token  eux-seuls. Ex: ".;()[]{}".
 */
bool QScanner::isSpecialChar(char c)
  {
  return member(c, specialChars);
  }

/*!
  Renvoie \c true si le caractre \e c appartient  la chaine \e str().
  Sinon retourne \c false.
 */
bool QScanner::member(char c, const char *str)
  {
  while(*str)
    if(c==*(str++))
      return true;
  return false;
  }

/*!
  Ferme le fichier seulement si il est ouvert.
 */
void QScanner::close(void)
  {
  if(valid)
    {
    valid = false;
    in.close();
    }
  }

/*!
  Permet d'obtenir le token qui a t remis avec ungetToken(). Si il n'y
  a rien dans le buffer correspondant, retourne "". Utilise par getToken().
 */
QString QScanner::getUngotToken(void)
  {
  if(!haveUngotToken)  // ne devrait pas arriver!
    return "";
  haveUngotToken = false;
  return ungotToken;
  }


/*!
  Lit un caractre dans le flux d'entre et le retourne. Cette fonction met 
  jour les informations sur l'tat courant du flux.
 */
char QScanner::get(void)
  {
  char c = in.get();
  if(c=='\n')
    {
    line++;
    return c;
    }
  else if(isCommentBegin(c))
    {
    if(readComment())
      return get();
    else
      return c;
    }
  else
    return c;
  }


/*!
  Remet dans le flux d'entre le dernier caractre lu. Met  jour l'tat
  courant du flux d'entre. Elle prend en parmtre le caractre  remettre, mais
  il est seulement utilis pour mettre  jour l'tat courant (numro de ligne).
  Attention: c'est effectivement le dernier caractre lu qui est remis dans le
  flux d'entre et non le caractre pass en paramtre. Il faut donc bien passer
   cette fonction le dernier caractre lu avec ifstream::get().
 */
void QScanner::unget(char c)
  {
  if(c=='\n')
    line--;
  in.unget();
  }

/*!
  Dfinit le style des commentaires.
 */
void QScanner::setCommentStyle(int style)
  {
  commentStyle = style;
  }

/*!
  Retourne le style actuel des commentaires.
 */
int QScanner::getCommentStyle(void)
  {
  return commentStyle;
  }

/*!
  Dit si un caractre marque ventuellement le dbut d'un commentaire,
  suivant le style courant.
 */
bool QScanner::isCommentBegin(char c)
  {
  switch(commentStyle)
    {
    case NO_STYLE:
      return false;
    case C_STYLE:
      return (c == '/');
    case CXX_STYLE:
      return (c == '/');
    case PASCAL_STYLE:
      return (c == '{');
    case SHELL_STYLE:
      return (c == '#');
    case LISP_STYLE:
      return (c == ';');
    default:
      return false;
    }
  }

/*!
  Passe un commentare suivant le style. Appelle la fonction adquate.
  Cette fonction doit retourner \c true si elle a effectivement trouv un
  commentaires, et \c false sinon. Certaines retournent toujours \c true
  (celles dont le commentaire est marqu par un seul caractre).
 */
bool QScanner::readComment(void)
  {
  switch(commentStyle)
    {
    case NO_STYLE:
      return false;
    case C_STYLE:
      return readCComment();
    case CXX_STYLE:
      return readCXXComment();
    case PASCAL_STYLE:
      return readPascalComment();
    case SHELL_STYLE:
      return readShellComment();
    case LISP_STYLE:
      return readLispComment();
    default:
      return false;
    }
  }

/*!
  Passe les commentaires de style C. Suppose que l'on a dj lu un '/'.
  Retourne \c true si il s'agissait vraiment d'un commentaire. Sinon,
  retourne \c false.
 */
bool QScanner::readCComment(void)
  {
  char c = in.get();
  if(c == '*') // C'est bien un commentaire
    {
    // On passe jusqu' rencontrer un '*' suivi d'un '/'
    bool fin = false;
    while(!fin && !in.eof())
      {
      c = in.get();
      if(c == '\n')
	line++;
      else if(c == '*')
	if(in.get() == '/')
	  fin = true;
      }
    return true;
    }
  else
    {
    in.unget(); // on remet le caractre que l'on n'aurait pas du lire.
    return false;
    }
  }


/*!
  Passe les commentaires de style C++. Suppose que l'on a dj lu un '/'.
  Retourne \c true si il s'agissait vraiment d'un commentaire. Sinon,
  retourne \c false.
 */
bool QScanner::readCXXComment(void)
  {
  char c = in.get();
  if(c == '/') // Commentaire sur une seule ligne
    {
    while(in.get() != '\n' && !in.eof()); // Passe jusqu' la fin de la ligne.
    line++; // Incrmente le compteur de lignes.
    return true;
    }
  else if(c == '*') // Commentaire sur plusieurs lignes
    {
    // On passe jusqu' rencontrer un '*' suivi d'un '/'
    bool fin = false;
    while(!fin && !in.eof())
      {
      c = in.get();
      if(c == '\n')
	line++;
      else if(c == '*')
	if(in.get() == '/')
	  fin = true;
      }
    return true;
    }
  else
    {
    in.unget(); // on remet le carctre que l'on n'aurait pas du lire.
    return false;
    }
  }


/*!
  Passe un commentaire de type Pascal. Suppose que l'on a dj lu un '{'.
  Retourne toujours \c true puisque si on a lu un '{' on est sur d'avoir
  un commentaire.
 */
bool QScanner::readPascalComment(void)
  {
  readUntil('}');
  return true;
  }

/*!
  Passe un commentaire de type Shell. Suppose que l'on a dj lu un '#'.
 */
bool QScanner::readShellComment(void)
  {
  readUntil('\n');
  return true;
  }

/*!
  Passe un commentaire de type Lisp. Suppose que l'on a dj lu un ';'.
 */
bool QScanner::readLispComment(void)
  {
  readUntil('\n');
  return true;
  }

/*!
  Lit des caractres jusqu' rencontrer un caractre donn. Met  jour
  la ligne courante. Retourne \c false si la fin de fichier est atteinte avant
  d'avoir trouv le caractre.
 */
bool QScanner::readUntil(char but)
  {
  char c;
  do
    {
    c = in.get();
    if(c=='\n')
      line++;
    } while(c != but && !in.eof());
  if(in.eof() && c != but)
    return false;
  else
    return true;
  }

/*!
  Dfinit les caractres sparateurs. Ils doivent se trouver dans la chaine
  donne en paramtre.
 */
void QScanner::setSeparators(char *str)
  {
  separators = str;
  }


/*!
  Dfinit les caractres spciaux. Ils doivent se trouver dans la chaine
  donne en paramtre.
 */
void QScanner::setSpecialChars(char *str)
  {
  specialChars = str;
  }

/*!
  Renvoie tout jusqu'au caractre. Renvoie "" si pas trouv.
 */
QString QScanner::getUntil(char but)
  {
  QString res = "";
  if(in.eof())
    return res;
  char c = get();
  while(!in.eof() && c != but)
    {
    res += c;
    c = get();
    }
  if(in.eof())
    return "";
  if(c == but)
    unget(c);
  return res;
  }
