/**
*** Program jdbcMysqlBase.java
***    in product twz1jdbcForMysql, 
***    Copyright 1997, 1998 by Terrence W. Zellers.
***   
***  All rights explicitly reserved.
***
***  See file "LICENSE" in this package for conditions of use.
**/

/** Locus of system defaults and containers of static classes for
*** various and sundry purposes.
**/

package twz1.jdbc.mysql;

import twz1.jdbc.mysql.jdbcMysqlDebug;
import twz1.jdbc.mysql.jdbcMysqlDriver;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.lang.System;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.sql.*;


public final class jdbcMysqlBase
{

static final int COM_SLEEP              = 0;
static final int COM_QUIT               = 1;  
static final int COM_INIT_DB            = 2;
static final int COM_QUERY              = 3; 
static final int COM_FIELD_LIST         = 4; 
static final int COM_CREATE_DB          = 5;
static final int COM_DROP_DB            = 6;
static final int COM_REFRESH            = 7;
static final int COM_SHUTDOWN           = 8;
static final int COM_STATISTICS         = 9;
static final int COM_PROCESS_INFO       = 10;
static final int COM_CONNECT            = 11;
static final int COM_PROCESS_KILL       = 12;
static final int COM_DEBUG              = 13;

// These are roll your own 
static final int COM_INSERT             = 101;
static final int COM_UPDATE             = 102;
static final int COM_DELETE             = 103;
static final int COM_CREATE_T           = 104;
static final int COM_DROP               = 105;
static final int COM_ALTER              = 106;
static final int COM_LOCK               = 107;
static final int COM_UNLOCK             = 109;
static final int COM_SET                = 110;
static final int COM_GRANT              = 111;
static final int COM_REPLACE            = 112;
static final int COM_COMMENT            = 190;
static final int COM_SELECT             = 201;
static final int COM_DESCRIBE           = 301;
static final int COM_EXPLAIN            = 302;
static final int COM_SHOW               = 304;

static final int NOT_NULL_FLAG          = 1;		
static final int PRI_KEY_FLAG           = 2;		
static final int UNIQUE_KEY_FLAG        = 4;		
static final int MULTIPLE_KEY_FLAG      = 8;		
static final int BLOB_FLAG              = 16;		
static final int UNSIGNED_FLAG          = 32;		
static final int ZEROFILL_FLAG          = 64;		
static final int BINARY_FLAG            = 128;
static final int ENUM_FLAG              = 256;
static final int PART_KEY_FLAG          = 512;	
static final int GROUP_FLAG             = 1024;	

static final int FIELD_TYPE_DECIMAL     = 0;
static final int FIELD_TYPE_TINY        = 1;
static final int FIELD_TYPE_CHAR        = 1;
static final int FIELD_TYPE_SHORT       = 2;
static final int FIELD_TYPE_LONG        = 3;
static final int FIELD_TYPE_FLOAT       = 4;
static final int FIELD_TYPE_DOUBLE      = 5; 
static final int FIELD_TYPE_NULL        = 6;
static final int FIELD_TYPE_TIMESTAMP   = 7;
static final int FIELD_TYPE_LONGLONG    = 8;
static final int FIELD_TYPE_INT24       = 9;
static final int FIELD_TYPE_DATE        = 10;   
static final int FIELD_TYPE_TIME        = 11;
static final int FIELD_TYPE_DATETIME    = 12;
static final int FIELD_TYPE_ENUM        = 247;
static final int FIELD_TYPE_INTERVAL    = 247; 
static final int FIELD_TYPE_SET         = 248;
static final int FIELD_TYPE_TINY_BLOB   = 249;
static final int FIELD_TYPE_MEDIUM_BLOB = 250;
static final int FIELD_TYPE_LONG_BLOB   = 251;
static final int FIELD_TYPE_BLOB        = 252;
static final int FIELD_TYPE_VAR_STRING  = 253;
static final int FIELD_TYPE_STRING      = 254;

final static String[] errs = {
        "E0300 setReadDump()",
        "E0301 setWriteDump()",
        "E0302 setMultipleQuery()",
        "E0303 isMultipleQuery()",
        "E0304 getLastInsertID()",
                             };


/** Reference self */
private static jdbcMysqlBase self;

/** The default values for our driver. */
private static Hashtable mysqlDefaults;

/** Circular reference to the driver. */
private static jdbcMysqlDriver driver;  

/** properties depth. */
private static int pdepth;

/** The debug object */
private static jdbcMysqlDebug debug;

/** Commands Hashtable */
static Hashtable cmds;

/** Discrete ojbect numbers . */
static int xoid;

/** Am I an applet? */
boolean applet;

/** Constructor for the defaults */
jdbcMysqlBase(jdbcMysqlDriver d)  
    {
    this.self = this;
    this.driver = d;
    this.mysqlDefaults = new Hashtable();
    this.debug = null;
    this.xoid  = 0;

    boolean fa = true;
    String uname = null;
    String ps = null;
    String hd = null;
    String pfile = ".jdbcMysql.properties";

    try { 
        Properties p = new Properties(System.getProperties()); 
        uname = p.getProperty("user.name");
        ps = p.getProperty("file.separator");
        hd = p.getProperty("user.home");
        }
    catch(SecurityException se){fa = false;}

    mysqlDefaults.put("host", "localhost");
    mysqlDefaults.put("port", "3306");
    mysqlDefaults.put("db", "mysql");
    if(uname != null)mysqlDefaults.put("user", uname);
    mysqlDefaults.put("version", "1.0.0");
    mysqlDefaults.put("majorVersion", "1");
    mysqlDefaults.put("minorVersion", "0");
    mysqlDefaults.put("product", "twz1jdbcForMysql");
    mysqlDefaults.put("Copyright", "1997, 1998 by Terrence W. Zellers");

    if(fa)
      if(pfile != null)
        if(!append(pfile))
          if(hd != null && ps != null) 
            append(hd + ps + pfile);

    String dn = getDefault("debugFile");
    if(dn != null) debug = new jdbcMysqlDebug(dn);

    this.cmds = new Hashtable();
    cmds.put("insert",   new Integer(COM_INSERT));
    cmds.put("replace",  new Integer(COM_REPLACE));
    cmds.put("update",   new Integer(COM_UPDATE));
    cmds.put("create",   new Integer(COM_CREATE_T));
    cmds.put("delete",   new Integer(COM_DELETE));
    cmds.put("alter",    new Integer(COM_ALTER));
    cmds.put("drop",     new Integer(COM_DROP));
    cmds.put("lock",     new Integer(COM_LOCK));
    cmds.put("unlock",   new Integer(COM_UNLOCK));
    cmds.put("set",      new Integer(COM_SET));
    cmds.put("grant",    new Integer(COM_GRANT));
    cmds.put("select",   new Integer(COM_SELECT));
    cmds.put("describe", new Integer(COM_DESCRIBE));
    cmds.put("explain",  new Integer(COM_EXPLAIN));
    cmds.put("show",     new Integer(COM_SHOW));
    }


/** Get an object id */
static int getOID() { return ++xoid;}

/** Use properties files potentially nested to a depth of 16 by
*** means of setting "moreProperties" in each.
*** @param fid The file name of the properties file.
*** @return true if no error, else false.
**/

private boolean append(String fid)
    {
    String name;
    String ffname;
    String pfname; 
    String val;
    String err = null;
    try { 
        File ff = new File(fid);
        if(ff.isFile() && ff.canRead())
            {
            ffname = ff.getCanonicalPath();
            Properties p = new Properties();
            InputStream is = new FileInputStream(ff);
            p.load(is);
            mysqlDefaults.remove("moreProperties");
            pdepth++;
            Enumeration en = p.propertyNames();
            while(en.hasMoreElements())
                {
                name = (String) en.nextElement();
                val  = p.getProperty(name);
                mysqlDefaults.put(name, val);
                }
            pfname = new String("PropertyFile" + pdepth);
            mysqlDefaults.put(pfname, ffname);
            if(pdepth < 17)
                {
                val = (String) mysqlDefaults.get("moreProperties");
                if(val != null) append(val);  
                }
            else err = "Properties files depth limit.";
            return true;
            }
        else err = new String(fid + " is not a readable file.");
        }
    catch(Exception e){ err = e.toString();  }

    if(err != null) mysqlDefaults.put("Properties.error", err);
    return false;
    }

/** Return a reference to myself */
static jdbcMysqlBase base() { return self; }

/** Return a reference to debug. */
static jdbcMysqlDebug getDebug() { return debug; }


/** Return the reference to our driver. */
static jdbcMysqlDriver getDriver() { return driver;  } 

/** Export a copy of the defaults hashtable. */

static Hashtable copyDefaults()
    {
    Hashtable ht = new Hashtable();
    Enumeration en = mysqlDefaults.keys();
    String key;
    while(en.hasMoreElements())
       {
       key = (String) en.nextElement();
       ht.put(key, (String) mysqlDefaults.get(key));
       }
    return ht;
    }


/*-------------------------------------------------------------------+
|              Get the numeric value of a string command             |
+-------------------------------------------------------------------*/

/** get a command value matching a string. */
static int getCommand(String s)
    {
    String ls = s.toLowerCase();
    Integer ret = (Integer) cmds.get(ls);
    if(ret == null)
        {
        if(ls.charAt(0) == '#') return COM_COMMENT;
        if(ls.substring(0, 2).equals("/*")) return COM_COMMENT;
        return -1;
        }
    int r = ret.intValue();
    return r;
    }

/*===================================================================+
||                Non API public methods for MySQL                  ||
+===================================================================*/

/** Return an array of the names in the Defaults Hashtable. */
public static String[] getDefaultNames()
    {
    int size = mysqlDefaults.size();
    String table[] = new String[size];
    Enumeration en = mysqlDefaults.keys();
    int i = 0;
    while (en.hasMoreElements())
        {
        table[i++] = ((String) en.nextElement());
        }
    return table;
    }

 
/** Return a default value for mMysql.
*** @param  what  The name of the property to return.
*** @return The value of the property as a string.
**/

public static String getDefault(String what)
    {
    return (String) mysqlDefaults.get(what);
    }

public static void setReadDump(jdbcMysqlConnex cx, boolean v)
        throws SQLException
    { 
    try { cx.setReadDump(v); } 
    catch(Exception e) { base().errHandlerM(0, e); }
    }

public static void setWriteDump(jdbcMysqlConnex cx, boolean v)
        throws SQLException
    { 
    try { cx.setWriteDump(v); }
    catch(Exception e) { base().errHandlerM(1, e); }
    }

public static void setMultipleQuery(jdbcMysqlConnex cx, boolean v)
        throws SQLException
    { 
    try { cx.setMultipleQuery(v); }
    catch(Exception e) { base().errHandlerM(2, e); }
    }

public static boolean isMultipleQuery(jdbcMysqlConnex cx)
        throws SQLException 
    { 
    boolean b = true;
    try { b = cx.isMultipleQuery(); } 
    catch(Exception e) { base().errHandlerM(3, e); }
    return b;
    }

/** Get the last_insert_ID MySQL returns from an insert. The result
*** will be -1 if the last update style statement was not an insert;
*** This result could be problematic if you actually insert a -1 in
*** an auto_increment column.
*** @param st The Statement on which the executeUpdate(insert ...)
***        was actually run.
*** @exception SQLException if st is invalid.
**/

public static long getLastInsertID(Statement st)
        throws SQLException
    {
    try { return ((jdbcMysqlStmt)st).lastInsertID; }
    catch(Exception e) { base().errHandlerM(4, e); }
    return -1L;
    }


public static long getRowCount(Statement st) throws SQLException
    {
    jdbcMysqlResult r = ((jdbcMysqlStmt)st).qResult;
    if(r == null) return -1L;
    return (long) r.realRowCount;
    }

/*===================================================================+
||                  Error handlers and debuggery                    ||
+===================================================================*/

private void errHandlerP(int n, Exception e) throws SQLException
    {
    String o = errMessage(n, e);
    errHandlerM(n, e);
    }

private void errHandlerS(int n, String s) throws SQLException
    {
    String o = errs[n] + s;
    debugMessage(o);
    throw new SQLException(o);
    }

private void errHandlerM(int n, Exception e) throws SQLException
    {
    String o = errMessage(n, e);
    debugMessage(o);
    throw new SQLException(o);
    }

private String errMessage(int n, Exception e)
    {
    String i, o;
    i = e.toString();
    o = errs[n] + " - " + i;
    return o;
    }

/*-------------------------------------------------------------------+
|                    Send a message to debug                         |
+-------------------------------------------------------------------*/

private void debugMessage(String s)
    {
    jdbcMysqlDebug DEBUG = debug;
    if(DEBUG == null) return;
    DEBUG.put(s);
    }


/*===================================================================+
||                common static methods for package                 ||
+===================================================================*/

/** Common error message handling with write to debug file if
*** operative.
*** @parm s  The error string thrown as an exception.
**/
static void errMessage(String s) throws SQLException
    {
    base().debugMessage(s);
    throw new SQLException(s);
    }

/** Put an emessage as something quasi intelligible
*** @param e Exception to show.
*** @return string of the message.
**/
static String eMessage(Exception e)
    {
    if(e == null) return "";
    String u = e.getMessage();
    if(u == null) u = "";
    if(u.trim().equals("")) u = "\n    " +  e.toString(); 
    else if(u.charAt(0) != '\n') u =  "\n" + u;
    return u;
    }

/** Common true or false values of a string.
*** @param s The string whose value is returned.
***  @param nv The value returned if the string is null.
*** @param fv The value returned if the string fails to parse.
*** @return 1 if s evaluates true, 0 if false, nv or fv depending.
**/
static int boolValue(String s, int nv, int fv)
    {
    if(s == null) return nv;
    if(s.equalsIgnoreCase("true")) return 1;
    if(s.equalsIgnoreCase("t")) return 1;
    if(s.equalsIgnoreCase("yes")) return 1;
    if(s.equalsIgnoreCase("y")) return 1;
    if(s.equals("1")) return 1;
    if(s.equalsIgnoreCase("false")) return 0;
    if(s.equalsIgnoreCase("f")) return 0;
    if(s.equalsIgnoreCase("no")) return 0;
    if(s.equalsIgnoreCase("n")) return 0;
    if(s.equals("0")) return 0;
    return fv;
    }

/** Common parse of a string into an integer value.
*** @param s  The string of which the value is to be returned.
*** @param nv The value to be returned if the string is null.
*** @param fv The value to be returned if the parse fails.
*** @return the integer value of the string or nv, or fv.
**/
static int intValue(String s, int nv, int fv)
   {
   if(s == null) return nv;
   int r = fv;
   try{ r = Integer.parseInt(s); }
   catch(NumberFormatException e){}
   return r;
   }
}
