/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.tasks;

import com.carrotsearch.hppc.ObjectIntHashMap;
import com.carrotsearch.hppc.ObjectIntMap;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.Assertions;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionResponse;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.lease.Releasables;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.ConcurrentMapLong;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.http.HttpTransportSettings;
import org.elasticsearch.tasks.CancellableTask;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.tasks.TaskAwareRequest;
import org.elasticsearch.tasks.TaskCancellationService;
import org.elasticsearch.tasks.TaskCancelledException;
import org.elasticsearch.tasks.TaskId;
import org.elasticsearch.tasks.TaskResult;
import org.elasticsearch.tasks.TaskResultsService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TcpChannel;

public class TaskManager
implements ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(TaskManager.class);
    private static final TimeValue WAIT_FOR_COMPLETION_POLL = TimeValue.timeValueMillis((long)100L);
    private final List<String> taskHeaders;
    private final ThreadPool threadPool;
    private final ConcurrentMapLong<Task> tasks = ConcurrentCollections.newConcurrentMapLongWithAggressiveConcurrency();
    private final ConcurrentMapLong<CancellableTaskHolder> cancellableTasks = ConcurrentCollections.newConcurrentMapLongWithAggressiveConcurrency();
    private final AtomicLong taskIdGenerator = new AtomicLong();
    private final Map<TaskId, String> banedParents = new ConcurrentHashMap<TaskId, String>();
    private TaskResultsService taskResultsService;
    private volatile DiscoveryNodes lastDiscoveryNodes = DiscoveryNodes.EMPTY_NODES;
    private final ByteSizeValue maxHeaderSize;
    private final Map<TcpChannel, ChannelPendingTaskTracker> channelPendingTaskTrackers = ConcurrentCollections.newConcurrentMap();
    private final SetOnce<TaskCancellationService> cancellationService = new SetOnce();

    public TaskManager(Settings settings, ThreadPool threadPool, Set<String> taskHeaders) {
        this.threadPool = threadPool;
        this.taskHeaders = new ArrayList<String>(taskHeaders);
        this.maxHeaderSize = HttpTransportSettings.SETTING_HTTP_MAX_HEADER_SIZE.get(settings);
    }

    public void setTaskResultsService(TaskResultsService taskResultsService) {
        assert (this.taskResultsService == null);
        this.taskResultsService = taskResultsService;
    }

    public void setTaskCancellationService(TaskCancellationService taskCancellationService) {
        this.cancellationService.set((Object)taskCancellationService);
    }

    public Task register(String type, String action, TaskAwareRequest request) {
        HashMap<String, String> headers = new HashMap<String, String>();
        long headerSize = 0L;
        long maxSize = this.maxHeaderSize.getBytes();
        ThreadContext threadContext = this.threadPool.getThreadContext();
        for (String key : this.taskHeaders) {
            String httpHeader = threadContext.getHeader(key);
            if (httpHeader == null) continue;
            if ((headerSize += (long)(key.length() * 2 + httpHeader.length() * 2)) > maxSize) {
                throw new IllegalArgumentException("Request exceeded the maximum size of task headers " + this.maxHeaderSize);
            }
            headers.put(key, httpHeader);
        }
        Task task = request.createTask(this.taskIdGenerator.incrementAndGet(), type, action, request.getParentTask(), headers);
        Objects.requireNonNull(task);
        assert (task.getParentTaskId().equals(request.getParentTask())) : "Request [ " + request + "] didn't preserve it parentTaskId";
        if (logger.isTraceEnabled()) {
            logger.trace("register {} [{}] [{}] [{}]", (Object)task.getId(), (Object)type, (Object)action, (Object)task.getDescription());
        }
        if (task instanceof CancellableTask) {
            this.registerCancellableTask(task);
        } else {
            Task previousTask = this.tasks.put(task.getId(), task);
            assert (previousTask == null);
        }
        return task;
    }

    private void registerCancellableTask(Task task) {
        String reason;
        CancellableTask cancellableTask = (CancellableTask)task;
        CancellableTaskHolder holder = new CancellableTaskHolder(cancellableTask);
        CancellableTaskHolder oldHolder = this.cancellableTasks.put(task.getId(), holder);
        assert (oldHolder == null);
        if (task.getParentTaskId().isSet() && !this.banedParents.isEmpty() && (reason = this.banedParents.get(task.getParentTaskId())) != null) {
            try {
                holder.cancel(reason);
                throw new TaskCancelledException("Task cancelled before it started: " + reason);
            }
            catch (Throwable throwable) {
                this.unregister(task);
                throw throwable;
            }
        }
    }

    public void cancel(CancellableTask task, String reason, Runnable listener) {
        CancellableTaskHolder holder = this.cancellableTasks.get(task.getId());
        if (holder != null) {
            logger.trace("cancelling task with id {}", (Object)task.getId());
            holder.cancel(reason, listener);
        } else {
            listener.run();
        }
    }

    public Task unregister(Task task) {
        logger.trace("unregister task for id: {}", (Object)task.getId());
        if (task instanceof CancellableTask) {
            CancellableTaskHolder holder = this.cancellableTasks.remove(task.getId());
            if (holder != null) {
                holder.finish();
                return holder.getTask();
            }
            return null;
        }
        return this.tasks.remove(task.getId());
    }

    public Releasable registerChildNode(long taskId, DiscoveryNode node) {
        CancellableTaskHolder holder = this.cancellableTasks.get(taskId);
        if (holder != null) {
            logger.trace("register child node [{}] task [{}]", (Object)node, (Object)taskId);
            holder.registerChildNode(node);
            return Releasables.releaseOnce(() -> {
                logger.trace("unregister child node [{}] task [{}]", (Object)node, (Object)taskId);
                holder.unregisterChildNode(node);
            });
        }
        return () -> {};
    }

    public DiscoveryNode localNode() {
        return this.lastDiscoveryNodes.getLocalNode();
    }

    public <Response extends ActionResponse> void storeResult(Task task, final Exception error, final ActionListener<Response> listener) {
        TaskResult taskResult;
        DiscoveryNode localNode = this.lastDiscoveryNodes.getLocalNode();
        if (localNode == null) {
            listener.onFailure(error);
            return;
        }
        try {
            taskResult = task.result(localNode, error);
        }
        catch (IOException ex) {
            logger.warn(() -> new ParameterizedMessage("couldn't store error {}", (Object)ExceptionsHelper.detailedMessage(error)), (Throwable)ex);
            listener.onFailure(ex);
            return;
        }
        this.taskResultsService.storeResult(taskResult, new ActionListener<Void>(){

            @Override
            public void onResponse(Void aVoid) {
                listener.onFailure(error);
            }

            @Override
            public void onFailure(Exception e) {
                logger.warn(() -> new ParameterizedMessage("couldn't store error {}", (Object)ExceptionsHelper.detailedMessage(error)), (Throwable)e);
                listener.onFailure(e);
            }
        });
    }

    public <Response extends ActionResponse> void storeResult(Task task, final Response response, final ActionListener<Response> listener) {
        TaskResult taskResult;
        DiscoveryNode localNode = this.lastDiscoveryNodes.getLocalNode();
        if (localNode == null) {
            logger.warn("couldn't store response {}, the node didn't join the cluster yet", response);
            listener.onResponse(response);
            return;
        }
        try {
            taskResult = task.result(localNode, response);
        }
        catch (IOException ex) {
            logger.warn(() -> new ParameterizedMessage("couldn't store response {}", (Object)response), (Throwable)ex);
            listener.onFailure(ex);
            return;
        }
        this.taskResultsService.storeResult(taskResult, new ActionListener<Void>(){

            @Override
            public void onResponse(Void aVoid) {
                listener.onResponse(response);
            }

            @Override
            public void onFailure(Exception e) {
                logger.warn(() -> new ParameterizedMessage("couldn't store response {}", (Object)response), (Throwable)e);
                listener.onFailure(e);
            }
        });
    }

    public Map<Long, Task> getTasks() {
        HashMap<Long, CancellableTask> taskHashMap = new HashMap<Long, CancellableTask>(this.tasks);
        for (CancellableTaskHolder holder : this.cancellableTasks.values()) {
            taskHashMap.put(holder.getTask().getId(), holder.getTask());
        }
        return Collections.unmodifiableMap(taskHashMap);
    }

    public Map<Long, CancellableTask> getCancellableTasks() {
        HashMap<Long, CancellableTask> taskHashMap = new HashMap<Long, CancellableTask>();
        for (CancellableTaskHolder holder : this.cancellableTasks.values()) {
            taskHashMap.put(holder.getTask().getId(), holder.getTask());
        }
        return Collections.unmodifiableMap(taskHashMap);
    }

    public Task getTask(long id) {
        Task task = this.tasks.get(id);
        if (task != null) {
            return task;
        }
        return this.getCancellableTask(id);
    }

    public CancellableTask getCancellableTask(long id) {
        CancellableTaskHolder holder = this.cancellableTasks.get(id);
        if (holder != null) {
            return holder.getTask();
        }
        return null;
    }

    public int getBanCount() {
        return this.banedParents.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<CancellableTask> setBan(TaskId parentTaskId, String reason) {
        logger.trace("setting ban for the parent task {} {}", (Object)parentTaskId, (Object)reason);
        Map<TaskId, String> map = this.banedParents;
        synchronized (map) {
            if (this.lastDiscoveryNodes.nodeExists(parentTaskId.getNodeId())) {
                this.banedParents.put(parentTaskId, reason);
            }
        }
        return this.cancellableTasks.values().stream().filter(t -> t.hasParent(parentTaskId)).map(t -> ((CancellableTaskHolder)t).task).collect(Collectors.toList());
    }

    public void removeBan(TaskId parentTaskId) {
        logger.trace("removing ban for the parent task {}", (Object)parentTaskId);
        this.banedParents.remove(parentTaskId);
    }

    public Set<TaskId> getBannedTaskIds() {
        return Collections.unmodifiableSet(this.banedParents.keySet());
    }

    public Collection<DiscoveryNode> startBanOnChildrenNodes(long taskId, Runnable onChildTasksCompleted) {
        CancellableTaskHolder holder = this.cancellableTasks.get(taskId);
        if (holder != null) {
            return holder.startBan(onChildTasksCompleted);
        }
        onChildTasksCompleted.run();
        return Collections.emptySet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        this.lastDiscoveryNodes = event.state().getNodes();
        if (event.nodesRemoved()) {
            Map<TaskId, String> map = this.banedParents;
            synchronized (map) {
                this.lastDiscoveryNodes = event.state().getNodes();
                Iterator<TaskId> banIterator = this.banedParents.keySet().iterator();
                while (banIterator.hasNext()) {
                    TaskId taskId = banIterator.next();
                    if (this.lastDiscoveryNodes.nodeExists(taskId.getNodeId())) continue;
                    logger.debug("Removing ban for the parent [{}] on the node [{}], reason: the parent node is gone", (Object)taskId, (Object)event.state().getNodes().getLocalNode());
                    banIterator.remove();
                }
            }
        }
    }

    public void waitForTaskCompletion(Task task, long untilInNanos) {
        while (System.nanoTime() - untilInNanos < 0L) {
            if (this.getTask(task.getId()) == null) {
                return;
            }
            try {
                Thread.sleep(WAIT_FOR_COMPLETION_POLL.millis());
            }
            catch (InterruptedException e) {
                throw new ElasticsearchException("Interrupted waiting for completion of [{}]", (Throwable)e, task);
            }
        }
        throw new ElasticsearchTimeoutException("Timed out waiting for completion of [{}]", task);
    }

    public Releasable startTrackingCancellableChannelTask(TcpChannel channel, CancellableTask task) {
        assert (this.cancellableTasks.containsKey(task.getId())) : "task [" + task.getId() + "] is not registered yet";
        ChannelPendingTaskTracker tracker = this.channelPendingTaskTrackers.compute(channel, (k, curr) -> {
            if (curr == null) {
                curr = new ChannelPendingTaskTracker();
            }
            curr.addTask(task);
            return curr;
        });
        if (tracker.registered.compareAndSet(false, true)) {
            channel.addCloseListener(ActionListener.wrap(r -> {
                ChannelPendingTaskTracker removedTracker = this.channelPendingTaskTrackers.remove(channel);
                assert (removedTracker == tracker);
                this.cancelTasksOnChannelClosed(tracker.drainTasks());
            }, e -> {
                assert (false) : new AssertionError("must not be here", (Throwable)e);
            }));
        }
        return () -> tracker.removeTask(task);
    }

    final int numberOfChannelPendingTaskTrackers() {
        return this.channelPendingTaskTrackers.size();
    }

    private void cancelTasksOnChannelClosed(final Set<CancellableTask> tasks) {
        if (!tasks.isEmpty()) {
            this.threadPool.generic().execute(new AbstractRunnable(){

                @Override
                public void onFailure(Exception e) {
                    logger.warn("failed to cancel tasks on channel closed", (Throwable)e);
                }

                @Override
                protected void doRun() {
                    for (CancellableTask task : tasks) {
                        TaskManager.this.cancelTaskAndDescendants(task, "channel was closed", false, ActionListener.wrap(() -> {}));
                    }
                }
            });
        }
    }

    public void cancelTaskAndDescendants(CancellableTask task, String reason, boolean waitForCompletion, ActionListener<Void> listener) {
        TaskCancellationService service = (TaskCancellationService)this.cancellationService.get();
        if (service == null) {
            assert (false) : "TaskCancellationService is not initialized";
            throw new IllegalStateException("TaskCancellationService is not initialized");
        }
        service.cancelTaskAndDescendants(task, reason, waitForCompletion, listener);
    }

    private static class CancellableTaskHolder {
        private final CancellableTask task;
        private boolean finished = false;
        private List<Runnable> cancellationListeners = null;
        private ObjectIntMap<DiscoveryNode> childTasksPerNode = null;
        private boolean banChildren = false;
        private List<Runnable> childTaskCompletedListeners = null;

        CancellableTaskHolder(CancellableTask task) {
            this.task = task;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void cancel(String reason, Runnable listener) {
            Runnable toRun;
            CancellableTaskHolder cancellableTaskHolder = this;
            synchronized (cancellableTaskHolder) {
                if (this.finished) {
                    assert (this.cancellationListeners == null);
                    toRun = listener;
                } else {
                    toRun = () -> {};
                    if (listener != null) {
                        if (this.cancellationListeners == null) {
                            this.cancellationListeners = new ArrayList<Runnable>();
                        }
                        this.cancellationListeners.add(listener);
                    }
                }
            }
            try {
                this.task.cancel(reason);
            }
            finally {
                if (toRun != null) {
                    toRun.run();
                }
            }
        }

        void cancel(String reason) {
            this.task.cancel(reason);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void finish() {
            List<Runnable> listeners;
            CancellableTaskHolder cancellableTaskHolder = this;
            synchronized (cancellableTaskHolder) {
                this.finished = true;
                if (this.cancellationListeners != null) {
                    listeners = this.cancellationListeners;
                    this.cancellationListeners = null;
                } else {
                    listeners = Collections.emptyList();
                }
            }
            this.notifyListeners(listeners);
        }

        private void notifyListeners(List<Runnable> listeners) {
            assert (!Thread.holdsLock(this));
            Exception rootException = null;
            for (Runnable listener : listeners) {
                try {
                    listener.run();
                }
                catch (RuntimeException inner) {
                    rootException = ExceptionsHelper.useOrSuppress(rootException, inner);
                }
            }
            ExceptionsHelper.reThrowIfNotNull(rootException);
        }

        public boolean hasParent(TaskId parentTaskId) {
            return this.task.getParentTaskId().equals(parentTaskId);
        }

        public CancellableTask getTask() {
            return this.task;
        }

        synchronized void registerChildNode(DiscoveryNode node) {
            if (this.banChildren) {
                throw new TaskCancelledException("The parent task was cancelled, shouldn't start any child tasks");
            }
            if (this.childTasksPerNode == null) {
                this.childTasksPerNode = new ObjectIntHashMap();
            }
            this.childTasksPerNode.addTo((Object)node, 1);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void unregisterChildNode(DiscoveryNode node) {
            List<Runnable> listeners;
            CancellableTaskHolder cancellableTaskHolder = this;
            synchronized (cancellableTaskHolder) {
                if (this.childTasksPerNode.addTo((Object)node, -1) == 0) {
                    this.childTasksPerNode.remove((Object)node);
                }
                if (this.childTasksPerNode.isEmpty() && this.childTaskCompletedListeners != null) {
                    listeners = this.childTaskCompletedListeners;
                    this.childTaskCompletedListeners = null;
                } else {
                    listeners = Collections.emptyList();
                }
            }
            this.notifyListeners(listeners);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        Set<DiscoveryNode> startBan(Runnable onChildTasksCompleted) {
            Runnable toRun;
            Set<DiscoveryNode> pendingChildNodes;
            CancellableTaskHolder cancellableTaskHolder = this;
            synchronized (cancellableTaskHolder) {
                this.banChildren = true;
                pendingChildNodes = this.childTasksPerNode == null ? Collections.emptySet() : StreamSupport.stream(this.childTasksPerNode.spliterator(), false).map(e -> (DiscoveryNode)e.key).collect(Collectors.toSet());
                if (pendingChildNodes.isEmpty()) {
                    assert (this.childTaskCompletedListeners == null);
                    toRun = onChildTasksCompleted;
                } else {
                    toRun = () -> {};
                    if (this.childTaskCompletedListeners == null) {
                        this.childTaskCompletedListeners = new ArrayList<Runnable>();
                    }
                    this.childTaskCompletedListeners.add(onChildTasksCompleted);
                }
            }
            toRun.run();
            return pendingChildNodes;
        }
    }

    private static class ChannelPendingTaskTracker {
        final AtomicBoolean registered = new AtomicBoolean();
        final Semaphore permits = Assertions.ENABLED ? new Semaphore(Integer.MAX_VALUE) : null;
        final Set<CancellableTask> pendingTasks = ConcurrentCollections.newConcurrentSet();

        private ChannelPendingTaskTracker() {
        }

        void addTask(CancellableTask task) {
            assert (this.permits.tryAcquire()) : "tracker was drained";
            boolean added = this.pendingTasks.add(task);
            assert (added) : "task " + task.getId() + " is in the pending list already";
            assert (this.releasePermit());
        }

        boolean acquireAllPermits() {
            this.permits.acquireUninterruptibly(Integer.MAX_VALUE);
            return true;
        }

        boolean releasePermit() {
            this.permits.release();
            return true;
        }

        Set<CancellableTask> drainTasks() {
            assert (this.acquireAllPermits());
            return Collections.unmodifiableSet(this.pendingTasks);
        }

        void removeTask(CancellableTask task) {
            boolean removed = this.pendingTasks.remove(task);
            assert (removed) : "task " + task.getId() + " is not in the pending list";
        }
    }
}

