"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isObjectDescriptor = isObjectDescriptor;
exports.kindToDescriptorName = kindToDescriptorName;
exports.getDescriptor = getDescriptor;
exports.TelemetryKinds = void 0;

var ts = _interopRequireWildcard(require("typescript"));

var _lodash = require("lodash");

var _utils = require("./utils");

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
let TelemetryKinds;
exports.TelemetryKinds = TelemetryKinds;

(function (TelemetryKinds) {
  TelemetryKinds[TelemetryKinds["MomentDate"] = 1000] = "MomentDate";
  TelemetryKinds[TelemetryKinds["Date"] = 10001] = "Date";
})(TelemetryKinds || (exports.TelemetryKinds = TelemetryKinds = {}));

function isObjectDescriptor(value) {
  if (typeof value === 'object') {
    if (typeof value.type === 'string' && value.type === 'object') {
      return true;
    }

    if (typeof value.type === 'undefined') {
      return true;
    }
  }

  return false;
}

function kindToDescriptorName(kind) {
  switch (kind) {
    case ts.SyntaxKind.StringKeyword:
    case ts.SyntaxKind.StringLiteral:
    case ts.SyntaxKind.SetKeyword:
    case TelemetryKinds.Date:
    case TelemetryKinds.MomentDate:
      return 'string';

    case ts.SyntaxKind.BooleanKeyword:
      return 'boolean';

    case ts.SyntaxKind.NumberKeyword:
    case ts.SyntaxKind.NumericLiteral:
      return 'number';

    default:
      throw new Error(`Unknown kind ${kind}`);
  }
}

function getDescriptor(node, program) {
  if (ts.isMethodSignature(node) || ts.isPropertySignature(node)) {
    if (node.type) {
      return getDescriptor(node.type, program);
    }
  }

  if (ts.isTypeLiteralNode(node) || ts.isInterfaceDeclaration(node)) {
    return node.members.reduce((acc, m) => {
      var _m$name;

      acc[((_m$name = m.name) === null || _m$name === void 0 ? void 0 : _m$name.getText()) || ''] = getDescriptor(m, program);
      return acc;
    }, {});
  }

  if (ts.SyntaxKind.FirstNode === node.kind) {
    return getDescriptor(node.right, program);
  }

  if (ts.isIdentifier(node)) {
    const identifierName = node.getText();

    if (identifierName === 'Date') {
      return {
        kind: TelemetryKinds.Date,
        type: 'Date'
      };
    }

    if (identifierName === 'Moment') {
      return {
        kind: TelemetryKinds.MomentDate,
        type: 'MomentDate'
      };
    }

    throw new Error(`Unsupported Identifier ${identifierName}.`);
  }

  if (ts.isTypeReferenceNode(node)) {
    const typeChecker = program.getTypeChecker();
    const symbol = typeChecker.getSymbolAtLocation(node.typeName);
    const symbolName = symbol === null || symbol === void 0 ? void 0 : symbol.getName();

    if (symbolName === 'Moment') {
      return {
        kind: TelemetryKinds.MomentDate,
        type: 'MomentDate'
      };
    }

    if (symbolName === 'Date') {
      return {
        kind: TelemetryKinds.Date,
        type: 'Date'
      };
    }

    const declaration = ((symbol === null || symbol === void 0 ? void 0 : symbol.getDeclarations()) || [])[0];

    if (declaration) {
      return getDescriptor(declaration, program);
    }

    return getDescriptor(node.typeName, program);
  }

  if (ts.isImportSpecifier(node)) {
    const source = node.getSourceFile();
    const importedModuleName = (0, _utils.getModuleSpecifier)(node);
    const declarationSource = (0, _utils.getResolvedModuleSourceFile)(source, program, importedModuleName);
    const declarationNode = (0, _utils.getIdentifierDeclarationFromSource)(node.name, declarationSource);
    return getDescriptor(declarationNode, program);
  }

  if (ts.isArrayTypeNode(node)) {
    return getDescriptor(node.elementType, program);
  }

  if (ts.isLiteralTypeNode(node)) {
    return {
      kind: node.literal.kind,
      type: ts.SyntaxKind[node.literal.kind]
    };
  }

  if (ts.isUnionTypeNode(node)) {
    const types = node.types.filter(typeNode => {
      return typeNode.kind !== ts.SyntaxKind.NullKeyword && typeNode.kind !== ts.SyntaxKind.UndefinedKeyword;
    });
    const kinds = types.map(typeNode => getDescriptor(typeNode, program));
    const uniqueKinds = (0, _lodash.uniq)(kinds, 'kind');

    if (uniqueKinds.length !== 1) {
      throw Error('Mapping does not support conflicting union types.');
    }

    return uniqueKinds[0];
  }

  switch (node.kind) {
    case ts.SyntaxKind.NumberKeyword:
    case ts.SyntaxKind.BooleanKeyword:
    case ts.SyntaxKind.StringKeyword:
    case ts.SyntaxKind.SetKeyword:
      return {
        kind: node.kind,
        type: ts.SyntaxKind[node.kind]
      };

    case ts.SyntaxKind.UnionType:
    case ts.SyntaxKind.AnyKeyword:
    default:
      throw new Error(`Unknown type ${ts.SyntaxKind[node.kind]}; ${node.getText()}`);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy90b29scy9zZXJpYWxpemVyLnRzIl0sIm5hbWVzIjpbIlRlbGVtZXRyeUtpbmRzIiwiaXNPYmplY3REZXNjcmlwdG9yIiwidmFsdWUiLCJ0eXBlIiwia2luZFRvRGVzY3JpcHRvck5hbWUiLCJraW5kIiwidHMiLCJTeW50YXhLaW5kIiwiU3RyaW5nS2V5d29yZCIsIlN0cmluZ0xpdGVyYWwiLCJTZXRLZXl3b3JkIiwiRGF0ZSIsIk1vbWVudERhdGUiLCJCb29sZWFuS2V5d29yZCIsIk51bWJlcktleXdvcmQiLCJOdW1lcmljTGl0ZXJhbCIsIkVycm9yIiwiZ2V0RGVzY3JpcHRvciIsIm5vZGUiLCJwcm9ncmFtIiwiaXNNZXRob2RTaWduYXR1cmUiLCJpc1Byb3BlcnR5U2lnbmF0dXJlIiwiaXNUeXBlTGl0ZXJhbE5vZGUiLCJpc0ludGVyZmFjZURlY2xhcmF0aW9uIiwibWVtYmVycyIsInJlZHVjZSIsImFjYyIsIm0iLCJuYW1lIiwiZ2V0VGV4dCIsIkZpcnN0Tm9kZSIsInJpZ2h0IiwiaXNJZGVudGlmaWVyIiwiaWRlbnRpZmllck5hbWUiLCJpc1R5cGVSZWZlcmVuY2VOb2RlIiwidHlwZUNoZWNrZXIiLCJnZXRUeXBlQ2hlY2tlciIsInN5bWJvbCIsImdldFN5bWJvbEF0TG9jYXRpb24iLCJ0eXBlTmFtZSIsInN5bWJvbE5hbWUiLCJnZXROYW1lIiwiZGVjbGFyYXRpb24iLCJnZXREZWNsYXJhdGlvbnMiLCJpc0ltcG9ydFNwZWNpZmllciIsInNvdXJjZSIsImdldFNvdXJjZUZpbGUiLCJpbXBvcnRlZE1vZHVsZU5hbWUiLCJkZWNsYXJhdGlvblNvdXJjZSIsImRlY2xhcmF0aW9uTm9kZSIsImlzQXJyYXlUeXBlTm9kZSIsImVsZW1lbnRUeXBlIiwiaXNMaXRlcmFsVHlwZU5vZGUiLCJsaXRlcmFsIiwiaXNVbmlvblR5cGVOb2RlIiwidHlwZXMiLCJmaWx0ZXIiLCJ0eXBlTm9kZSIsIk51bGxLZXl3b3JkIiwiVW5kZWZpbmVkS2V5d29yZCIsImtpbmRzIiwibWFwIiwidW5pcXVlS2luZHMiLCJsZW5ndGgiLCJVbmlvblR5cGUiLCJBbnlLZXl3b3JkIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7O0FBbUJBOztBQUNBOztBQUNBOzs7Ozs7QUFyQkE7Ozs7Ozs7Ozs7Ozs7Ozs7OztJQTJCWUEsYzs7O1dBQUFBLGM7QUFBQUEsRUFBQUEsYyxDQUFBQSxjO0FBQUFBLEVBQUFBLGMsQ0FBQUEsYztHQUFBQSxjLDhCQUFBQSxjOztBQWNMLFNBQVNDLGtCQUFULENBQTRCQyxLQUE1QixFQUF3QztBQUM3QyxNQUFJLE9BQU9BLEtBQVAsS0FBaUIsUUFBckIsRUFBK0I7QUFDN0IsUUFBSSxPQUFPQSxLQUFLLENBQUNDLElBQWIsS0FBc0IsUUFBdEIsSUFBa0NELEtBQUssQ0FBQ0MsSUFBTixLQUFlLFFBQXJELEVBQStEO0FBQzdELGFBQU8sSUFBUDtBQUNEOztBQUVELFFBQUksT0FBT0QsS0FBSyxDQUFDQyxJQUFiLEtBQXNCLFdBQTFCLEVBQXVDO0FBQ3JDLGFBQU8sSUFBUDtBQUNEO0FBQ0Y7O0FBRUQsU0FBTyxLQUFQO0FBQ0Q7O0FBRU0sU0FBU0Msb0JBQVQsQ0FBOEJDLElBQTlCLEVBQTRDO0FBQ2pELFVBQVFBLElBQVI7QUFDRSxTQUFLQyxFQUFFLENBQUNDLFVBQUgsQ0FBY0MsYUFBbkI7QUFDQSxTQUFLRixFQUFFLENBQUNDLFVBQUgsQ0FBY0UsYUFBbkI7QUFDQSxTQUFLSCxFQUFFLENBQUNDLFVBQUgsQ0FBY0csVUFBbkI7QUFDQSxTQUFLVixjQUFjLENBQUNXLElBQXBCO0FBQ0EsU0FBS1gsY0FBYyxDQUFDWSxVQUFwQjtBQUNFLGFBQU8sUUFBUDs7QUFDRixTQUFLTixFQUFFLENBQUNDLFVBQUgsQ0FBY00sY0FBbkI7QUFDRSxhQUFPLFNBQVA7O0FBQ0YsU0FBS1AsRUFBRSxDQUFDQyxVQUFILENBQWNPLGFBQW5CO0FBQ0EsU0FBS1IsRUFBRSxDQUFDQyxVQUFILENBQWNRLGNBQW5CO0FBQ0UsYUFBTyxRQUFQOztBQUNGO0FBQ0UsWUFBTSxJQUFJQyxLQUFKLENBQVcsZ0JBQWVYLElBQUssRUFBL0IsQ0FBTjtBQWJKO0FBZUQ7O0FBRU0sU0FBU1ksYUFBVCxDQUF1QkMsSUFBdkIsRUFBc0NDLE9BQXRDLEVBQXlGO0FBQzlGLE1BQUliLEVBQUUsQ0FBQ2MsaUJBQUgsQ0FBcUJGLElBQXJCLEtBQThCWixFQUFFLENBQUNlLG1CQUFILENBQXVCSCxJQUF2QixDQUFsQyxFQUFnRTtBQUM5RCxRQUFJQSxJQUFJLENBQUNmLElBQVQsRUFBZTtBQUNiLGFBQU9jLGFBQWEsQ0FBQ0MsSUFBSSxDQUFDZixJQUFOLEVBQVlnQixPQUFaLENBQXBCO0FBQ0Q7QUFDRjs7QUFDRCxNQUFJYixFQUFFLENBQUNnQixpQkFBSCxDQUFxQkosSUFBckIsS0FBOEJaLEVBQUUsQ0FBQ2lCLHNCQUFILENBQTBCTCxJQUExQixDQUFsQyxFQUFtRTtBQUNqRSxXQUFPQSxJQUFJLENBQUNNLE9BQUwsQ0FBYUMsTUFBYixDQUFvQixDQUFDQyxHQUFELEVBQU1DLENBQU4sS0FBWTtBQUFBOztBQUNyQ0QsTUFBQUEsR0FBRyxDQUFDLFlBQUFDLENBQUMsQ0FBQ0MsSUFBRixvREFBUUMsT0FBUixPQUFxQixFQUF0QixDQUFILEdBQStCWixhQUFhLENBQUNVLENBQUQsRUFBSVIsT0FBSixDQUE1QztBQUNBLGFBQU9PLEdBQVA7QUFDRCxLQUhNLEVBR0osRUFISSxDQUFQO0FBSUQ7O0FBRUQsTUFBSXBCLEVBQUUsQ0FBQ0MsVUFBSCxDQUFjdUIsU0FBZCxLQUE0QlosSUFBSSxDQUFDYixJQUFyQyxFQUEyQztBQUN6QyxXQUFPWSxhQUFhLENBQUVDLElBQUQsQ0FBY2EsS0FBZixFQUFzQlosT0FBdEIsQ0FBcEI7QUFDRDs7QUFFRCxNQUFJYixFQUFFLENBQUMwQixZQUFILENBQWdCZCxJQUFoQixDQUFKLEVBQTJCO0FBQ3pCLFVBQU1lLGNBQWMsR0FBR2YsSUFBSSxDQUFDVyxPQUFMLEVBQXZCOztBQUNBLFFBQUlJLGNBQWMsS0FBSyxNQUF2QixFQUErQjtBQUM3QixhQUFPO0FBQUU1QixRQUFBQSxJQUFJLEVBQUVMLGNBQWMsQ0FBQ1csSUFBdkI7QUFBNkJSLFFBQUFBLElBQUksRUFBRTtBQUFuQyxPQUFQO0FBQ0Q7O0FBQ0QsUUFBSThCLGNBQWMsS0FBSyxRQUF2QixFQUFpQztBQUMvQixhQUFPO0FBQUU1QixRQUFBQSxJQUFJLEVBQUVMLGNBQWMsQ0FBQ1ksVUFBdkI7QUFBbUNULFFBQUFBLElBQUksRUFBRTtBQUF6QyxPQUFQO0FBQ0Q7O0FBQ0QsVUFBTSxJQUFJYSxLQUFKLENBQVcsMEJBQXlCaUIsY0FBZSxHQUFuRCxDQUFOO0FBQ0Q7O0FBRUQsTUFBSTNCLEVBQUUsQ0FBQzRCLG1CQUFILENBQXVCaEIsSUFBdkIsQ0FBSixFQUFrQztBQUNoQyxVQUFNaUIsV0FBVyxHQUFHaEIsT0FBTyxDQUFDaUIsY0FBUixFQUFwQjtBQUNBLFVBQU1DLE1BQU0sR0FBR0YsV0FBVyxDQUFDRyxtQkFBWixDQUFnQ3BCLElBQUksQ0FBQ3FCLFFBQXJDLENBQWY7QUFDQSxVQUFNQyxVQUFVLEdBQUdILE1BQUgsYUFBR0EsTUFBSCx1QkFBR0EsTUFBTSxDQUFFSSxPQUFSLEVBQW5COztBQUNBLFFBQUlELFVBQVUsS0FBSyxRQUFuQixFQUE2QjtBQUMzQixhQUFPO0FBQUVuQyxRQUFBQSxJQUFJLEVBQUVMLGNBQWMsQ0FBQ1ksVUFBdkI7QUFBbUNULFFBQUFBLElBQUksRUFBRTtBQUF6QyxPQUFQO0FBQ0Q7O0FBQ0QsUUFBSXFDLFVBQVUsS0FBSyxNQUFuQixFQUEyQjtBQUN6QixhQUFPO0FBQUVuQyxRQUFBQSxJQUFJLEVBQUVMLGNBQWMsQ0FBQ1csSUFBdkI7QUFBNkJSLFFBQUFBLElBQUksRUFBRTtBQUFuQyxPQUFQO0FBQ0Q7O0FBQ0QsVUFBTXVDLFdBQVcsR0FBRyxDQUFDLENBQUFMLE1BQU0sU0FBTixJQUFBQSxNQUFNLFdBQU4sWUFBQUEsTUFBTSxDQUFFTSxlQUFSLE9BQTZCLEVBQTlCLEVBQWtDLENBQWxDLENBQXBCOztBQUNBLFFBQUlELFdBQUosRUFBaUI7QUFDZixhQUFPekIsYUFBYSxDQUFDeUIsV0FBRCxFQUFjdkIsT0FBZCxDQUFwQjtBQUNEOztBQUNELFdBQU9GLGFBQWEsQ0FBQ0MsSUFBSSxDQUFDcUIsUUFBTixFQUFnQnBCLE9BQWhCLENBQXBCO0FBQ0Q7O0FBRUQsTUFBSWIsRUFBRSxDQUFDc0MsaUJBQUgsQ0FBcUIxQixJQUFyQixDQUFKLEVBQWdDO0FBQzlCLFVBQU0yQixNQUFNLEdBQUczQixJQUFJLENBQUM0QixhQUFMLEVBQWY7QUFDQSxVQUFNQyxrQkFBa0IsR0FBRywrQkFBbUI3QixJQUFuQixDQUEzQjtBQUVBLFVBQU04QixpQkFBaUIsR0FBRyx3Q0FBNEJILE1BQTVCLEVBQW9DMUIsT0FBcEMsRUFBNkM0QixrQkFBN0MsQ0FBMUI7QUFDQSxVQUFNRSxlQUFlLEdBQUcsK0NBQW1DL0IsSUFBSSxDQUFDVSxJQUF4QyxFQUE4Q29CLGlCQUE5QyxDQUF4QjtBQUNBLFdBQU8vQixhQUFhLENBQUNnQyxlQUFELEVBQWtCOUIsT0FBbEIsQ0FBcEI7QUFDRDs7QUFFRCxNQUFJYixFQUFFLENBQUM0QyxlQUFILENBQW1CaEMsSUFBbkIsQ0FBSixFQUE4QjtBQUM1QixXQUFPRCxhQUFhLENBQUNDLElBQUksQ0FBQ2lDLFdBQU4sRUFBbUJoQyxPQUFuQixDQUFwQjtBQUNEOztBQUVELE1BQUliLEVBQUUsQ0FBQzhDLGlCQUFILENBQXFCbEMsSUFBckIsQ0FBSixFQUFnQztBQUM5QixXQUFPO0FBQ0xiLE1BQUFBLElBQUksRUFBRWEsSUFBSSxDQUFDbUMsT0FBTCxDQUFhaEQsSUFEZDtBQUVMRixNQUFBQSxJQUFJLEVBQUVHLEVBQUUsQ0FBQ0MsVUFBSCxDQUFjVyxJQUFJLENBQUNtQyxPQUFMLENBQWFoRCxJQUEzQjtBQUZELEtBQVA7QUFJRDs7QUFFRCxNQUFJQyxFQUFFLENBQUNnRCxlQUFILENBQW1CcEMsSUFBbkIsQ0FBSixFQUE4QjtBQUM1QixVQUFNcUMsS0FBSyxHQUFHckMsSUFBSSxDQUFDcUMsS0FBTCxDQUFXQyxNQUFYLENBQW1CQyxRQUFELElBQWM7QUFDNUMsYUFDRUEsUUFBUSxDQUFDcEQsSUFBVCxLQUFrQkMsRUFBRSxDQUFDQyxVQUFILENBQWNtRCxXQUFoQyxJQUNBRCxRQUFRLENBQUNwRCxJQUFULEtBQWtCQyxFQUFFLENBQUNDLFVBQUgsQ0FBY29ELGdCQUZsQztBQUlELEtBTGEsQ0FBZDtBQU9BLFVBQU1DLEtBQUssR0FBR0wsS0FBSyxDQUFDTSxHQUFOLENBQVdKLFFBQUQsSUFBY3hDLGFBQWEsQ0FBQ3dDLFFBQUQsRUFBV3RDLE9BQVgsQ0FBckMsQ0FBZDtBQUVBLFVBQU0yQyxXQUFXLEdBQUcsa0JBQUtGLEtBQUwsRUFBWSxNQUFaLENBQXBCOztBQUVBLFFBQUlFLFdBQVcsQ0FBQ0MsTUFBWixLQUF1QixDQUEzQixFQUE4QjtBQUM1QixZQUFNL0MsS0FBSyxDQUFDLG1EQUFELENBQVg7QUFDRDs7QUFFRCxXQUFPOEMsV0FBVyxDQUFDLENBQUQsQ0FBbEI7QUFDRDs7QUFFRCxVQUFRNUMsSUFBSSxDQUFDYixJQUFiO0FBQ0UsU0FBS0MsRUFBRSxDQUFDQyxVQUFILENBQWNPLGFBQW5CO0FBQ0EsU0FBS1IsRUFBRSxDQUFDQyxVQUFILENBQWNNLGNBQW5CO0FBQ0EsU0FBS1AsRUFBRSxDQUFDQyxVQUFILENBQWNDLGFBQW5CO0FBQ0EsU0FBS0YsRUFBRSxDQUFDQyxVQUFILENBQWNHLFVBQW5CO0FBQ0UsYUFBTztBQUFFTCxRQUFBQSxJQUFJLEVBQUVhLElBQUksQ0FBQ2IsSUFBYjtBQUFtQkYsUUFBQUEsSUFBSSxFQUFFRyxFQUFFLENBQUNDLFVBQUgsQ0FBY1csSUFBSSxDQUFDYixJQUFuQjtBQUF6QixPQUFQOztBQUNGLFNBQUtDLEVBQUUsQ0FBQ0MsVUFBSCxDQUFjeUQsU0FBbkI7QUFDQSxTQUFLMUQsRUFBRSxDQUFDQyxVQUFILENBQWMwRCxVQUFuQjtBQUNBO0FBQ0UsWUFBTSxJQUFJakQsS0FBSixDQUFXLGdCQUFlVixFQUFFLENBQUNDLFVBQUgsQ0FBY1csSUFBSSxDQUFDYixJQUFuQixDQUF5QixLQUFJYSxJQUFJLENBQUNXLE9BQUwsRUFBZSxFQUF0RSxDQUFOO0FBVEo7QUFXRCIsInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBMaWNlbnNlZCB0byBFbGFzdGljc2VhcmNoIEIuVi4gdW5kZXIgb25lIG9yIG1vcmUgY29udHJpYnV0b3JcbiAqIGxpY2Vuc2UgYWdyZWVtZW50cy4gU2VlIHRoZSBOT1RJQ0UgZmlsZSBkaXN0cmlidXRlZCB3aXRoXG4gKiB0aGlzIHdvcmsgZm9yIGFkZGl0aW9uYWwgaW5mb3JtYXRpb24gcmVnYXJkaW5nIGNvcHlyaWdodFxuICogb3duZXJzaGlwLiBFbGFzdGljc2VhcmNoIEIuVi4gbGljZW5zZXMgdGhpcyBmaWxlIHRvIHlvdSB1bmRlclxuICogdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKTsgeW91IG1heVxuICogbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiAqIFlvdSBtYXkgb2J0YWluIGEgY29weSBvZiB0aGUgTGljZW5zZSBhdFxuICpcbiAqICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZyxcbiAqIHNvZnR3YXJlIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuXG4gKiBcIkFTIElTXCIgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFUyBPUiBDT05ESVRJT05TIE9GIEFOWVxuICogS0lORCwgZWl0aGVyIGV4cHJlc3Mgb3IgaW1wbGllZC4gIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlXG4gKiBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kIGxpbWl0YXRpb25zXG4gKiB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgKiBhcyB0cyBmcm9tICd0eXBlc2NyaXB0JztcbmltcG9ydCB7IHVuaXEgfSBmcm9tICdsb2Rhc2gnO1xuaW1wb3J0IHtcbiAgZ2V0UmVzb2x2ZWRNb2R1bGVTb3VyY2VGaWxlLFxuICBnZXRJZGVudGlmaWVyRGVjbGFyYXRpb25Gcm9tU291cmNlLFxuICBnZXRNb2R1bGVTcGVjaWZpZXIsXG59IGZyb20gJy4vdXRpbHMnO1xuXG5leHBvcnQgZW51bSBUZWxlbWV0cnlLaW5kcyB7XG4gIE1vbWVudERhdGUgPSAxMDAwLFxuICBEYXRlID0gMTAwMDEsXG59XG5cbmludGVyZmFjZSBEZXNjcmlwdG9yVmFsdWUge1xuICBraW5kOiB0cy5TeW50YXhLaW5kIHwgVGVsZW1ldHJ5S2luZHM7XG4gIHR5cGU6IGtleW9mIHR5cGVvZiB0cy5TeW50YXhLaW5kIHwga2V5b2YgdHlwZW9mIFRlbGVtZXRyeUtpbmRzO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIERlc2NyaXB0b3Ige1xuICBbbmFtZTogc3RyaW5nXTogRGVzY3JpcHRvciB8IERlc2NyaXB0b3JWYWx1ZTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzT2JqZWN0RGVzY3JpcHRvcih2YWx1ZTogYW55KSB7XG4gIGlmICh0eXBlb2YgdmFsdWUgPT09ICdvYmplY3QnKSB7XG4gICAgaWYgKHR5cGVvZiB2YWx1ZS50eXBlID09PSAnc3RyaW5nJyAmJiB2YWx1ZS50eXBlID09PSAnb2JqZWN0Jykge1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuXG4gICAgaWYgKHR5cGVvZiB2YWx1ZS50eXBlID09PSAndW5kZWZpbmVkJykge1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIGZhbHNlO1xufVxuXG5leHBvcnQgZnVuY3Rpb24ga2luZFRvRGVzY3JpcHRvck5hbWUoa2luZDogbnVtYmVyKSB7XG4gIHN3aXRjaCAoa2luZCkge1xuICAgIGNhc2UgdHMuU3ludGF4S2luZC5TdHJpbmdLZXl3b3JkOlxuICAgIGNhc2UgdHMuU3ludGF4S2luZC5TdHJpbmdMaXRlcmFsOlxuICAgIGNhc2UgdHMuU3ludGF4S2luZC5TZXRLZXl3b3JkOlxuICAgIGNhc2UgVGVsZW1ldHJ5S2luZHMuRGF0ZTpcbiAgICBjYXNlIFRlbGVtZXRyeUtpbmRzLk1vbWVudERhdGU6XG4gICAgICByZXR1cm4gJ3N0cmluZyc7XG4gICAgY2FzZSB0cy5TeW50YXhLaW5kLkJvb2xlYW5LZXl3b3JkOlxuICAgICAgcmV0dXJuICdib29sZWFuJztcbiAgICBjYXNlIHRzLlN5bnRheEtpbmQuTnVtYmVyS2V5d29yZDpcbiAgICBjYXNlIHRzLlN5bnRheEtpbmQuTnVtZXJpY0xpdGVyYWw6XG4gICAgICByZXR1cm4gJ251bWJlcic7XG4gICAgZGVmYXVsdDpcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBraW5kICR7a2luZH1gKTtcbiAgfVxufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0RGVzY3JpcHRvcihub2RlOiB0cy5Ob2RlLCBwcm9ncmFtOiB0cy5Qcm9ncmFtKTogRGVzY3JpcHRvciB8IERlc2NyaXB0b3JWYWx1ZSB7XG4gIGlmICh0cy5pc01ldGhvZFNpZ25hdHVyZShub2RlKSB8fCB0cy5pc1Byb3BlcnR5U2lnbmF0dXJlKG5vZGUpKSB7XG4gICAgaWYgKG5vZGUudHlwZSkge1xuICAgICAgcmV0dXJuIGdldERlc2NyaXB0b3Iobm9kZS50eXBlLCBwcm9ncmFtKTtcbiAgICB9XG4gIH1cbiAgaWYgKHRzLmlzVHlwZUxpdGVyYWxOb2RlKG5vZGUpIHx8IHRzLmlzSW50ZXJmYWNlRGVjbGFyYXRpb24obm9kZSkpIHtcbiAgICByZXR1cm4gbm9kZS5tZW1iZXJzLnJlZHVjZSgoYWNjLCBtKSA9PiB7XG4gICAgICBhY2NbbS5uYW1lPy5nZXRUZXh0KCkgfHwgJyddID0gZ2V0RGVzY3JpcHRvcihtLCBwcm9ncmFtKTtcbiAgICAgIHJldHVybiBhY2M7XG4gICAgfSwge30gYXMgYW55KTtcbiAgfVxuXG4gIGlmICh0cy5TeW50YXhLaW5kLkZpcnN0Tm9kZSA9PT0gbm9kZS5raW5kKSB7XG4gICAgcmV0dXJuIGdldERlc2NyaXB0b3IoKG5vZGUgYXMgYW55KS5yaWdodCwgcHJvZ3JhbSk7XG4gIH1cblxuICBpZiAodHMuaXNJZGVudGlmaWVyKG5vZGUpKSB7XG4gICAgY29uc3QgaWRlbnRpZmllck5hbWUgPSBub2RlLmdldFRleHQoKTtcbiAgICBpZiAoaWRlbnRpZmllck5hbWUgPT09ICdEYXRlJykge1xuICAgICAgcmV0dXJuIHsga2luZDogVGVsZW1ldHJ5S2luZHMuRGF0ZSwgdHlwZTogJ0RhdGUnIH07XG4gICAgfVxuICAgIGlmIChpZGVudGlmaWVyTmFtZSA9PT0gJ01vbWVudCcpIHtcbiAgICAgIHJldHVybiB7IGtpbmQ6IFRlbGVtZXRyeUtpbmRzLk1vbWVudERhdGUsIHR5cGU6ICdNb21lbnREYXRlJyB9O1xuICAgIH1cbiAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIElkZW50aWZpZXIgJHtpZGVudGlmaWVyTmFtZX0uYCk7XG4gIH1cblxuICBpZiAodHMuaXNUeXBlUmVmZXJlbmNlTm9kZShub2RlKSkge1xuICAgIGNvbnN0IHR5cGVDaGVja2VyID0gcHJvZ3JhbS5nZXRUeXBlQ2hlY2tlcigpO1xuICAgIGNvbnN0IHN5bWJvbCA9IHR5cGVDaGVja2VyLmdldFN5bWJvbEF0TG9jYXRpb24obm9kZS50eXBlTmFtZSk7XG4gICAgY29uc3Qgc3ltYm9sTmFtZSA9IHN5bWJvbD8uZ2V0TmFtZSgpO1xuICAgIGlmIChzeW1ib2xOYW1lID09PSAnTW9tZW50Jykge1xuICAgICAgcmV0dXJuIHsga2luZDogVGVsZW1ldHJ5S2luZHMuTW9tZW50RGF0ZSwgdHlwZTogJ01vbWVudERhdGUnIH07XG4gICAgfVxuICAgIGlmIChzeW1ib2xOYW1lID09PSAnRGF0ZScpIHtcbiAgICAgIHJldHVybiB7IGtpbmQ6IFRlbGVtZXRyeUtpbmRzLkRhdGUsIHR5cGU6ICdEYXRlJyB9O1xuICAgIH1cbiAgICBjb25zdCBkZWNsYXJhdGlvbiA9IChzeW1ib2w/LmdldERlY2xhcmF0aW9ucygpIHx8IFtdKVswXTtcbiAgICBpZiAoZGVjbGFyYXRpb24pIHtcbiAgICAgIHJldHVybiBnZXREZXNjcmlwdG9yKGRlY2xhcmF0aW9uLCBwcm9ncmFtKTtcbiAgICB9XG4gICAgcmV0dXJuIGdldERlc2NyaXB0b3Iobm9kZS50eXBlTmFtZSwgcHJvZ3JhbSk7XG4gIH1cblxuICBpZiAodHMuaXNJbXBvcnRTcGVjaWZpZXIobm9kZSkpIHtcbiAgICBjb25zdCBzb3VyY2UgPSBub2RlLmdldFNvdXJjZUZpbGUoKTtcbiAgICBjb25zdCBpbXBvcnRlZE1vZHVsZU5hbWUgPSBnZXRNb2R1bGVTcGVjaWZpZXIobm9kZSk7XG5cbiAgICBjb25zdCBkZWNsYXJhdGlvblNvdXJjZSA9IGdldFJlc29sdmVkTW9kdWxlU291cmNlRmlsZShzb3VyY2UsIHByb2dyYW0sIGltcG9ydGVkTW9kdWxlTmFtZSk7XG4gICAgY29uc3QgZGVjbGFyYXRpb25Ob2RlID0gZ2V0SWRlbnRpZmllckRlY2xhcmF0aW9uRnJvbVNvdXJjZShub2RlLm5hbWUsIGRlY2xhcmF0aW9uU291cmNlKTtcbiAgICByZXR1cm4gZ2V0RGVzY3JpcHRvcihkZWNsYXJhdGlvbk5vZGUsIHByb2dyYW0pO1xuICB9XG5cbiAgaWYgKHRzLmlzQXJyYXlUeXBlTm9kZShub2RlKSkge1xuICAgIHJldHVybiBnZXREZXNjcmlwdG9yKG5vZGUuZWxlbWVudFR5cGUsIHByb2dyYW0pO1xuICB9XG5cbiAgaWYgKHRzLmlzTGl0ZXJhbFR5cGVOb2RlKG5vZGUpKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGtpbmQ6IG5vZGUubGl0ZXJhbC5raW5kLFxuICAgICAgdHlwZTogdHMuU3ludGF4S2luZFtub2RlLmxpdGVyYWwua2luZF0gYXMga2V5b2YgdHlwZW9mIHRzLlN5bnRheEtpbmQsXG4gICAgfTtcbiAgfVxuXG4gIGlmICh0cy5pc1VuaW9uVHlwZU5vZGUobm9kZSkpIHtcbiAgICBjb25zdCB0eXBlcyA9IG5vZGUudHlwZXMuZmlsdGVyKCh0eXBlTm9kZSkgPT4ge1xuICAgICAgcmV0dXJuIChcbiAgICAgICAgdHlwZU5vZGUua2luZCAhPT0gdHMuU3ludGF4S2luZC5OdWxsS2V5d29yZCAmJlxuICAgICAgICB0eXBlTm9kZS5raW5kICE9PSB0cy5TeW50YXhLaW5kLlVuZGVmaW5lZEtleXdvcmRcbiAgICAgICk7XG4gICAgfSk7XG5cbiAgICBjb25zdCBraW5kcyA9IHR5cGVzLm1hcCgodHlwZU5vZGUpID0+IGdldERlc2NyaXB0b3IodHlwZU5vZGUsIHByb2dyYW0pKTtcblxuICAgIGNvbnN0IHVuaXF1ZUtpbmRzID0gdW5pcShraW5kcywgJ2tpbmQnKTtcblxuICAgIGlmICh1bmlxdWVLaW5kcy5sZW5ndGggIT09IDEpIHtcbiAgICAgIHRocm93IEVycm9yKCdNYXBwaW5nIGRvZXMgbm90IHN1cHBvcnQgY29uZmxpY3RpbmcgdW5pb24gdHlwZXMuJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHVuaXF1ZUtpbmRzWzBdO1xuICB9XG5cbiAgc3dpdGNoIChub2RlLmtpbmQpIHtcbiAgICBjYXNlIHRzLlN5bnRheEtpbmQuTnVtYmVyS2V5d29yZDpcbiAgICBjYXNlIHRzLlN5bnRheEtpbmQuQm9vbGVhbktleXdvcmQ6XG4gICAgY2FzZSB0cy5TeW50YXhLaW5kLlN0cmluZ0tleXdvcmQ6XG4gICAgY2FzZSB0cy5TeW50YXhLaW5kLlNldEtleXdvcmQ6XG4gICAgICByZXR1cm4geyBraW5kOiBub2RlLmtpbmQsIHR5cGU6IHRzLlN5bnRheEtpbmRbbm9kZS5raW5kXSBhcyBrZXlvZiB0eXBlb2YgdHMuU3ludGF4S2luZCB9O1xuICAgIGNhc2UgdHMuU3ludGF4S2luZC5VbmlvblR5cGU6XG4gICAgY2FzZSB0cy5TeW50YXhLaW5kLkFueUtleXdvcmQ6XG4gICAgZGVmYXVsdDpcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biB0eXBlICR7dHMuU3ludGF4S2luZFtub2RlLmtpbmRdfTsgJHtub2RlLmdldFRleHQoKX1gKTtcbiAgfVxufVxuIl19