from datetime import datetime
from typing import IO, Any, Callable, Generator, Mapping, MutableMapping, Text

_serializer = Any  # must be an object that has "dumps" and "loads" attributes (e.g. the json module)

def want_bytes(s: Text | bytes, encoding: Text = ..., errors: Text = ...) -> bytes: ...

class BadData(Exception):
    message: str
    def __init__(self, message: str) -> None: ...

class BadPayload(BadData):
    original_error: Exception | None
    def __init__(self, message: str, original_error: Exception | None = ...) -> None: ...

class BadSignature(BadData):
    payload: Any | None
    def __init__(self, message: str, payload: Any | None = ...) -> None: ...

class BadTimeSignature(BadSignature):
    date_signed: int | None
    def __init__(self, message: str, payload: Any | None = ..., date_signed: int | None = ...) -> None: ...

class BadHeader(BadSignature):
    header: Any
    original_error: Any
    def __init__(
        self, message: str, payload: Any | None = ..., header: Any | None = ..., original_error: Any | None = ...
    ) -> None: ...

class SignatureExpired(BadTimeSignature): ...

def base64_encode(string: Text | bytes) -> bytes: ...
def base64_decode(string: Text | bytes) -> bytes: ...

class SigningAlgorithm(object):
    def get_signature(self, key: bytes, value: bytes) -> bytes: ...
    def verify_signature(self, key: bytes, value: bytes, sig: bytes) -> bool: ...

class NoneAlgorithm(SigningAlgorithm):
    def get_signature(self, key: bytes, value: bytes) -> bytes: ...

class HMACAlgorithm(SigningAlgorithm):
    default_digest_method: Callable[..., Any]
    digest_method: Callable[..., Any]
    def __init__(self, digest_method: Callable[..., Any] | None = ...) -> None: ...
    def get_signature(self, key: bytes, value: bytes) -> bytes: ...

class Signer(object):
    default_digest_method: Callable[..., Any] = ...
    default_key_derivation: str = ...

    secret_key: bytes
    sep: bytes
    salt: Text | bytes
    key_derivation: str
    digest_method: Callable[..., Any]
    algorithm: SigningAlgorithm
    def __init__(
        self,
        secret_key: Text | bytes,
        salt: Text | bytes | None = ...,
        sep: Text | bytes | None = ...,
        key_derivation: str | None = ...,
        digest_method: Callable[..., Any] | None = ...,
        algorithm: SigningAlgorithm | None = ...,
    ) -> None: ...
    def derive_key(self) -> bytes: ...
    def get_signature(self, value: Text | bytes) -> bytes: ...
    def sign(self, value: Text | bytes) -> bytes: ...
    def verify_signature(self, value: bytes, sig: Text | bytes) -> bool: ...
    def unsign(self, signed_value: Text | bytes) -> bytes: ...
    def validate(self, signed_value: Text | bytes) -> bool: ...

class TimestampSigner(Signer):
    def get_timestamp(self) -> int: ...
    def timestamp_to_datetime(self, ts: float) -> datetime: ...
    def sign(self, value: Text | bytes) -> bytes: ...
    def unsign(
        self, value: Text | bytes, max_age: int | None = ..., return_timestamp: bool = ...
    ) -> Any: ...  # morally -> bytes | Tuple[bytes, datetime]
    def validate(self, signed_value: Text | bytes, max_age: int | None = ...) -> bool: ...

class Serializer(object):
    default_serializer: _serializer = ...
    default_signer: Callable[..., Signer] = ...

    secret_key: bytes
    salt: bytes
    serializer: _serializer
    is_text_serializer: bool
    signer: Callable[..., Signer]
    signer_kwargs: MutableMapping[str, Any]
    def __init__(
        self,
        secret_key: Text | bytes,
        salt: Text | bytes | None = ...,
        serializer: _serializer | None = ...,
        signer: Callable[..., Signer] | None = ...,
        signer_kwargs: MutableMapping[str, Any] | None = ...,
    ) -> None: ...
    def load_payload(self, payload: bytes, serializer: _serializer | None = ...) -> Any: ...
    def dump_payload(self, obj: Any) -> bytes: ...
    def make_signer(self, salt: Text | bytes | None = ...) -> Signer: ...
    def iter_unsigners(self, salt: Text | bytes | None = ...) -> Generator[Any, None, None]: ...
    def dumps(self, obj: Any, salt: Text | bytes | None = ...) -> Any: ...  # morally -> str | bytes
    def dump(self, obj: Any, f: IO[Any], salt: Text | bytes | None = ...) -> None: ...
    def loads(self, s: Text | bytes, salt: Text | bytes | None = ...) -> Any: ...
    def load(self, f: IO[Any], salt: Text | bytes | None = ...) -> Any: ...
    def loads_unsafe(self, s: Text | bytes, salt: Text | bytes | None = ...) -> tuple[bool, Any | None]: ...
    def load_unsafe(self, f: IO[Any], salt: Text | bytes | None = ...) -> tuple[bool, Any | None]: ...

class TimedSerializer(Serializer):
    def loads(
        self, s: Text | bytes, salt: Text | bytes | None = ..., max_age: int | None = ..., return_timestamp: bool = ...
    ) -> Any: ...  # morally -> Any | Tuple[Any, datetime]
    def loads_unsafe(self, s: Text | bytes, salt: Text | bytes | None = ..., max_age: int | None = ...) -> tuple[bool, Any]: ...

class JSONWebSignatureSerializer(Serializer):
    jws_algorithms: MutableMapping[Text, SigningAlgorithm] = ...
    default_algorithm: Text = ...
    default_serializer: Any = ...

    algorithm_name: Text
    algorithm: SigningAlgorithm
    def __init__(
        self,
        secret_key: Text | bytes,
        salt: Text | bytes | None = ...,
        serializer: _serializer | None = ...,
        signer: Callable[..., Signer] | None = ...,
        signer_kwargs: MutableMapping[str, Any] | None = ...,
        algorithm_name: Text | None = ...,
    ) -> None: ...
    def load_payload(
        self, payload: Text | bytes, serializer: _serializer | None = ..., return_header: bool = ...
    ) -> Any: ...  # morally -> Any | Tuple[Any, MutableMapping[str, Any]]
    def dump_payload(self, header: Mapping[str, Any], obj: Any) -> bytes: ...  # type: ignore
    def make_algorithm(self, algorithm_name: Text) -> SigningAlgorithm: ...
    def make_signer(self, salt: Text | bytes | None = ..., algorithm: SigningAlgorithm = ...) -> Signer: ...
    def make_header(self, header_fields: Mapping[str, Any] | None) -> MutableMapping[str, Any]: ...
    def dumps(self, obj: Any, salt: Text | bytes | None = ..., header_fields: Mapping[str, Any] | None = ...) -> bytes: ...
    def loads(
        self, s: Text | bytes, salt: Text | bytes | None = ..., return_header: bool = ...
    ) -> Any: ...  # morally -> Any | Tuple[Any, MutableMapping[str, Any]]
    def loads_unsafe(self, s: Text | bytes, salt: Text | bytes | None = ..., return_header: bool = ...) -> tuple[bool, Any]: ...

class TimedJSONWebSignatureSerializer(JSONWebSignatureSerializer):
    DEFAULT_EXPIRES_IN: int = ...
    expires_in: int
    def __init__(
        self,
        secret_key: Text | bytes,
        expires_in: int | None = ...,
        salt: Text | bytes | None = ...,
        serializer: _serializer | None = ...,
        signer: Callable[..., Signer] | None = ...,
        signer_kwargs: MutableMapping[str, Any] | None = ...,
        algorithm_name: Text | None = ...,
    ) -> None: ...
    def make_header(self, header_fields: Mapping[str, Any] | None) -> MutableMapping[str, Any]: ...
    def loads(
        self, s: Text | bytes, salt: Text | bytes | None = ..., return_header: bool = ...
    ) -> Any: ...  # morally -> Any | Tuple[Any, MutableMapping[str, Any]]
    def get_issue_date(self, header: Mapping[str, Any]) -> datetime | None: ...
    def now(self) -> int: ...

class _URLSafeSerializerMixin(object):
    default_serializer: _serializer = ...
    def load_payload(self, payload: bytes, serializer: _serializer | None = ...) -> Any: ...
    def dump_payload(self, obj: Any) -> bytes: ...

class URLSafeSerializer(_URLSafeSerializerMixin, Serializer): ...
class URLSafeTimedSerializer(_URLSafeSerializerMixin, TimedSerializer): ...
