"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;

var _uuid = require("uuid");

var _sample_panels = require("../../common/helpers/custom_panels/sample_panels");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CustomPanelsAdaptor {
  constructor() {
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });

    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });

    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });

    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });

    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });

    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });

    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };

      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }

      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });

    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });

    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });

    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });

    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        user_configs: visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs) : {},
        sub_type: visualization.savedVisualization.hasOwnProperty('sub_type') ? visualization.savedVisualization.sub_type : '',
        units_of_measure: visualization.savedVisualization.hasOwnProperty('units_of_measure') ? visualization.savedVisualization.units_of_measure : '',
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });

    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });

    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const x_left = Math.max(bb1.x1, bb2.x1);
      const y_top = Math.max(bb1.y1, bb2.y1);
      const x_right = Math.min(bb1.x2, bb2.x2);
      const y_bottom = Math.min(bb1.y2, bb2.y2);
      if (x_right < x_left || y_bottom < y_top) return 0;
      return (x_right - x_left) * (y_bottom - y_top);
    });

    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });

    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0; // check if we can place the new visualization at default location

      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      } // else place the new visualization at the bottom of the panel


      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });

    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];

        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }

        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "addMultipleVisualizations", async (client, panelId, savedVisualizationIds) => {
      try {
        let allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [...allPanelVisualizations];
        savedVisualizationIds.map(savedVisualizationId => {
          newDimensions = this.getNewVizDimensions(visualizationsList);
          visualizationsList = [...visualizationsList, {
            id: 'panel_viz_' + (0, _uuid.v4)(),
            savedVisualizationId,
            ...newDimensions
          }];
        });
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: visualizationsList
        });
        return visualizationsList;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];

        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({ ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }

        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _sample_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  }

}

exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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