<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2023 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\OAuth\Server;

use fkooman\OAuth\Server\Exception\InsufficientScopeException;

class Scope
{
    private string $scope;

    public function __construct(string $scope)
    {
        $this->scope = $scope;
    }

    public function __toString(): string
    {
        return $this->scope;
    }

    /**
     * @param array<string> $scopeList
     *
     * @throws \fkooman\OAuth\Server\Exception\InsufficientScopeException
     */
    public function requireAll(array $scopeList): void
    {
        $grantedList = explode(' ', $this->scope);
        foreach ($scopeList as $scope) {
            if (!\in_array($scope, $grantedList, true)) {
                throw new InsufficientScopeException(sprintf('scope "%s" not granted', $scope));
            }
        }
    }

    public function has(string $scope): bool
    {
        return in_array($scope, explode(' ', $this->scope));
    }

    /**
     * @param array<string> $scopeList
     */
    public function hasAny(array $scopeList): bool
    {
        $grantedList = explode(' ', $this->scope);
        foreach ($scopeList as $scope) {
            if (\in_array($scope, $grantedList, true)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param array<string> $scopeList
     *
     * @throws \fkooman\OAuth\Server\Exception\InsufficientScopeException
     */
    public function requireAny(array $scopeList): void
    {
        if (!$this->hasAny($scopeList)) {
            throw new InsufficientScopeException(sprintf('not any of scopes "%s" granted', implode(' ', $scopeList)));
        }
    }
}
