**Summary**: Very Simple RADIUS Client

**Description**: Very Simple RADIUS Client that supports only PAP.

**License**: MIT

# Introduction

We wrote a very simple RADIUS client that supports only PAP for password 
authentication. We needed this in order to support RADIUS authentication with
various software projects written in PHP.

It aims at implementing just the part of the specification that is required in
order to make RADIUS authentication work for most use case.

We tested this library with:

* OpenBSD [radiusd](https://man.openbsd.org/radiusd.8)

# Why

Due to lack of (proper) support for PHP 8 (and OpenSSL 3.0) we were not 
satisfied with the available options. We investigated:

* [php-pecl-radius](https://pecl.php.net/package/radius)
* [dapphp/radius](https://github.com/dapphp/radius)

We wanted something simpler that would just be used to verify credentials and
(optionally) return some attributes.

# Features

* Supports multiple RADIUS servers
* Always adds and verifies the 
  [Message-Authenticator](https://www.rfc-editor.org/rfc/rfc2869#section-5.14) 
  attribute
* Supports connections to RADIUS servers over IPv4 and IPv6
* Simple API

# API

```php
<?php

require_once 'example/autoload.php';

try {
    $radiusAuth = new fkooman\Radius\RadiusClient('my-nas-id');
    $radiusAuth->addServer('[fdd4:56b7:5ebc:4e82::1]', 's3cr3t', 1812);
    $radiusAuth->addServer('10.253.109.1', 's3cr3t', 1812);

    $accessResponse = $radiusAuth->accessRequest('foo', 'aaaaaaaaaaaaaaaaaaaa');

    // print the full response
    echo $accessResponse . PHP_EOL;

    // to test/get one attribute by name
    if (null !== $attributeValue = $accessResponse->attributeByName('Reply-Message')) {
        echo 'Reply-Message: ' . $attributeValue . PHP_EOL;
    }

    // to test/get one attribute by type
    if (null !== $attributeValue = $accessResponse->attributeByType(18)) {
        echo 'Reply-Message: ' . $attributeValue . PHP_EOL;
    }
} catch (fkooman\Radius\Exception\AuthException $e) {
    // Authentication Failed
    echo $e->accessResponse() . PHP_EOL;
    exit(1);
}
```

# Logging

TODO.
See `example/client.php` for now.

# Testing

If you'd like to test this library against your own RADIUS infrastructure, you
can using the included example client. Make sure you have PHP installed, more
specifically you can run `php` from the command line.

```bash
$ git clone https://codeberg.org/fkooman/php-radius
$ cd php-radius
```

In order to run the tests:

```bash
$ SERVER_ADDRESS=10.253.109.1 SERVER_PORT=1812 NAS_ID=my-nas-id SHARED_SECRET=s3cr3t php example/client.php foo aaaaaaaaaaaaaaaaaaaa
```

Example of possible output:

```
I Access-Request for User-Name "foo"
D RADIUS Server: udp://10.253.109.1:1812
D Request Authenticator: f80c8f151401eec175624836aef9e0a2
D --> 01000058f80c8f151401eec175624836aef9e0a20105666f6f0222ae6e6e5107ffaceca385d1f0d4909f8ef7427b5516e3fe2b5cbbadb21da24b48200b6d792d6e61732d696450127531f94adb78ed239d7569e11c49ffe6
D <-- 02000040e201e5a01da5d8210df230238bb82b2a121a41757468656e7469636174696f6e207375636365656465645012c5c362effd009e2011a877accf78815b
I Access-Accept for User-Name "foo"
***OK***
Code: Access-Accept (2)
Attributes:
	Reply-Message (18) => Authentication succeeded
	Message-Authenticator (80) => c5c362effd009e2011a877accf78815b
```

When the credentials are not correct:

```bash
$ SERVER_ADDRESS=10.253.109.1 SERVER_PORT=1812 NAS_ID=my-nas-id SHARED_SECRET=s3cr3t php example/client.php foo bar
```

Example of possible output:

```
I Access-Request for User-Name "foo"
D RADIUS Server: udp://10.253.109.1:1812
D Request Authenticator: 45e8483f59ee54e3aaf92f470f9faef5
D --> 0100004845e8483f59ee54e3aaf92f470f9faef50105666f6f0212fb6326a35644751f605143f3a69623f7200b6d792d6e61732d69645012d5e7eabe0143afdcadde7d7f80db0b38
D <-- 0300003dc19c1f2e65790b319cfdb49d5bdaf381121741757468656e7469636174696f6e206661696c65645012cd617972848d8e179988c8b4b40d9dce
I Access-Reject for User-Name "foo"
***FAIL***
Code: Access-Reject (3)
Attributes:
	Reply-Message (18) => Authentication failed
	Message-Authenticator (80) => cd617972848d8e179988c8b4b40d9dce
```

**NOTE**: in the output above, attribute values that are NOT of type `text` are
hex encoded to avoid displaying binary data in the terminal. See the list of 
types per attribute 
[here](https://www.iana.org/assignments/radius-types/radius-types.xhtml).

**NOTE**: do NOT use REAL credentials when providing us with the debug output, 
but create a test account and a (temporary) test shared secret!

**NOTE**: please [report](https://codeberg.org/fkooman/php-radius/issues/new) 
any issues you find when testing with your RADIUS server!
