/*
 * Decompiled with CFR 0.152.
 */
package net.sf.freecol.server;

import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.function.Predicate;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import javax.xml.stream.XMLStreamException;
import net.sf.freecol.FreeCol;
import net.sf.freecol.common.FreeColException;
import net.sf.freecol.common.FreeColSeed;
import net.sf.freecol.common.debug.FreeColDebugger;
import net.sf.freecol.common.io.FreeColDirectories;
import net.sf.freecol.common.io.FreeColSavegameFile;
import net.sf.freecol.common.io.FreeColXMLReader;
import net.sf.freecol.common.io.FreeColXMLWriter;
import net.sf.freecol.common.metaserver.MetaServerUtils;
import net.sf.freecol.common.metaserver.ServerInfo;
import net.sf.freecol.common.model.Constants;
import net.sf.freecol.common.model.Game;
import net.sf.freecol.common.model.IndianSettlement;
import net.sf.freecol.common.model.Map;
import net.sf.freecol.common.model.Nation;
import net.sf.freecol.common.model.NationOptions;
import net.sf.freecol.common.model.Player;
import net.sf.freecol.common.model.Specification;
import net.sf.freecol.common.model.Stance;
import net.sf.freecol.common.model.Tension;
import net.sf.freecol.common.model.Unit;
import net.sf.freecol.common.networking.ChangeSet;
import net.sf.freecol.common.networking.Connection;
import net.sf.freecol.common.networking.GameStateMessage;
import net.sf.freecol.common.networking.LogoutMessage;
import net.sf.freecol.common.networking.Message;
import net.sf.freecol.common.networking.TrivialMessage;
import net.sf.freecol.common.networking.VacantPlayersMessage;
import net.sf.freecol.common.option.OptionGroup;
import net.sf.freecol.common.util.CollectionUtils;
import net.sf.freecol.common.util.LogBuilder;
import net.sf.freecol.common.util.Utils;
import net.sf.freecol.server.ai.AIInGameInputHandler;
import net.sf.freecol.server.ai.AIMain;
import net.sf.freecol.server.ai.AIPlayer;
import net.sf.freecol.server.control.Controller;
import net.sf.freecol.server.control.InGameController;
import net.sf.freecol.server.control.PreGameController;
import net.sf.freecol.server.control.ServerInputHandler;
import net.sf.freecol.server.control.UserConnectionHandler;
import net.sf.freecol.server.generator.MapGenerator;
import net.sf.freecol.server.generator.SimpleMapGenerator;
import net.sf.freecol.server.model.ServerGame;
import net.sf.freecol.server.model.ServerPlayer;
import net.sf.freecol.server.model.Session;
import net.sf.freecol.server.networking.DummyConnection;
import net.sf.freecol.server.networking.Server;

public final class FreeColServer {
    private static final Logger logger = Logger.getLogger(FreeColServer.class.getName());
    public static final String MAP_EDITOR_NAME = "mapEditor";
    public static final String ACTIVE_UNIT_TAG = "activeUnit";
    public static final String OLD_SERVER_OBJECTS_TAG = "serverObjects";
    public static final String DEBUG_TAG = "debug";
    public static final String RANDOM_STATE_TAG = "randomState";
    public static final String OWNER_TAG = "owner";
    public static final String PUBLIC_SERVER_TAG = "publicServer";
    public static final String SAVED_GAME_TAG = "savedGame";
    public static final String SINGLE_PLAYER_TAG = "singleplayer";
    public static final int SAVEGAME_VERSION = 14;
    public static final int MINIMUM_SAVEGAME_VERSION = 12;
    public static final String DEFAULT_SPEC = "freecol";
    private String name;
    private boolean publicServer = false;
    private boolean singlePlayer;
    private Random random = null;
    private ServerGame serverGame;
    private Server server;
    private ServerState serverState = ServerState.PRE_GAME;
    private final UserConnectionHandler userConnectionHandler;
    private final PreGameController preGameController;
    private final InGameController inGameController;
    private final ServerInputHandler inputHandler;
    private AIMain aiMain;
    private MapGenerator mapGenerator = null;
    private Constants.IntegrityType integrity = Constants.IntegrityType.INTEGRITY_GOOD;

    private FreeColServer(String name, InetAddress address, int port) throws IOException {
        this.name = name;
        this.server = this.createServer(address, port);
        this.server.start();
        this.userConnectionHandler = new UserConnectionHandler(this);
        this.preGameController = new PreGameController(this);
        this.inGameController = new InGameController(this);
        this.inputHandler = new ServerInputHandler(this);
    }

    private Server createServer(InetAddress address, int firstPort) throws IOException {
        ArrayList<InetAddress> addresses = new ArrayList<InetAddress>();
        if (address == null) {
            addresses.add(InetAddress.getLoopbackAddress());
            try {
                addresses.add(InetAddress.getLocalHost());
            }
            catch (UnknownHostException uhe) {
                logger.warning("Could not resolve local host name");
            }
        } else {
            addresses.add(address);
        }
        int tryMax = 1;
        if (firstPort < 0) {
            firstPort = FreeCol.getServerPort();
            tryMax = 10;
        }
        IOException ex = null;
        for (InetAddress ia : addresses) {
            String host = ia.getHostAddress();
            int port = firstPort;
            for (int i = tryMax; i > 0; --i) {
                try {
                    Server ret = new Server(this, host, port);
                    logger.finest("Server started: " + ret.getHost() + ", " + ret.getPort());
                    return ret;
                }
                catch (IOException ioe) {
                    ex = ioe;
                    ++port;
                    continue;
                }
            }
        }
        throw ex;
    }

    public FreeColServer(boolean publicServer, boolean singlePlayer, Specification specification, InetAddress address, int port, String name) throws IOException {
        this(name, address, port);
        this.setPublicServer(publicServer);
        this.singlePlayer = singlePlayer;
        this.random = new Random(FreeColSeed.getFreeColSeed());
        this.serverGame = new ServerGame(specification, this.random);
        this.inGameController.setRandom(this.random);
        this.mapGenerator = new SimpleMapGenerator(this.random);
    }

    public FreeColServer(FreeColSavegameFile savegame, Specification specification, InetAddress address, int port, String name) throws FreeColException, IOException, XMLStreamException {
        this(name, address, port);
        this.serverGame = this.loadGame(savegame, specification);
        Session.clearAll();
        if (FreeColSeed.hasFreeColSeed()) {
            this.random = new Random(FreeColSeed.getFreeColSeed());
        }
        this.inGameController.setRandom(this.random);
        this.mapGenerator = null;
    }

    public String getName() {
        return this.name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public synchronized boolean getPublicServer() {
        return this.publicServer;
    }

    public synchronized void setPublicServer(boolean publicServer) {
        this.publicServer = publicServer;
    }

    public boolean getSinglePlayer() {
        return this.singlePlayer;
    }

    public void setSinglePlayer(boolean singlePlayer) {
        this.singlePlayer = singlePlayer;
    }

    public Random getServerRandom() {
        return this.random;
    }

    public void setServerRandom(Random random) {
        this.random = random;
    }

    public ServerGame getGame() {
        return this.serverGame;
    }

    public void setGame(ServerGame serverGame) {
        this.serverGame = serverGame;
    }

    public Specification getSpecification() {
        return this.serverGame == null ? null : this.serverGame.getSpecification();
    }

    public Server getServer() {
        return this.server;
    }

    public String getHost() {
        return this.server == null ? null : this.server.getHost();
    }

    public int getPort() {
        return this.server == null ? -1 : this.server.getPort();
    }

    public void shutdown() {
        this.server.shutdown();
    }

    public ServerState getServerState() {
        return this.serverState;
    }

    private ServerState changeServerState(ServerState serverState) {
        ServerState ret = this.serverState;
        this.serverState = serverState;
        switch (this.serverState) {
            case PRE_GAME: 
            case LOAD_GAME: 
            case IN_GAME: {
                this.getServer().setMessageHandlerToAllConnections(this.inputHandler);
                break;
            }
            default: {
                this.getServer().setMessageHandlerToAllConnections(null);
            }
        }
        return ret;
    }

    public UserConnectionHandler getUserConnectionHandler() {
        return this.userConnectionHandler;
    }

    public Controller getController() {
        return this.getServerState() == ServerState.IN_GAME ? this.inGameController : this.preGameController;
    }

    public ServerInputHandler getInputHandler() {
        switch (this.serverState) {
            case PRE_GAME: 
            case LOAD_GAME: 
            case IN_GAME: {
                return this.inputHandler;
            }
        }
        return null;
    }

    public PreGameController getPreGameController() {
        return this.preGameController;
    }

    public InGameController getInGameController() {
        return this.inGameController;
    }

    public void setAIMain(AIMain aiMain) {
        this.aiMain = aiMain;
    }

    public AIMain getAIMain() {
        return this.aiMain;
    }

    public MapGenerator getMapGenerator() {
        return this.mapGenerator;
    }

    public void setMapGenerator(MapGenerator mapGenerator) {
        this.mapGenerator = mapGenerator;
    }

    public Constants.IntegrityType getIntegrity() {
        return this.integrity;
    }

    public void endGame() {
        this.changeServerState(ServerState.END_GAME);
        for (Player p : this.getGame().getLiveEuropeanPlayerList(new Player[0])) {
            if (p.isAdmin()) continue;
            p.send(new ChangeSet().add(ChangeSet.See.only(p), new LogoutMessage(p, Game.LogoutReason.QUIT)));
        }
    }

    public void addNewUserConnection(Socket socket) throws FreeColException, IOException, XMLStreamException {
        String name = socket.getInetAddress() + ":" + socket.getPort();
        Connection c = new Connection(socket, "FreeColServer:" + name).setMessageHandler(this.userConnectionHandler);
        this.getServer().addConnection(c);
        c.startReceiving();
        c.send(new GameStateMessage(this.serverState));
        if (this.serverState == ServerState.IN_GAME) {
            c.send(new VacantPlayersMessage().setVacantPlayers(this.getGame()));
        }
        logger.info("Client connected from " + name);
    }

    public void addPlayerConnection(Connection connection) {
        switch (this.serverState) {
            case PRE_GAME: 
            case LOAD_GAME: 
            case IN_GAME: {
                connection.setMessageHandler(this.inputHandler);
                break;
            }
            default: {
                return;
            }
        }
        this.getServer().addConnection(connection);
        this.updateMetaServer();
    }

    public void removePlayerConnection(Player player) {
        Connection conn = player.getConnection();
        if (conn != null) {
            this.getServer().removeConnection(conn);
            conn.close();
            player.setConnection(null);
        }
    }

    private void addAIConnection(Player aiPlayer) {
        DummyConnection theConnection = new DummyConnection("Server-to-AI-" + aiPlayer.getSuffix());
        theConnection.setMessageHandler(this.inputHandler);
        theConnection.setWriteScope(FreeColXMLWriter.WriteScope.toClient(aiPlayer));
        DummyConnection aiConnection = new DummyConnection("AI-" + aiPlayer.getSuffix() + "-to-Server");
        aiConnection.setMessageHandler(new AIInGameInputHandler(this, aiPlayer, this.getAIMain()));
        aiConnection.setWriteScope(FreeColXMLWriter.WriteScope.toServer());
        aiConnection.setOtherConnection(theConnection);
        theConnection.setOtherConnection(aiConnection);
        aiPlayer.setConnection(theConnection);
        this.getServer().addDummyConnection(theConnection);
    }

    public ServerGame waitForGame() {
        int timeStep = 1000;
        int timeOut = 20000;
        ServerGame serverGame = null;
        while ((serverGame = this.getGame()) == null) {
            Utils.delay(1000L, "waitForGame delay interrupt");
            if ((timeOut -= 1000) > 0) continue;
            break;
        }
        return serverGame;
    }

    public void startGame() throws FreeColException {
        logger.info("Server starting game: " + this.serverState);
        switch (this.serverState) {
            case PRE_GAME: {
                Game game = this.buildGame();
                for (Player player : CollectionUtils.transform(game.getLivePlayers(new Player[0]), p -> !p.isAI())) {
                    player.invalidateCanSeeTiles();
                    ChangeSet cs = new ChangeSet();
                    cs.add(ChangeSet.See.only(player), game);
                    player.send(cs);
                }
                break;
            }
            case LOAD_GAME: {
                break;
            }
            default: {
                logger.warning("Invalid startGame when server state = " + this.serverState);
                return;
            }
        }
        this.changeServerState(ServerState.IN_GAME);
        this.sendToAll((Message)TrivialMessage.startGameMessage, (Player)null);
        this.updateMetaServer();
    }

    private void sendToAll(Message msg, Connection conn) {
        this.getServer().sendToAll(msg, conn);
    }

    public void sendToAll(Message msg, Player player) {
        this.sendToAll(msg, player == null ? null : player.getConnection());
    }

    public void saveMapEditorGame(File file, BufferedImage image) throws IOException {
        this.setAIMain(null);
        Specification spec = this.getSpecification();
        this.getGame().setSpecification(null);
        this.saveGame(file, MAP_EDITOR_NAME, null, null, image);
        this.getGame().setSpecification(spec);
    }

    public void saveGame(File file, OptionGroup options, Unit active) throws IOException {
        this.saveGame(file, null, options, active, null);
    }

    private void saveGame(File file, String owner, OptionGroup options, Unit active, BufferedImage image) throws IOException {
        Utils.garbageCollect();
        try (JarOutputStream fos = new JarOutputStream(Files.newOutputStream(file.toPath(), new OpenOption[0]));){
            if (image != null) {
                fos.putNextEntry(new JarEntry("thumbnail.png"));
                ImageIO.write((RenderedImage)image, "png", fos);
                fos.closeEntry();
            }
            if (options != null) {
                fos.putNextEntry(new JarEntry("client-options.xml"));
                options.save(fos, null, true);
                fos.closeEntry();
            }
            Properties properties = new Properties();
            properties.setProperty("map.width", Integer.toString(this.serverGame.getMap().getWidth()));
            properties.setProperty("map.height", Integer.toString(this.serverGame.getMap().getHeight()));
            fos.putNextEntry(new JarEntry("savegame.properties"));
            properties.store(fos, null);
            fos.closeEntry();
            fos.putNextEntry(new JarEntry("savegame.xml"));
            try (FreeColXMLWriter xw = new FreeColXMLWriter(fos, FreeColXMLWriter.WriteScope.toSave(), false);){
                xw.writeStartDocument("UTF-8", "1.0");
                xw.writeComment(FreeCol.getConfiguration().toString());
                xw.writeCharacters("\n");
                xw.writeStartElement(SAVED_GAME_TAG);
                xw.writeAttribute(OWNER_TAG, owner != null ? owner : FreeCol.getName());
                xw.writeAttribute(PUBLIC_SERVER_TAG, this.getPublicServer());
                xw.writeAttribute(SINGLE_PLAYER_TAG, this.singlePlayer);
                xw.writeAttribute("version", 14);
                xw.writeAttribute(RANDOM_STATE_TAG, Utils.getRandomState(this.random));
                xw.writeAttribute(DEBUG_TAG, FreeColDebugger.getDebugModes());
                if (active != null) {
                    this.serverGame.setInitialActiveUnitId(active.getId());
                }
                this.serverGame.toXML(xw);
                if (this.aiMain != null) {
                    this.aiMain.toXML(xw);
                }
                xw.writeEndElement();
                xw.writeEndDocument();
            }
            fos.closeEntry();
        }
        catch (XMLStreamException e) {
            throw new IOException("Failed to save (XML): " + file.getName(), e);
        }
    }

    public ServerGame loadGame(FreeColSavegameFile fis) throws IOException, FreeColException, XMLStreamException {
        return this.loadGame(fis, null);
    }

    public static Map readMap(File file, Specification spec) throws FreeColException, IOException, XMLStreamException {
        ServerGame serverGame = FreeColServer.readGame(file, spec, null);
        return serverGame == null ? null : serverGame.getMap();
    }

    private static ServerGame readGame(File file, Specification spec, FreeColServer freeColServer) throws FreeColException, IOException, XMLStreamException {
        ServerGame serverGame = FreeColServer.readGame(new FreeColSavegameFile(file), spec, freeColServer);
        logger.info("Read file " + file.getPath());
        File startGame = FreeColDirectories.getStartMapFile();
        if (startGame != null && startGame.getPath().equals(file.getPath())) {
            Utils.deleteFile(file);
        }
        return serverGame;
    }

    public static ServerGame readGame(FreeColSavegameFile fis, Specification specification, FreeColServer freeColServer) throws FreeColException, IOException, XMLStreamException {
        int savegameVersion = fis.getSavegameVersion();
        logger.info("Found savegame version " + savegameVersion);
        if (savegameVersion < 12) {
            throw new FreeColException("server.incompatibleVersions: " + savegameVersion + " < 12");
        }
        Game serverGame = null;
        try (FreeColXMLReader xr = fis.getSavedGameFreeColXMLReader();){
            xr.setReadScope(FreeColXMLReader.ReadScope.SERVER);
            String active = null;
            xr.nextTag();
            if (freeColServer != null) {
                String owner = xr.getAttribute(OWNER_TAG, null);
                if (specification == null && MAP_EDITOR_NAME.equals(owner)) {
                    throw new FreeColException("Can not start a map editor map as a game: " + fis.getPath());
                }
                freeColServer.setSinglePlayer(xr.getAttribute(SINGLE_PLAYER_TAG, true));
                freeColServer.setPublicServer(xr.getAttribute(PUBLIC_SERVER_TAG, false));
                String r = xr.getAttribute(RANDOM_STATE_TAG, null);
                freeColServer.setServerRandom(Utils.restoreRandomState(r));
                FreeColDebugger.setDebugModes(xr.getAttribute(DEBUG_TAG, null));
                active = xr.getAttribute(ACTIVE_UNIT_TAG, null);
            }
            while (xr.moreTags()) {
                String tag = xr.getLocalName();
                if (OLD_SERVER_OBJECTS_TAG.equals(tag)) {
                    xr.swallowTag(OLD_SERVER_OBJECTS_TAG);
                    continue;
                }
                if ("game".equals(tag)) {
                    serverGame = new ServerGame(specification, xr);
                    serverGame.setCurrentPlayer(null);
                    if (freeColServer == null) continue;
                    freeColServer.setGame((ServerGame)serverGame);
                    continue;
                }
                if ("aiMain".equals(tag)) {
                    if (freeColServer != null) {
                        AIMain aiMain = new AIMain(freeColServer, xr);
                        freeColServer.setAIMain(aiMain);
                        continue;
                    }
                    xr.swallowTag("aiMain");
                    continue;
                }
                throw new XMLStreamException("Unknown tag reading server game: " + tag);
            }
            if (serverGame != null && active != null) {
                serverGame.setInitialActiveUnitId(active);
            }
        }
        return serverGame;
    }

    private ServerGame loadGame(FreeColSavegameFile fis, Specification specification) throws FreeColException, IOException, XMLStreamException {
        this.changeServerState(ServerState.LOAD_GAME);
        ServerGame serverGame = FreeColServer.readGame(fis, specification, this);
        LogBuilder lb = new LogBuilder(512);
        this.integrity = serverGame.checkIntegrity(true, lb);
        switch (this.integrity) {
            case INTEGRITY_GOOD: {
                logger.info("Game integrity test succeeded.");
                break;
            }
            case INTEGRITY_FIXED: {
                logger.info("Game integrity test failed, but fixed." + lb);
                break;
            }
            default: {
                logger.warning("Game integrity test failed." + lb);
            }
        }
        serverGame.getMap().resetContiguity();
        serverGame.establishUnknownEnemy();
        specification = this.getSpecification();
        specification.disableEditing();
        AIMain aiMain = this.getAIMain();
        lb.truncate(0);
        Constants.IntegrityType aiIntegrity = Constants.IntegrityType.INTEGRITY_GOOD;
        if (aiMain == null) {
            aiIntegrity = aiIntegrity.fail();
            lb.add("\n  AIMain missing.");
        } else {
            aiIntegrity = aiMain.checkIntegrity(true, lb);
        }
        switch (aiIntegrity) {
            case INTEGRITY_GOOD: {
                logger.info("AI integrity test succeeded.");
                break;
            }
            case INTEGRITY_FIXED: {
                logger.info("AI integrity test failed, but fixed." + lb);
                break;
            }
            default: {
                aiMain = new AIMain(this);
                aiMain.findNewObjects(true);
                this.setAIMain(aiMain);
                logger.warning("AI integrity test failed, replaced AIMain." + lb);
            }
        }
        serverGame.setFreeColGameObjectListener(aiMain);
        serverGame.sortPlayers(Player.playerComparator);
        for (Player player : serverGame.getLivePlayerList(new Player[0])) {
            if (player.isAI()) {
                this.addAIConnection(player);
            }
            if (!player.isEuropean()) continue;
            player.canSee(serverGame.getMap().getTile(0, 0));
        }
        return serverGame;
    }

    private Game buildGame() throws FreeColException {
        ServerGame serverGame = this.getGame();
        Specification spec = serverGame.getSpecification();
        if (this.getAIMain() == null) {
            AIMain aiMain = new AIMain(this);
            serverGame.setFreeColGameObjectListener(aiMain);
            this.setAIMain(aiMain);
        }
        serverGame.establishUnknownEnemy();
        Predicate<Map.Entry> availablePred = e -> e.getValue() != NationOptions.NationState.NOT_AVAILABLE && serverGame.getPlayerByNationId(((Nation)e.getKey()).getId()) == null;
        List<Player> toUpdate = CollectionUtils.transform(serverGame.getNationOptions().getNations().entrySet(), availablePred, e -> this.makeAIPlayer((Nation)e.getKey()), Player.playerComparator);
        toUpdate.add((ServerPlayer)serverGame.getUnknownEnemy());
        serverGame.updatePlayers(toUpdate);
        if (serverGame.getMap() == null) {
            this.generateMap(true);
            spec.generateDynamicOptions();
            Random random = this.getServerRandom();
            for (Player player : serverGame.getLivePlayerList(new Player[0])) {
                ((ServerPlayer)player).randomizeGame(random);
                if (!player.isIndian()) continue;
                int alarm = (Tension.Level.HAPPY.getLimit() + Tension.Level.CONTENT.getLimit()) / 2;
                for (Player other : serverGame.getLiveNativePlayerList(player)) {
                    player.setStance(other, Stance.PEACE);
                    for (IndianSettlement is : player.getIndianSettlementList()) {
                        is.setAlarm(other, new Tension(alarm));
                    }
                }
            }
        }
        spec.getMapGeneratorOptions().setEditable(false);
        spec.getGameOptions().setEditable(false);
        spec.getOptionGroup("difficultyLevels").setEditable(false);
        this.getAIMain().findNewObjects(true);
        return serverGame;
    }

    public Map generateEmptyMap(int width, int height) {
        return this.getMapGenerator().generateEmptyMap(this.getGame(), width, height, new LogBuilder(-1));
    }

    public Map generateMap(boolean generateEuropeanPlayerUnits) {
        ServerGame serverGame = this.getGame();
        LogBuilder lb = new LogBuilder(256);
        File importFile = serverGame.getMapGeneratorOptions().getFile("model.option.importFile");
        Map importMap = null;
        if (importFile != null) {
            try {
                importMap = FreeColServer.readMap(importFile, serverGame.getSpecification());
            }
            catch (IOException | XMLStreamException | FreeColException ex) {
                logger.log(Level.WARNING, "Failed to import map: " + importFile.getName(), ex);
            }
        }
        Map ret = this.getMapGenerator().generateMap(serverGame, importMap, generateEuropeanPlayerUnits, lb);
        lb.shrink("\n");
        lb.log(logger, Level.FINER);
        return ret;
    }

    public ServerPlayer makeAIPlayer(Nation nation) {
        ServerPlayer aiPlayer = new ServerPlayer(this.getGame(), false, nation);
        aiPlayer.setAI(true);
        aiPlayer.setReady(true);
        this.addAIConnection(aiPlayer);
        this.getGame().addPlayer(aiPlayer);
        this.getAIMain().setFreeColGameObject(aiPlayer.getId(), aiPlayer);
        return aiPlayer;
    }

    public void exploreMapForAllPlayers(boolean reveal) {
        Specification spec = this.getSpecification();
        for (Player p : this.getGame().getLiveEuropeanPlayerList(new Player[0])) {
            ((ServerPlayer)p).exploreMap(reveal);
        }
        if (reveal) {
            FreeColDebugger.setNormalGameFogOfWar(spec.getBoolean("model.option.fogOfWar"));
            spec.setBoolean("model.option.fogOfWar", false);
        } else {
            spec.setBoolean("model.option.fogOfWar", FreeColDebugger.getNormalGameFogOfWar());
        }
        for (Player p : this.getGame().getLiveEuropeanPlayerList(new Player[0])) {
            p.getConnection().sendReconnect();
        }
    }

    public ServerPlayer getPlayer(Connection conn) {
        Predicate<Player> connPred = CollectionUtils.matchKeyEquals(conn, Player::getConnection);
        return (ServerPlayer)this.getGame().getPlayer(connPred);
    }

    public AIPlayer getAIPlayer(Player player) {
        return this.getAIMain().getAIPlayer(player);
    }

    private ServerInfo getServerInfo() {
        Predicate<Player> absentAI = p -> !p.isREF() && p.isAI() && !p.isConnected();
        Predicate<Player> liveHuman = p -> !p.isAI() && p.isConnected();
        int slots = CollectionUtils.count(this.getGame().getLiveEuropeanPlayers(new Player[0]), absentAI);
        int players = CollectionUtils.count(this.getGame().getLiveEuropeanPlayers(new Player[0]), liveHuman);
        return new ServerInfo(this.getName(), this.getHost(), this.getPort(), slots, players, this.serverState == ServerState.IN_GAME, FreeCol.getVersion(), this.getServerState().getMetaServerState());
    }

    public synchronized boolean registerWithMetaServer() {
        if (!this.publicServer) {
            return true;
        }
        return MetaServerUtils.registerServer(this.getServerInfo());
    }

    public synchronized boolean removeFromMetaServer() {
        if (!this.publicServer) {
            return false;
        }
        boolean ret = MetaServerUtils.removeServer(this.getServerInfo());
        return ret;
    }

    public synchronized boolean updateMetaServer() {
        if (!this.publicServer) {
            return false;
        }
        boolean ret = MetaServerUtils.updateServer(this.getServerInfo());
        return ret;
    }

    public static enum ServerState {
        PRE_GAME(0),
        LOAD_GAME(0),
        IN_GAME(1),
        END_GAME(2);

        private int metaServerState;

        private ServerState(int metaServerState) {
            this.metaServerState = metaServerState;
        }

        public int getMetaServerState() {
            return this.metaServerState;
        }
    }
}

