<?php declare(strict_types = 0);
/*
** Zabbix
** Copyright (C) 2001-2024 Zabbix SIA
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/


namespace Widgets\TrigOver\Includes;

use CCol,
	CLink,
	CMenuPopupHelper,
	CSettingsHelper,
	CSeverityHelper,
	CSpan,
	CTableInfo,
	CUrl;

class ViewHelper {

	/**
	 * Creates and returns a trigger status cell for the trigger overview table.
	 *
	 * @param array $trigger
	 * @param array $dependencies
	 *
	 * @return CCol
	 */
	public static function getTriggerOverviewCell(array $trigger, array $dependencies): CCol {
		$ack = $trigger['problem']['acknowledged'] == 1 ? (new CSpan())->addClass(ZBX_STYLE_ICON_ACKN) : null;
		$desc = array_key_exists($trigger['triggerid'], $dependencies)
			? self::makeTriggerDependencies($dependencies[$trigger['triggerid']])
			: [];

		$column = (new CCol([$desc, $ack]))
			->addClass(CSeverityHelper::getStyle((int) $trigger['priority'], $trigger['value'] == TRIGGER_VALUE_TRUE))
			->addClass(ZBX_STYLE_CURSOR_POINTER);

		$eventid = 0;
		$blink_period = timeUnitToSeconds(CSettingsHelper::get(CSettingsHelper::BLINK_PERIOD));
		$duration = time() - $trigger['lastchange'];

		if ($blink_period > 0 && $duration < $blink_period) {
			$column->addClass('blink');
			$column->setAttribute('data-time-to-blink', $blink_period - $duration);
			$column->setAttribute('data-toggle-class', ZBX_STYLE_BLINK_HIDDEN);
		}

		if ($trigger['value'] == TRIGGER_VALUE_TRUE) {
			$eventid = $trigger['problem']['eventid'];
			$show_update_problem = true;
		}
		else {
			$show_update_problem = false;
		}

		$column->setMenuPopup(CMenuPopupHelper::getTrigger([
			'triggerid' => $trigger['triggerid'],
			'backurl' => (new CUrl('zabbix.php'))
				->setArgument('action', 'dashboard.view')
				->getUrl(),
			'eventid' => $eventid,
			'show_update_problem' => $show_update_problem
		]));

		return $column;
	}

	/**
	 * Returns icons with tooltips for triggers with dependencies.
	 *
	 * @param array $dependencies
	 *        array $dependencies['up']    (optional) The list of "Dependent" triggers.
	 *        array $dependencies['down']  (optional) The list of "Depends on" triggers.
	 *
	 * @return array
	 */
	public static function makeTriggerDependencies(array $dependencies): array {
		$result = [];

		foreach (['down', 'up'] as $type) {
			if (array_key_exists($type, $dependencies)) {
				$header = $type === 'down' ? _('Depends on') : _('Dependent');
				$class = $type === 'down' ? ZBX_STYLE_ICON_DEPEND_DOWN : ZBX_STYLE_ICON_DEPEND_UP;

				$table = (new CTableInfo())
					->setAttribute('style', 'max-width: '.ZBX_TEXTAREA_STANDARD_WIDTH.'px;')
					->setHeader([$header]);

				foreach ($dependencies[$type] as $description) {
					$table->addRow($description);
				}

				$result[] = (new CLink())
					->addClass($class)
					->addClass(ZBX_STYLE_CURSOR_POINTER)
					->setHint($table, '', false);
			}
		}

		return $result;
	}
}
