"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultipleAuthentication = void 0;

var _authentication_type = require("../authentication_type");

var _common = require("../../../../common");

var _next_url = require("../../../utils/next_url");

var _routes = require("./routes");

var _types = require("../../types");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   *   Copyright OpenSearch Contributors
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   Licensed under the Apache License, Version 2.0 (the "License").
                                                                                                                                                                                                                   *   You may not use this file except in compliance with the License.
                                                                                                                                                                                                                   *   A copy of the License is located at
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *       http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   or in the "license" file accompanying this file. This file is distributed
                                                                                                                                                                                                                   *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
                                                                                                                                                                                                                   *   express or implied. See the License for the specific language governing
                                                                                                                                                                                                                   *   permissions and limitations under the License.
                                                                                                                                                                                                                   */

class MultipleAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);

    _defineProperty(this, "authTypes", void 0);

    _defineProperty(this, "authHandlers", void 0);

    this.authTypes = this.config.auth.type;
    this.authHandlers = new Map();
  }

  async init() {
    const routes = new _routes.MultiAuthRoutes(this.router, this.sessionStorageFactory);
    routes.setupRoutes();

    for (let i = 0; i < this.authTypes.length; i++) {
      switch (this.authTypes[i].toLowerCase()) {
        case _common.AuthType.BASIC:
          {
            const BasicAuth = new _types.BasicAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await BasicAuth.init();
            this.authHandlers.set(_common.AuthType.BASIC, BasicAuth);
            break;
          }

        case _common.AuthType.OPEN_ID:
          {
            const OidcAuth = new _types.OpenIdAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await OidcAuth.init();
            this.authHandlers.set(_common.AuthType.OPEN_ID, OidcAuth);
            break;
          }

        case _common.AuthType.SAML:
          {
            const SamlAuth = new _types.SamlAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await SamlAuth.init();
            this.authHandlers.set(_common.AuthType.SAML, SamlAuth);
            break;
          }

        default:
          {
            throw new Error(`Unsupported authentication type: ${this.authTypes[i]}`);
          }
      }
    }
  } // override functions inherited from AuthenticationType


  requestIncludesAuthInfo(request) {
    for (const key of this.authHandlers.keys()) {
      if (this.authHandlers.get(key).requestIncludesAuthInfo(request)) {
        return true;
      }
    }

    return false;
  }

  async getAdditionalAuthHeader(request) {
    var _cookie$authType;

    // To Do: refactor this method to improve the effiency to get cookie, get cookie from input parameter
    const cookie = await this.sessionStorageFactory.asScoped(request).get();
    const reqAuthType = cookie === null || cookie === void 0 ? void 0 : (_cookie$authType = cookie.authType) === null || _cookie$authType === void 0 ? void 0 : _cookie$authType.toLowerCase();

    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return await this.authHandlers.get(reqAuthType).getAdditionalAuthHeader(request);
    } else {
      return {};
    }
  }

  getCookie(request, authInfo) {
    return {};
  }

  async isValidCookie(cookie, request) {
    var _cookie$authType2;

    const reqAuthType = cookie === null || cookie === void 0 ? void 0 : (_cookie$authType2 = cookie.authType) === null || _cookie$authType2 === void 0 ? void 0 : _cookie$authType2.toLowerCase();

    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).isValidCookie(cookie, request);
    } else {
      return false;
    }
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      const nextUrlParam = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);

      if (this.config.auth.anonymous_auth_enabled) {
        const redirectLocation = `${this.coreSetup.http.basePath.serverBasePath}${_common.ANONYMOUS_AUTH_LOGIN}?${nextUrlParam}`;
        return response.redirected({
          headers: {
            location: `${redirectLocation}`
          }
        });
      }

      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}?${nextUrlParam}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$authType3;

    const reqAuthType = cookie === null || cookie === void 0 ? void 0 : (_cookie$authType3 = cookie.authType) === null || _cookie$authType3 === void 0 ? void 0 : _cookie$authType3.toLowerCase();

    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).buildAuthHeaderFromCookie(cookie, request);
    } else {
      return {};
    }
  }

}

exports.MultipleAuthentication = MultipleAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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