"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../../utils/constants");

var _helpers = require("./utils/helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class MonitorService {
  constructor(esDriver) {
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const createResponse = await callAsCurrentUser('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const createResponse = await callAsCurrentUser('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const response = await callAsCurrentUser('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const response = await callAsCurrentUser('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await callAsCurrentUser('alerting.getMonitor', params);

        let monitor = _lodash.default.get(getResponse, 'monitor', null);

        const version = _lodash.default.get(getResponse, '_version', null);

        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);

        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);

        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);

        if (monitor) {
          const {
            callAsCurrentUser
          } = this.esDriver.asScoped(req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await callAsCurrentUser('alerting.getMonitors', aggsParams);

          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);

          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);

          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);

          if (associated_workflows) {
            monitor = { ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }

          monitor = { ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await callAsCurrentUser('alerting.getWorkflow', params);

        let workflow = _lodash.default.get(getResponse, 'workflow', null);

        const version = _lodash.default.get(getResponse, '_version', null);

        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);

        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);

        workflow.monitor_type = workflow.workflow_type;
        workflow = { ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body; // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?

        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;

        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }

        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const updateResponse = await callAsCurrentUser(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };

        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }

        const should = [];
        const mustList = [must];

        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }

        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }

        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };

        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }

        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const {
          callAsCurrentUser: alertingCallAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await alertingCallAsCurrentUser('alerting.getMonitors', params);

        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);

        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});

        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};

        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });

        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };

        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }

        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const esAggsResponse = await callAsCurrentUser('alerting.getMonitors', aggsParams);

        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return { ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });

        const unusedMonitors = [...monitorMap.values()].map(monitor => ({ ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));

        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]); // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination


        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }

        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitors', err);

        if ((0, _helpers.isIndexNotFoundError)(err)) {
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: []
              }
            }
          });
        }

        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const acknowledgeResponse = await callAsCurrentUser('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const acknowledgeResponse = await callAsCurrentUser('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const executeResponse = await callAsCurrentUser('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const results = await callAsCurrentUser('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - searchMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    this.esDriver = esDriver;
  } //TODO: This is temporarily a pass through call which needs to be deprecated


}

exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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