"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationsManager = void 0;

var _path = _interopRequireDefault(require("path"));

var _metrics_helper = require("../../common/metrics/metrics_helper");

var _integrations_builder = require("./integrations_builder");

var _repository = require("./repository/repository");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class IntegrationsManager {
  constructor(client, repository) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "instanceBuilder", void 0);

    _defineProperty(this, "repository", void 0);

    _defineProperty(this, "deleteIntegrationInstance", async id => {
      let children;

      try {
        children = await this.client.get('integration-instance', id);
      } catch (err) {
        var _err$output;

        return ((_err$output = err.output) === null || _err$output === void 0 ? void 0 : _err$output.statusCode) === 404 ? Promise.resolve([id]) : Promise.reject(err);
      }

      const toDelete = children.attributes.assets.filter(i => i.assetId).map(i => {
        return {
          id: i.assetId,
          type: i.assetType
        };
      });
      toDelete.push({
        id,
        type: 'integration-instance'
      });
      const result = Promise.all(toDelete.map(async asset => {
        try {
          await this.client.delete(asset.type, asset.id);
          return Promise.resolve(asset.id);
        } catch (err) {
          var _err$output2;

          (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', err);
          return ((_err$output2 = err.output) === null || _err$output2 === void 0 ? void 0 : _err$output2.statusCode) === 404 ? Promise.resolve(asset.id) : Promise.reject(err);
        }
      }));
      (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', 'count');
      return result;
    });

    _defineProperty(this, "_getAllIntegrationTemplates", async () => {
      const integrationList = await this.repository.getIntegrationList();
      const configResults = await Promise.all(integrationList.map(x => x.getConfig()));
      const configs = configResults.filter(cfg => cfg.ok);
      return Promise.resolve({
        hits: configs.map(cfg => cfg.value)
      });
    });

    _defineProperty(this, "_getIntegrationTemplatesByName", async name => {
      const integration = await this.repository.getIntegration(name);
      const config = await (integration === null || integration === void 0 ? void 0 : integration.getConfig());

      if (!config || !config.ok) {
        return Promise.resolve({
          hits: []
        });
      }

      return Promise.resolve({
        hits: [config.value]
      });
    });

    _defineProperty(this, "getIntegrationTemplates", async query => {
      if (query !== null && query !== void 0 && query.name) {
        return this._getIntegrationTemplatesByName(query.name);
      }

      return this._getAllIntegrationTemplates();
    });

    _defineProperty(this, "getIntegrationInstances", async _query => {
      var _result$saved_objects;

      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.find({
        type: 'integration-instance'
      });
      return Promise.resolve({
        total: result.total,
        hits: (_result$saved_objects = result.saved_objects) === null || _result$saved_objects === void 0 ? void 0 : _result$saved_objects.map(x => ({ ...x.attributes,
          id: x.id
        }))
      });
    });

    _defineProperty(this, "getIntegrationInstance", async query => {
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.get('integration-instance', `${query.id}`);
      return Promise.resolve(this.buildInstanceResponse(result));
    });

    _defineProperty(this, "buildInstanceResponse", async savedObj => {
      var _savedObj$attributes;

      const assets = (_savedObj$attributes = savedObj.attributes) === null || _savedObj$attributes === void 0 ? void 0 : _savedObj$attributes.assets;
      const status = assets ? await this.getAssetStatus(assets) : 'available';
      return {
        id: savedObj.id,
        status,
        ...savedObj.attributes
      };
    });

    _defineProperty(this, "getAssetStatus", async assets => {
      const statuses = await Promise.all(assets.map(async asset => {
        try {
          await this.client.get(asset.assetType, asset.assetId);
          return {
            id: asset.assetId,
            status: 'available'
          };
        } catch (err) {
          var _err$output3;

          const statusCode = (_err$output3 = err.output) === null || _err$output3 === void 0 ? void 0 : _err$output3.statusCode;

          if (statusCode && 400 <= statusCode && statusCode < 500) {
            return {
              id: asset.assetId,
              status: 'unavailable'
            };
          }

          console.error('Failed to get asset status', err);
          return {
            id: asset.assetId,
            status: 'unknown'
          };
        }
      }));
      const [available, unavailable, unknown] = [statuses.filter(x => x.status === 'available').length, statuses.filter(x => x.status === 'unavailable').length, statuses.filter(x => x.status === 'unknown').length];
      if (unknown > 0) return 'unknown';
      if (unavailable > 0 && available > 0) return 'partially-available';
      if (unavailable > 0) return 'unavailable';
      return 'available';
    });

    _defineProperty(this, "loadIntegrationInstance", async (templateName, name, dataSource) => {
      const template = await this.repository.getIntegration(templateName);

      if (template === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      try {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', 'count');
        const result = await this.instanceBuilder.build(template, {
          name,
          dataSource
        });
        const test = await this.client.create('integration-instance', result);
        return Promise.resolve({ ...result,
          id: test.id
        });
      } catch (err) {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', err);
        return Promise.reject({
          message: err.message,
          statusCode: 500
        });
      }
    });

    _defineProperty(this, "getStatic", async (templateName, staticPath) => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      const data = await integration.getStatic(staticPath);

      if (data.ok) {
        return data.value;
      }

      const is404 = data.error.code === 'ENOENT';
      return Promise.reject({
        message: data.error.message,
        statusCode: is404 ? 404 : 500
      });
    });

    _defineProperty(this, "getSchemas", async templateName => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      const result = await integration.getSchemas();

      if (result.ok) {
        return result.value;
      }

      const is404 = result.error.code === 'ENOENT';
      return Promise.reject({
        message: result.error.message,
        statusCode: is404 ? 404 : 500
      });
    });

    _defineProperty(this, "getAssets", async templateName => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      const assets = await integration.getAssets();

      if (assets.ok) {
        return assets.value;
      }

      const is404 = assets.error.code === 'ENOENT';
      return Promise.reject({
        message: assets.error.message,
        statusCode: is404 ? 404 : 500
      });
    });

    _defineProperty(this, "getSampleData", async templateName => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      const sampleData = await integration.getSampleData();

      if (sampleData.ok) {
        return sampleData.value;
      }

      const is404 = sampleData.error.code === 'ENOENT';
      return Promise.reject({
        message: sampleData.error.message,
        statusCode: is404 ? 404 : 500
      });
    });

    this.client = client;
    this.repository = repository !== null && repository !== void 0 ? repository : new _repository.RepositoryReader(_path.default.join(__dirname, '__data__/repository'));
    this.instanceBuilder = new _integrations_builder.IntegrationInstanceBuilder(this.client);
  } // Internal; use getIntegrationTemplates.
  // Internal; use getIntegrationTemplates.


}

exports.IntegrationsManager = IntegrationsManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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