/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.security;

import com.codahale.metrics.Counter;
import com.codahale.metrics.Meter;
import com.codahale.metrics.Timer;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.Closeable;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.util.ExecutorUtil;
import org.apache.solr.common.util.SolrNamedThreadFactory;
import org.apache.solr.common.util.TimeSource;
import org.apache.solr.core.SolrInfoBean;
import org.apache.solr.metrics.SolrMetricsContext;
import org.apache.solr.security.AuditEvent;
import org.apache.solr.util.TimeOut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AuditLoggerPlugin
implements Closeable,
Runnable,
SolrInfoBean {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final String PARAM_EVENT_TYPES = "eventTypes";
    static final String PARAM_ASYNC = "async";
    static final String PARAM_BLOCKASYNC = "blockAsync";
    static final String PARAM_QUEUE_SIZE = "queueSize";
    static final String PARAM_NUM_THREADS = "numThreads";
    static final String PARAM_MUTE_RULES = "muteRules";
    private static final int DEFAULT_QUEUE_SIZE = 4096;
    private static final int DEFAULT_NUM_THREADS = Math.max(2, Runtime.getRuntime().availableProcessors() / 2);
    BlockingQueue<AuditEvent> queue;
    AtomicInteger auditsInFlight = new AtomicInteger(0);
    boolean async;
    boolean blockAsync;
    int blockingQueueSize;
    protected AuditEventFormatter formatter;
    private Set<String> metricNames = ConcurrentHashMap.newKeySet();
    private ExecutorService executorService;
    private boolean closed;
    private MuteRules muteRules;
    protected SolrMetricsContext solrMetricsContext;
    protected Meter numErrors = new Meter();
    protected Meter numLost = new Meter();
    protected Meter numLogged = new Meter();
    protected Timer requestTimes = new Timer();
    protected Timer queuedTime = new Timer();
    protected Counter totalTime = new Counter();
    protected List<String> eventTypes = Arrays.asList(AuditEvent.EventType.COMPLETED.name(), AuditEvent.EventType.ERROR.name(), AuditEvent.EventType.REJECTED.name(), AuditEvent.EventType.UNAUTHORIZED.name(), AuditEvent.EventType.ANONYMOUS_REJECTED.name());

    public void init(Map<String, Object> pluginConfig) {
        this.formatter = new JSONAuditEventFormatter();
        if (pluginConfig.containsKey(PARAM_EVENT_TYPES)) {
            this.eventTypes = (List)pluginConfig.get(PARAM_EVENT_TYPES);
        }
        pluginConfig.remove(PARAM_EVENT_TYPES);
        this.async = Boolean.parseBoolean(String.valueOf(pluginConfig.getOrDefault(PARAM_ASYNC, true)));
        this.blockAsync = Boolean.parseBoolean(String.valueOf(pluginConfig.getOrDefault(PARAM_BLOCKASYNC, false)));
        this.blockingQueueSize = this.async ? Integer.parseInt(String.valueOf(pluginConfig.getOrDefault(PARAM_QUEUE_SIZE, 4096))) : 1;
        int numThreads = this.async ? Integer.parseInt(String.valueOf(pluginConfig.getOrDefault(PARAM_NUM_THREADS, DEFAULT_NUM_THREADS))) : 1;
        this.muteRules = new MuteRules(pluginConfig.remove(PARAM_MUTE_RULES));
        pluginConfig.remove(PARAM_ASYNC);
        pluginConfig.remove(PARAM_BLOCKASYNC);
        pluginConfig.remove(PARAM_QUEUE_SIZE);
        pluginConfig.remove(PARAM_NUM_THREADS);
        if (this.async) {
            this.queue = new ArrayBlockingQueue<AuditEvent>(this.blockingQueueSize);
            this.executorService = ExecutorUtil.newMDCAwareFixedThreadPool((int)numThreads, (ThreadFactory)new SolrNamedThreadFactory("audit"));
            this.executorService.submit(this);
        }
        pluginConfig.remove("class");
        log.debug("AuditLogger initialized in {} mode with event types {}", (Object)(this.async ? PARAM_ASYNC : "syncronous"), this.eventTypes);
    }

    protected abstract void audit(AuditEvent var1);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void doAudit(AuditEvent event) {
        if (this.shouldMute(event)) {
            log.debug("Event muted due to mute rule(s)");
            return;
        }
        if (this.async) {
            this.auditAsync(event);
        } else {
            Timer.Context timer = this.requestTimes.time();
            this.numLogged.mark();
            try {
                this.audit(event);
            }
            catch (Exception e) {
                this.numErrors.mark();
                log.error("Exception when attempting to audit log", (Throwable)e);
            }
            finally {
                this.totalTime.inc(timer.stop());
            }
        }
    }

    protected boolean shouldMute(AuditEvent event) {
        return this.muteRules.shouldMute(event);
    }

    protected final void auditAsync(AuditEvent event) {
        assert (this.async);
        if (this.blockAsync) {
            try {
                this.queue.put(event);
            }
            catch (InterruptedException e) {
                log.warn("Interrupted while waiting to insert AuditEvent into blocking queue");
                Thread.currentThread().interrupt();
            }
        } else if (!this.queue.offer(event)) {
            log.warn("Audit log async queue is full (size={}), not blocking since {}==false", (Object)this.blockingQueueSize, (Object)PARAM_BLOCKASYNC);
            this.numLost.mark();
        }
    }

    @Override
    public void run() {
        assert (this.async);
        while (!this.closed && !Thread.currentThread().isInterrupted()) {
            try {
                AuditEvent event = this.queue.poll(1000L, TimeUnit.MILLISECONDS);
                this.auditsInFlight.incrementAndGet();
                if (event == null) continue;
                if (event.getDate() != null) {
                    this.queuedTime.update(new Date().getTime() - event.getDate().getTime(), TimeUnit.MILLISECONDS);
                }
                Timer.Context timer = this.requestTimes.time();
                this.audit(event);
                this.numLogged.mark();
                this.totalTime.inc(timer.stop());
            }
            catch (InterruptedException e) {
                log.warn("Interrupted while waiting for next audit log event");
                Thread.currentThread().interrupt();
            }
            catch (Exception ex) {
                log.error("Exception when attempting to audit log asynchronously", (Throwable)ex);
                this.numErrors.mark();
            }
            finally {
                this.auditsInFlight.decrementAndGet();
            }
        }
    }

    public boolean shouldLog(AuditEvent.EventType eventType) {
        boolean shouldLog = this.eventTypes.contains(eventType.name());
        if (!shouldLog && log.isDebugEnabled()) {
            log.debug("Event type {} is not configured for audit logging", (Object)eventType.name());
        }
        return shouldLog;
    }

    public void setFormatter(AuditEventFormatter formatter) {
        this.formatter = formatter;
    }

    @Override
    public void initializeMetrics(SolrMetricsContext parentContext, String scope) {
        this.solrMetricsContext = parentContext.getChildContext(this);
        String className = this.getClass().getSimpleName();
        log.debug("Initializing metrics for {}", (Object)className);
        this.numErrors = this.solrMetricsContext.meter("errors", this.getCategory().toString(), scope, className);
        this.numLost = this.solrMetricsContext.meter("lost", this.getCategory().toString(), scope, className);
        this.numLogged = this.solrMetricsContext.meter("count", this.getCategory().toString(), scope, className);
        this.requestTimes = this.solrMetricsContext.timer("requestTimes", this.getCategory().toString(), scope, className);
        this.totalTime = this.solrMetricsContext.counter("totalTime", this.getCategory().toString(), scope, className);
        if (this.async) {
            this.solrMetricsContext.gauge(() -> this.blockingQueueSize, true, "queueCapacity", this.getCategory().toString(), scope, className);
            this.solrMetricsContext.gauge(() -> this.blockingQueueSize - this.queue.remainingCapacity(), true, PARAM_QUEUE_SIZE, this.getCategory().toString(), scope, className);
            this.queuedTime = this.solrMetricsContext.timer("queuedTime", this.getCategory().toString(), scope, className);
        }
        this.solrMetricsContext.gauge(() -> this.async, true, PARAM_ASYNC, this.getCategory().toString(), scope, className);
    }

    @Override
    public String getName() {
        return this.getClass().getName();
    }

    @Override
    public String getDescription() {
        return "Auditlogger Plugin " + this.getClass().getName();
    }

    @Override
    public SolrInfoBean.Category getCategory() {
        return SolrInfoBean.Category.SECURITY;
    }

    @Override
    public SolrMetricsContext getSolrMetricsContext() {
        return this.solrMetricsContext;
    }

    @Override
    public void close() throws IOException {
        if (this.async && this.executorService != null) {
            this.waitForQueueToDrain(30);
            this.closed = true;
            log.info("Shutting down async Auditlogger background thread(s)");
            ExecutorUtil.shutdownNowAndAwaitTermination((ExecutorService)this.executorService);
            this.executorService = null;
            try {
                SolrInfoBean.super.close();
            }
            catch (Exception e) {
                throw new IOException("Exception closing", e);
            }
        }
    }

    protected void waitForQueueToDrain(int timeoutSeconds) {
        if (this.async && this.executorService != null) {
            TimeOut timeOut = new TimeOut(timeoutSeconds, TimeUnit.SECONDS, TimeSource.NANO_TIME);
            while (!(this.queue.isEmpty() && this.auditsInFlight.get() <= 0 || timeOut.hasTimedOut())) {
                try {
                    if (log.isInfoEnabled()) {
                        log.info("Async auditlogger queue still has {} elements and {} audits in-flight, sleeping to drain...", (Object)this.queue.size(), (Object)this.auditsInFlight.get());
                    }
                    timeOut.sleep(1000L);
                }
                catch (InterruptedException ignored) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    public static interface MuteRule {
        public boolean shouldMute(AuditEvent var1);
    }

    private static class MuteRules {
        private List<List<MuteRule>> rules = new ArrayList<List<MuteRule>>();

        MuteRules(Object o) {
            if (o != null) {
                if (o instanceof List) {
                    ((List)o).forEach(l -> {
                        if (l instanceof String) {
                            this.rules.add(Collections.singletonList(this.parseRule(l)));
                        } else if (l instanceof List) {
                            ArrayList rl = new ArrayList();
                            ((List)l).forEach(r -> rl.add(this.parseRule(r)));
                            this.rules.add(rl);
                        }
                    });
                } else {
                    throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "The muteRules configuration must be a list");
                }
            }
        }

        private MuteRule parseRule(Object ruleObj) {
            try {
                String rule = (String)ruleObj;
                if (rule.startsWith("type:")) {
                    AuditEvent.RequestType muteType = AuditEvent.RequestType.valueOf(rule.substring("type:".length()));
                    return event -> event.getRequestType() != null && event.getRequestType().equals((Object)muteType);
                }
                if (rule.startsWith("collection:")) {
                    return event -> event.getCollections() != null && event.getCollections().contains(rule.substring("collection:".length()));
                }
                if (rule.startsWith("user:")) {
                    return event -> event.getUsername() != null && event.getUsername().equals(rule.substring("user:".length()));
                }
                if (rule.startsWith("path:")) {
                    return event -> event.getResource() != null && event.getResource().startsWith(rule.substring("path:".length()));
                }
                if (rule.startsWith("ip:")) {
                    return event -> event.getClientIp() != null && event.getClientIp().equals(rule.substring("ip:".length()));
                }
                if (rule.startsWith("param:")) {
                    String[] kv = rule.substring("param:".length()).split("=");
                    if (kv.length == 2) {
                        return event -> event.getSolrParams() != null && kv[1].equals(event.getSolrParamAsString(kv[0]));
                    }
                    throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "The 'param' muteRule must be of format 'param:key=value', got " + rule);
                }
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Unkonwn mute rule " + rule);
            }
            catch (ClassCastException | IllegalArgumentException e) {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "There was a problem parsing muteRules. Must be a list of valid rule strings", (Throwable)e);
            }
        }

        boolean shouldMute(AuditEvent event) {
            if (this.rules == null) {
                return false;
            }
            return this.rules.stream().anyMatch(rl -> rl.stream().allMatch(r -> r.shouldMute(event)));
        }
    }

    public static class JSONAuditEventFormatter
    implements AuditEventFormatter {
        private static ObjectMapper mapper = new ObjectMapper().configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false).setSerializationInclusion(JsonInclude.Include.NON_NULL);

        @Override
        public String formatEvent(AuditEvent event) {
            try {
                StringWriter sw = new StringWriter();
                mapper.writeValue((Writer)sw, (Object)event);
                return sw.toString();
            }
            catch (IOException e) {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error converting Event to JSON", (Throwable)e);
            }
        }
    }

    public static interface AuditEventFormatter {
        public String formatEvent(AuditEvent var1);
    }
}

