/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.database.connection.util;

import java.util.Random;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryHelper {
    private static final Logger logger = LoggerFactory.getLogger(RetryHelper.class);
    private final ScheduledExecutorService executorService;
    private final long minRetryDelayAfterFailure;
    private final long maxRetryDelay;
    private final double jitterFactor;
    private final double retryExponent;
    private final Random random = new Random();
    private ScheduledFuture<?> scheduledRetry;
    private long currentRetryDelay;
    private boolean lastWasSuccess = true;

    private RetryHelper(ScheduledExecutorService executorService, long minRetryDelayAfterFailure, long maxRetryDelay, double retryExponent, double jitterFactor) {
        this.executorService = executorService;
        this.minRetryDelayAfterFailure = minRetryDelayAfterFailure;
        this.maxRetryDelay = maxRetryDelay;
        this.retryExponent = retryExponent;
        this.jitterFactor = jitterFactor;
    }

    public void retry(final Runnable runnable) {
        long delay;
        if (this.scheduledRetry != null) {
            logger.debug("Cancelling previous scheduled retry");
            this.scheduledRetry.cancel(false);
            this.scheduledRetry = null;
        }
        if (this.lastWasSuccess) {
            delay = 0L;
        } else {
            if (this.currentRetryDelay == 0L) {
                this.currentRetryDelay = this.minRetryDelayAfterFailure;
            } else {
                long newDelay = (long)((double)this.currentRetryDelay * this.retryExponent);
                this.currentRetryDelay = Math.min(newDelay, this.maxRetryDelay);
            }
            delay = (long)((1.0 - this.jitterFactor) * (double)this.currentRetryDelay + this.jitterFactor * (double)this.currentRetryDelay * this.random.nextDouble());
        }
        this.lastWasSuccess = false;
        logger.debug("Scheduling retry in {}ms", (Object)delay);
        Runnable wrapped = new Runnable(){

            @Override
            public void run() {
                RetryHelper.this.scheduledRetry = null;
                runnable.run();
            }
        };
        this.scheduledRetry = this.executorService.schedule(wrapped, delay, TimeUnit.MILLISECONDS);
    }

    public void signalSuccess() {
        this.lastWasSuccess = true;
        this.currentRetryDelay = 0L;
    }

    public void setMaxDelay() {
        this.currentRetryDelay = this.maxRetryDelay;
    }

    public void cancel() {
        if (this.scheduledRetry != null) {
            logger.debug("Cancelling existing retry attempt");
            this.scheduledRetry.cancel(false);
            this.scheduledRetry = null;
        } else {
            logger.debug("No existing retry attempt to cancel");
        }
        this.currentRetryDelay = 0L;
    }

    public static class Builder {
        private final ScheduledExecutorService service;
        private long minRetryDelayAfterFailure = 1000L;
        private double jitterFactor = 0.5;
        private long retryMaxDelay = 30000L;
        private double retryExponent = 1.3;

        public Builder(ScheduledExecutorService service, Class tag) {
            this.service = service;
        }

        public Builder withMinDelayAfterFailure(long delay) {
            this.minRetryDelayAfterFailure = delay;
            return this;
        }

        public Builder withMaxDelay(long delay) {
            this.retryMaxDelay = delay;
            return this;
        }

        public Builder withRetryExponent(double exponent) {
            this.retryExponent = exponent;
            return this;
        }

        public Builder withJitterFactor(double random) {
            if (random < 0.0 || random > 1.0) {
                throw new IllegalArgumentException("Argument out of range: " + random);
            }
            this.jitterFactor = random;
            return this;
        }

        public RetryHelper build() {
            return new RetryHelper(this.service, this.minRetryDelayAfterFailure, this.retryMaxDelay, this.retryExponent, this.jitterFactor);
        }
    }
}

