/* Copyright (C) 1991-2023 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <https://www.gnu.org/licenses/>.  */

/*
 *	ISO C99 Standard: 7.23 Date and time	<time.h>
 */

#ifndef	_TIME_H
#define _TIME_H	1

#include <features.h>

#define __need_size_t
#define __need_NULL
#include <stddef.h>

/* This defines CLOCKS_PER_SEC, which is the number of processor clock
   ticks per second, and possibly a number of other constants.   */
#include <bits/time.h>

/* Many of the typedefs and structs whose official home is this header
   may also need to be defined by other headers.  */
#include <bits/types/clock_t.h>
#include <bits/types/time_t.h>
#include <bits/types/struct_tm.h>

#if defined __USE_POSIX199309 || defined __USE_ISOC11
# include <bits/types/struct_timespec.h>
#endif

#ifdef __USE_POSIX199309
# include <bits/types/clockid_t.h>
# include <bits/types/timer_t.h>
# include <bits/types/struct_itimerspec.h>
struct sigevent;
#endif

#ifdef __USE_XOPEN2K
# ifndef __pid_t_defined
typedef __pid_t pid_t;
#  define __pid_t_defined
# endif
#endif

#ifdef __USE_XOPEN2K8
# include <bits/types/locale_t.h>
#endif

#ifdef __USE_ISOC11
/* Time base values for timespec_get.  */
# define TIME_UTC 1
#endif

__BEGIN_DECLS

/* Time used by the program so far (user time + system time).
   The result / CLOCKS_PER_SEC is program time in seconds.  */
extern clock_t clock (void) __THROW;

#ifndef __USE_TIME_BITS64
/* Return the current time and put it in *TIMER if TIMER is not NULL.  */
extern time_t time (time_t *__timer) __THROW;

/* Return the difference between TIME1 and TIME0.  */
extern double difftime (time_t __time1, time_t __time0)
     __THROW __attribute__ ((__const__));

/* Return the `time_t' representation of TP and normalize TP.  */
extern time_t mktime (struct tm *__tp) __THROW;
#else
# ifdef __REDIRECT_NTH
extern time_t __REDIRECT_NTH (time, (time_t *__timer), __time64);
extern double __REDIRECT_NTH (difftime, (time_t __time1, time_t __time0),
                              __difftime64) __attribute__ ((__const__));
extern time_t __REDIRECT_NTH (mktime, (struct tm *__tp), __mktime64);
# else
#  define time __time64
#  define difftime __difftime64
#  define mktime __mktime64
# endif
#endif

/* Format TP into S according to FORMAT.
   Write no more than MAXSIZE characters and return the number
   of characters written, or 0 if it would exceed MAXSIZE.  */
extern size_t strftime (char *__restrict __s, size_t __maxsize,
			const char *__restrict __format,
			const struct tm *__restrict __tp)
			__THROW __nonnull((1, 3, 4));

#ifdef __USE_XOPEN
/* Parse S according to FORMAT and store binary time information in TP.
   The return value is a pointer to the first unparsed character in S.  */
extern char *strptime (const char *__restrict __s,
		       const char *__restrict __fmt, struct tm *__tp)
     __THROW;
#endif

#ifdef __USE_XOPEN2K8
/* Similar to the two functions above but take the information from
   the provided locale and not the global locale.  */

extern size_t strftime_l (char *__restrict __s, size_t __maxsize,
			  const char *__restrict __format,
			  const struct tm *__restrict __tp,
			  locale_t __loc) __THROW;
#endif

#ifdef __USE_GNU
extern char *strptime_l (const char *__restrict __s,
			 const char *__restrict __fmt, struct tm *__tp,
			 locale_t __loc) __THROW;
#endif


#ifndef __USE_TIME_BITS64
/* Return the `struct tm' representation of *TIMER
   in Universal Coordinated Time (aka Greenwich Mean Time).  */
extern struct tm *gmtime (const time_t *__timer) __THROW;

/* Return the `struct tm' representation
   of *TIMER in the local timezone.  */
extern struct tm *localtime (const time_t *__timer) __THROW;

#else
# ifdef __REDIRECT_NTH
extern struct tm*__REDIRECT_NTH (gmtime, (const time_t *__timer), __gmtime64);
extern struct tm *__REDIRECT_NTH (localtime, (const time_t *__timer),
				  __localtime64);
# else
#  define gmtime __gmtime64
#  define localtime __localtime64
# endif
#endif


#if defined __USE_POSIX || __GLIBC_USE (ISOC2X)
# ifndef __USE_TIME_BITS64
/* Return the `struct tm' representation of *TIMER in UTC,
   using *TP to store the result.  */
extern struct tm *gmtime_r (const time_t *__restrict __timer,
			    struct tm *__restrict __tp) __THROW;

/* Return the `struct tm' representation of *TIMER in local time,
   using *TP to store the result.  */
extern struct tm *localtime_r (const time_t *__restrict __timer,
			       struct tm *__restrict __tp) __THROW;
# else
#  ifdef __REDIRECT_NTH
extern struct tm*__REDIRECT_NTH (gmtime_r, (const time_t *__restrict __timer,
                                            struct tm *__restrict __tp),
                                 __gmtime64_r);

extern struct tm*__REDIRECT_NTH (localtime_r, (const time_t *__restrict __t,
                                               struct tm *__restrict __tp),
                                 __localtime64_r);
#  else
#   define gmtime_r __gmtime64_r
#   define localtime_r __localtime_r
#  endif
# endif
#endif	/* POSIX || C2X */

/* Return a string of the form "Day Mon dd hh:mm:ss yyyy\n"
   that is the representation of TP in this format.  */
extern char *asctime (const struct tm *__tp) __THROW;

/* Equivalent to `asctime (localtime (timer))'.  */
#ifndef __USE_TIME_BITS64
extern char *ctime (const time_t *__timer) __THROW;
#else
# ifdef __REDIRECT_NTH
extern char *__REDIRECT_NTH (ctime, (const time_t *__timer), __ctime64);
# else
#  define ctime __ctime64
# endif
#endif

#ifdef __USE_POSIX
/* Reentrant versions of the above functions.  */

/* Return in BUF a string of the form "Day Mon dd hh:mm:ss yyyy\n"
   that is the representation of TP in this format.  */
extern char *asctime_r (const struct tm *__restrict __tp,
			char *__restrict __buf) __THROW;

/* Equivalent to `asctime_r (localtime_r (timer, *TMP*), buf)'.  */
#ifndef __USE_TIME_BITS64
extern char *ctime_r (const time_t *__restrict __timer,
		      char *__restrict __buf) __THROW;
#else
# ifdef __REDIRECT_NTH
extern char *__REDIRECT_NTH (ctime_r, (const time_t *__restrict __timer,
                                       char *__restrict __buf), __ctime64_r);
# else
#  define ctime_r __ctime64_r
# endif
#endif

#endif	/* POSIX */


/* Defined in localtime.c.  */
extern char *__tzname[2];	/* Current time zone abbreviations.  */
extern int __daylight;		/* If daylight-saving time is ever in use.  */
extern long int __timezone;	/* Seconds west of UTC.  */


#ifdef	__USE_POSIX
/* Same as above.  */
extern char *tzname[2];

/* Set time conversion information from the TZ environment variable.
   If TZ is not defined, a locale-dependent default is used.  */
extern void tzset (void) __THROW;
#endif

#if defined __USE_MISC || defined __USE_XOPEN
extern int daylight;
extern long int timezone;
#endif


/* Nonzero if YEAR is a leap year (every 4 years,
   except every 100th isn't, and every 400th is).  */
#define __isleap(year)	\
  ((year) % 4 == 0 && ((year) % 100 != 0 || (year) % 400 == 0))


#if defined __USE_MISC || __GLIBC_USE (ISOC2X)
# ifndef __USE_TIME_BITS64
/* Like `mktime', but for TP represents Universal Time, not local time.  */
extern time_t timegm (struct tm *__tp) __THROW;
# else
#  ifdef __REDIRECT_NTH
extern time_t __REDIRECT_NTH (timegm, (struct tm *__tp), __timegm64);
#  else
#   define timegm __timegm64
#  endif
# endif
#endif


#ifdef __USE_MISC
/* Miscellaneous functions many Unices inherited from the public domain
   localtime package.  These are included only for compatibility.  */

#ifndef __USE_TIME_BITS64
/* Another name for `mktime'.  */
extern time_t timelocal (struct tm *__tp) __THROW;
#else
# ifdef __REDIRECT_NTH
extern time_t __REDIRECT_NTH (timelocal, (struct tm *__tp), __mktime64);
# endif
#endif

/* Return the number of days in YEAR.  */
extern int dysize (int __year) __THROW  __attribute__ ((__const__));
#endif


#ifdef __USE_POSIX199309
# ifndef __USE_TIME_BITS64
/* Pause execution for a number of nanoseconds.

   This function is a cancellation point and therefore not marked with
   __THROW.  */
extern int nanosleep (const struct timespec *__requested_time,
		      struct timespec *__remaining);

/* Get resolution of clock CLOCK_ID.  */
extern int clock_getres (clockid_t __clock_id, struct timespec *__res) __THROW;

/* Get current value of clock CLOCK_ID and store it in TP.  */
extern int clock_gettime (clockid_t __clock_id, struct timespec *__tp)
     __THROW __nonnull((2));

/* Set clock CLOCK_ID to value TP.  */
extern int clock_settime (clockid_t __clock_id, const struct timespec *__tp)
     __THROW __nonnull((2));
# else
#  ifdef __REDIRECT
extern int __REDIRECT (nanosleep, (const struct timespec *__requested_time,
                                   struct timespec *__remaining),
                       __nanosleep64);
extern int __REDIRECT_NTH (clock_getres, (clockid_t __clock_id,
                                          struct timespec *__res),
                           __clock_getres64);
extern int __REDIRECT_NTH (clock_gettime, (clockid_t __clock_id, struct
                                           timespec *__tp), __clock_gettime64)
                           __nonnull((2));
extern int __REDIRECT_NTH (clock_settime, (clockid_t __clock_id, const struct
                                           timespec *__tp), __clock_settime64)
                           __nonnull((2));
#  else
#   define nanosleep __nanosleep64
#   define clock_getres __clock_getres64
#   define clock_gettime __clock_gettime64
#   define clock_settime __clock_settime64
#  endif
# endif


# ifdef __USE_XOPEN2K
/* High-resolution sleep with the specified clock.

   This function is a cancellation point and therefore not marked with
   __THROW.  */
#  ifndef __USE_TIME_BITS64
extern int clock_nanosleep (clockid_t __clock_id, int __flags,
			    const struct timespec *__req,
			    struct timespec *__rem);
#  else
#   ifdef __REDIRECT
extern int __REDIRECT (clock_nanosleep, (clockid_t __clock_id, int __flags,
                                         const struct timespec *__req,
                                         struct timespec *__rem),
                       __clock_nanosleep_time64);
#   else
#    define clock_nanosleep __clock_nanosleep_time64
#   endif
#  endif

/* Return clock ID for CPU-time clock.  */
extern int clock_getcpuclockid (pid_t __pid, clockid_t *__clock_id) __THROW;
# endif


/* Create new per-process timer using CLOCK_ID.  */
extern int timer_create (clockid_t __clock_id,
			 struct sigevent *__restrict __evp,
			 timer_t *__restrict __timerid) __THROW;

/* Delete timer TIMERID.  */
extern int timer_delete (timer_t __timerid) __THROW;

/* Set timer TIMERID to VALUE, returning old value in OVALUE.  */
# ifndef __USE_TIME_BITS64
extern int timer_settime (timer_t __timerid, int __flags,
			  const struct itimerspec *__restrict __value,
			  struct itimerspec *__restrict __ovalue) __THROW;

/* Get current value of timer TIMERID and store it in VALUE.  */
extern int timer_gettime (timer_t __timerid, struct itimerspec *__value)
     __THROW;
# else
#  ifdef __REDIRECT_NTH
extern int __REDIRECT_NTH (timer_settime, (timer_t __timerid, int __flags,
     const struct itimerspec *__restrict __value,
     struct itimerspec *__restrict __ovalue),
                          __timer_settime64);

extern int __REDIRECT_NTH (timer_gettime, (timer_t __timerid,
                                           struct itimerspec *__value),
                           __timer_gettime64);
#  else
#   define timer_settime __timer_settime64
#   define timer_gettime __timer_gettime64
#  endif
# endif

/* Get expiration overrun for timer TIMERID.  */
extern int timer_getoverrun (timer_t __timerid) __THROW;
#endif


#ifdef __USE_ISOC11
# ifndef __USE_TIME_BITS64
/* Set TS to calendar time based in time base BASE.  */
extern int timespec_get (struct timespec *__ts, int __base)
     __THROW __nonnull ((1));
# else
#  ifdef __REDIRECT_NTH
extern int __REDIRECT_NTH (timespec_get, (struct timespec *__ts, int __base),
                           __timespec_get64) __nonnull ((1));
#  else
#   define timespec_get __timespec_get64
#  endif
# endif
#endif


#if __GLIBC_USE (ISOC2X)
# ifndef __USE_TIME_BITS64
/* Set TS to resolution of time base BASE.  */
extern int timespec_getres (struct timespec *__ts, int __base)
     __THROW;
# else
#  ifdef __REDIRECT_NTH
extern int __REDIRECT_NTH (timespec_getres, (struct timespec *__ts,
                                             int __base),
                           __timespec_getres64);
#  else
#   define timespec_getres __timespec_getres64
#  endif
# endif
#endif


#ifdef __USE_XOPEN_EXTENDED
/* Set to one of the following values to indicate an error.
     1  the DATEMSK environment variable is null or undefined,
     2  the template file cannot be opened for reading,
     3  failed to get file status information,
     4  the template file is not a regular file,
     5  an error is encountered while reading the template file,
     6  memory allication failed (not enough memory available),
     7  there is no line in the template that matches the input,
     8  invalid input specification Example: February 31 or a time is
	specified that can not be represented in a time_t (representing
	the time in seconds since 00:00:00 UTC, January 1, 1970) */
extern int getdate_err;

/* Parse the given string as a date specification and return a value
   representing the value.  The templates from the file identified by
   the environment variable DATEMSK are used.  In case of an error
   `getdate_err' is set.

   This function is a possible cancellation point and therefore not
   marked with __THROW.  */
extern struct tm *getdate (const char *__string);
#endif

#ifdef __USE_GNU
/* Since `getdate' is not reentrant because of the use of `getdate_err'
   and the static buffer to return the result in, we provide a thread-safe
   variant.  The functionality is the same.  The result is returned in
   the buffer pointed to by RESBUFP and in case of an error the return
   value is != 0 with the same values as given above for `getdate_err'.

   This function is not part of POSIX and therefore no official
   cancellation point.  But due to similarity with an POSIX interface
   or due to the implementation it is a cancellation point and
   therefore not marked with __THROW.  */
extern int getdate_r (const char *__restrict __string,
		      struct tm *__restrict __resbufp);
#endif

__END_DECLS

#endif /* time.h.  */