"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _lodash = require("lodash");
var _adHelpers = require("./utils/adHelpers");
var _helpers = require("../utils/helpers");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_indices/{dataSourceId}', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_aliases/{dataSourceId}', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.get('/_mappings/{dataSourceId}', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.put('/create_index/{dataSourceId}', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/bulk/{dataSourceId}', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
}
class OpenSearchService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;

      //@ts-ignore
      const index = request.body.index;
      //@ts-ignore
      const body = request.body.body;
      const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
      try {
        await callWithRequest('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
      try {
        const response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      const body = request.body;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;
      try {
        await callWithRequest('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err);
        // Ignore the error if it's an index_not_found_exception
        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('indices.getMapping', {
          index
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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