"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatRoutes = registerChatRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _olly_chat_service = require("../services/chat/olly_chat_service");
var _agent_framework_storage_service = require("../services/storage/agent_framework_storage_service");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const llmRequestRoute = {
  path: _llm.ASSISTANT_API.SEND_MESSAGE,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.maybe(_configSchema.schema.string()),
      messages: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
      input: _configSchema.schema.object({
        type: _configSchema.schema.literal('input'),
        context: _configSchema.schema.object({
          appId: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        content: _configSchema.schema.string(),
        contentType: _configSchema.schema.literal('text')
      })
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const abortAgentExecutionRoute = {
  path: `${_llm.ASSISTANT_API.ABORT_AGENT_EXECUTION}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const regenerateRoute = {
  path: `${_llm.ASSISTANT_API.REGENERATE}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string(),
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationsRoute = {
  path: _llm.ASSISTANT_API.CONVERSATIONS,
  validate: {
    query: _configSchema.schema.object({
      perPage: _configSchema.schema.number({
        min: 0,
        defaultValue: 20
      }),
      page: _configSchema.schema.number({
        min: 0,
        defaultValue: 1
      }),
      sortOrder: _configSchema.schema.maybe(_configSchema.schema.string()),
      sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
      fields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
      search: _configSchema.schema.maybe(_configSchema.schema.string()),
      searchFields: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const deleteConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const updateConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      title: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getTracesRoute = {
  path: `${_llm.ASSISTANT_API.TRACE}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const feedbackRoute = {
  path: `${_llm.ASSISTANT_API.FEEDBACK}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      satisfaction: _configSchema.schema.boolean()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const accountRoute = {
  path: `${_llm.ASSISTANT_API.ACCOUNT}`,
  validate: {}
};
function registerChatRoutes(router, routeOptions) {
  const createStorageService = async (context, dataSourceId) => new _agent_framework_storage_service.AgentFrameworkStorageService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }), routeOptions.messageParsers);
  const createChatService = async (context, dataSourceId) => new _olly_chat_service.OllyChatService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }));
  router.post(llmRequestRoute, async (context, request, response) => {
    var _outputs, _outputs2;
    const {
      messages = [],
      input,
      conversationId: conversationIdInRequestBody
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.requestLLM({
        messages,
        input,
        conversationId: conversationIdInRequestBody
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }

    /**
     * Retrieve latest interactions from memory
     */
    const conversationId = ((_outputs = outputs) === null || _outputs === void 0 ? void 0 : _outputs.conversationId) || conversationIdInRequestBody;
    const interactionId = ((_outputs2 = outputs) === null || _outputs2 === void 0 ? void 0 : _outputs2.interactionId) || '';
    try {
      if (!conversationId) {
        throw new Error('Not a valid conversation');
      }
      const resultPayload = {
        messages: [],
        interactions: [],
        conversationId
      };
      if (!conversationIdInRequestBody) {
        /**
         * If no conversationId is provided in request payload,
         * it means it is a brand new conversation,
         * need to fetch all the details including title.
         */
        const conversation = await storageService.getConversation(conversationId);
        resultPayload.interactions = conversation.interactions;
        resultPayload.messages = conversation.messages;
        resultPayload.title = conversation.title;
      } else {
        /**
         * Only response with the latest interaction.
         * It may have some issues in Concurrent case like a user may use two tabs to chat with Chatbot in one conversation.
         * But for now we will ignore this case, can be optimized by always fetching conversation if we need to take this case into consideration.
         */
        const interaction = await storageService.getInteraction(conversationId, interactionId);
        resultPayload.interactions = [interaction].filter(item => item);
        resultPayload.messages = resultPayload.interactions.length ? await storageService.getMessagesFromInteractions(resultPayload.interactions) : [];
      }
      return response.ok({
        body: resultPayload
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationsRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversations(request.query);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete(deleteConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.deleteConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(updateConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.updateConversation(request.params.conversationId, request.body.title);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getTracesRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getTraces(request.params.interactionId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post(abortAgentExecutionRoute, async (context, request, response) => {
    const chatService = await createChatService(context, request.query.dataSourceId);
    try {
      chatService.abortAgentExecution(request.body.conversationId);
      context.assistant_plugin.logger.info(`Abort agent execution: ${request.body.conversationId}`);
      return response.ok();
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(regenerateRoute, async (context, request, response) => {
    const {
      conversationId,
      interactionId
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.regenerate({
        conversationId,
        interactionId
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }

    /**
     * Retrieve latest interactions from memory
     */
    try {
      var _outputs3;
      const interaction = await storageService.getInteraction(conversationId, ((_outputs3 = outputs) === null || _outputs3 === void 0 ? void 0 : _outputs3.interactionId) || '');
      const finalInteractions = [interaction].filter(item => item);
      const messages = finalInteractions.length ? await storageService.getMessagesFromInteractions(finalInteractions) : [];
      return response.ok({
        body: {
          interactions: finalInteractions,
          messages,
          conversationId
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(feedbackRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const {
      interactionId
    } = request.params;
    try {
      const updateResponse = await storageService.updateInteraction(interactionId, {
        feedback: request.body
      });
      return response.ok({
        body: {
          ...updateResponse,
          success: true
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(accountRoute, async (context, request, response) => {
    try {
      var _auth$state$authInfo$, _auth$state;
      const auth = routeOptions.auth.get(request);
      return response.ok({
        body: {
          user_name: (_auth$state$authInfo$ = auth === null || auth === void 0 || (_auth$state = auth.state) === null || _auth$state === void 0 || (_auth$state = _auth$state.authInfo) === null || _auth$state === void 0 ? void 0 : _auth$state.user_name) !== null && _auth$state$authInfo$ !== void 0 ? _auth$state$authInfo$ : _llm.DEFAULT_USER_NAME
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.ok({
        body: {
          user_name: _llm.DEFAULT_USER_NAME
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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