/**
 * \file
 * \brief parse SVG path specifications
 *
 * Copyright 2007 MenTaLguY <mental@rydia.net>
 * Copyright 2007 Aaron Spike <aaron@ekips.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it either under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation
 * (the "LGPL") or, at your option, under the terms of the Mozilla
 * Public License Version 1.1 (the "MPL"). If you do not alter this
 * notice, a recipient may use your version of this file under either
 * the MPL or the LGPL.
 *
 * You should have received a copy of the LGPL along with this library
 * in the file COPYING-LGPL-2.1; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * You should have received a copy of the MPL along with this library
 * in the file COPYING-MPL-1.1
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY
 * OF ANY KIND, either express or implied. See the LGPL or the MPL for
 * the specific language governing rights and limitations.
 *
 */


#include <cmath>
#include <vector>
#include <glib.h>

#include <2geom/point.h>
#include <2geom/svg-path-parser.h>
#include <2geom/angle.h>

namespace Geom {

namespace {

class Parser {
public:
    Parser(PathSink &sink) : _absolute(false), _sink(sink) {}

    void parse(char const *str) throw(SVGPathParseError);

private:
    bool _absolute;
    Point _current;
    Point _initial;
    Point _cubic_tangent;
    Point _quad_tangent;
    std::vector<double> _params;
    PathSink &_sink;

    void _reset() {
        _absolute = false;
        _current = _initial = Point(0, 0);
        _quad_tangent = _cubic_tangent = Point(0, 0);
        _params.clear();
    }

    void _push(double value) {
        _params.push_back(value);
    }

    double _pop() {
        double value = _params.back();
        _params.pop_back();
        return value;
    }

    bool _pop_flag() {
        return _pop() != 0.0;
    }

    double _pop_coord(Geom::Dim2 axis) {
        if (_absolute) {
            return _pop();
        } else {
            return _pop() + _current[axis];
        }
    }

    Point _pop_point() {
        double y = _pop_coord(Geom::Y);
        double x = _pop_coord(Geom::X);
        return Point(x, y);
    }

    void _moveTo(Point p) {
        _quad_tangent = _cubic_tangent = _current = _initial = p;
        _sink.moveTo(p);
    }
    
    void _hlineTo(Point p) {
        _quad_tangent = _cubic_tangent = _current = p;
        _sink.hlineTo(p[Geom::X]);
    }
    
    void _vlineTo(Point p) {
        _quad_tangent = _cubic_tangent = _current = p;
        _sink.vlineTo(p[Geom::Y]);
    }

    void _lineTo(Point p) {
        _quad_tangent = _cubic_tangent = _current = p;
        _sink.lineTo(p);
    }

    void _curveTo(Point c0, Point c1, Point p) {
        _quad_tangent = _current = p;
        _cubic_tangent = p + ( p - c1 );
        _sink.curveTo(c0, c1, p);
    }

    void _quadTo(Point c, Point p) {
        _cubic_tangent = _current = p;
        _quad_tangent = p + ( p - c );
        _sink.quadTo(c, p);
    }

    void _arcTo(double rx, double ry, double angle,
                bool large_arc, bool sweep, Point p)
    {
        if (are_near(_current, p)) {
            return;
        }
        _quad_tangent = _cubic_tangent = _current = p;
        _sink.arcTo(rx, ry, angle, large_arc, sweep, p);
    }

    void _closePath() {
        _quad_tangent = _cubic_tangent = _current = _initial;
        _sink.closePath();
    }
};


static const char _svg_path_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 5, 1, 15, 1, 
	16, 2, 1, 0, 2, 1, 6, 2, 
	1, 7, 2, 1, 8, 2, 1, 9, 
	2, 1, 10, 2, 1, 11, 2, 1, 
	12, 2, 1, 13, 2, 1, 14, 2, 
	4, 0, 2, 5, 0, 2, 15, 16, 
	3, 1, 6, 0, 3, 1, 6, 16, 
	3, 1, 7, 0, 3, 1, 7, 16, 
	3, 1, 8, 0, 3, 1, 8, 16, 
	3, 1, 9, 0, 3, 1, 9, 16, 
	3, 1, 10, 0, 3, 1, 10, 16, 
	3, 1, 11, 0, 3, 1, 11, 16, 
	3, 1, 12, 0, 3, 1, 12, 16, 
	3, 1, 13, 0, 3, 1, 13, 16, 
	3, 1, 14, 0, 3, 1, 14, 16
	
};

static const short _svg_path_key_offsets[] = {
	0, 0, 7, 16, 25, 28, 30, 42, 
	52, 55, 57, 90, 121, 124, 126, 138, 
	148, 151, 153, 186, 195, 207, 216, 249, 
	256, 263, 265, 275, 283, 290, 292, 304, 
	314, 317, 319, 328, 335, 341, 346, 353, 
	359, 364, 374, 377, 379, 391, 401, 404, 
	406, 437, 466, 476, 480, 482, 490, 499, 
	508, 511, 513, 525, 535, 538, 540, 552, 
	562, 565, 567, 579, 589, 592, 594, 606, 
	616, 619, 621, 633, 643, 646, 648, 681, 
	712, 724, 733, 745, 754, 766, 775, 787, 
	796, 808, 817, 850, 854, 856, 887, 896, 
	905, 908, 910, 943, 974, 1007, 1011, 1013, 
	1044, 1053, 1062, 1071, 1074, 1076, 1088, 1098, 
	1101, 1103, 1115, 1125, 1128, 1130, 1142, 1152, 
	1155, 1157, 1190, 1221, 1233, 1242, 1254, 1263, 
	1275, 1284, 1317, 1321, 1323, 1354, 1363, 1372, 
	1375, 1377, 1389, 1399, 1402, 1404, 1416, 1426, 
	1429, 1431, 1443, 1453, 1456, 1458, 1491, 1522, 
	1534, 1543, 1555, 1564, 1576, 1585, 1618, 1622, 
	1624, 1655, 1664, 1673, 1676, 1678, 1690, 1700, 
	1703, 1705, 1738, 1769, 1781, 1790, 1823, 1827, 
	1829, 1860, 1869, 1878, 1881, 1883, 1916, 1947, 
	1980, 1984, 1986, 2017, 2042, 2067, 2074, 2083, 
	2092, 2101, 2110, 2122, 2131, 2164, 2168, 2170, 
	2201, 2210, 2219, 2228, 2237, 2241, 2243, 2253, 
	2257, 2259, 2269, 2273, 2275, 2285, 2289, 2291, 
	2301, 2305, 2307, 2317, 2321, 2323, 2333, 2337, 
	2339, 2349, 2353, 2355, 2365, 2369, 2371, 2381, 
	2385, 2387, 2397, 2401, 2403, 2413, 2417, 2419, 
	2429, 2433, 2435, 2445, 2449, 2451, 2480, 2511, 
	2520, 2524, 2526, 2536, 2548, 2557, 2562, 2567, 
	2571, 2573, 2580, 2590, 2599, 2603, 2605, 2615, 
	2627, 2631, 2633, 2664, 2668, 2670, 2680
};

static const char _svg_path_trans_keys[] = {
	0, 13, 32, 77, 109, 9, 10, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 46, 9, 10, 48, 57, 
	13, 32, 46, 9, 10, 48, 57, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 48, 57, 13, 32, 44, 46, 9, 
	10, 48, 57, 13, 32, 46, 9, 10, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 46, 48, 57, 48, 57, 13, 
	32, 44, 69, 101, 9, 10, 48, 57, 
	13, 32, 44, 48, 49, 9, 10, 13, 
	32, 48, 49, 9, 10, 13, 32, 44, 
	9, 10, 13, 32, 44, 48, 49, 9, 
	10, 13, 32, 48, 49, 9, 10, 13, 
	32, 44, 9, 10, 13, 32, 44, 46, 
	9, 10, 43, 45, 48, 57, 46, 48, 
	57, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 44, 46, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 48, 57, 43, 45, 48, 57, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 46, 48, 57, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 43, 45, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 46, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 46, 48, 57, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 46, 48, 57, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	46, 48, 57, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 44, 46, 9, 10, 43, 
	45, 48, 57, 46, 48, 57, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 43, 45, 48, 57, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 46, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 46, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 46, 48, 57, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 46, 48, 57, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	46, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 69, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 101, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 46, 48, 57, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 43, 45, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 46, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 46, 48, 57, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 46, 48, 57, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 69, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 101, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 43, 45, 48, 57, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 43, 45, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 46, 48, 
	57, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 69, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 101, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 69, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 101, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 43, 45, 48, 57, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 0, 13, 32, 65, 67, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 0, 13, 32, 65, 67, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 13, 32, 46, 9, 10, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 69, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 101, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 43, 45, 48, 57, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 43, 45, 48, 
	57, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 44, 
	9, 10, 13, 32, 44, 9, 10, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	9, 10, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 43, 45, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	0
};

static const char _svg_path_single_lengths[] = {
	0, 5, 5, 5, 1, 0, 6, 4, 
	1, 0, 27, 25, 1, 0, 6, 4, 
	1, 0, 27, 5, 6, 5, 27, 3, 
	3, 0, 6, 4, 3, 0, 6, 4, 
	1, 0, 5, 5, 4, 3, 5, 4, 
	3, 4, 1, 0, 6, 4, 1, 0, 
	27, 25, 6, 2, 0, 4, 5, 5, 
	1, 0, 6, 4, 1, 0, 6, 4, 
	1, 0, 6, 4, 1, 0, 6, 4, 
	1, 0, 6, 4, 1, 0, 27, 25, 
	6, 5, 6, 5, 6, 5, 6, 5, 
	6, 5, 27, 2, 0, 25, 5, 5, 
	1, 0, 27, 25, 27, 2, 0, 25, 
	5, 5, 5, 1, 0, 6, 4, 1, 
	0, 6, 4, 1, 0, 6, 4, 1, 
	0, 27, 25, 6, 5, 6, 5, 6, 
	5, 27, 2, 0, 25, 5, 5, 1, 
	0, 6, 4, 1, 0, 6, 4, 1, 
	0, 6, 4, 1, 0, 27, 25, 6, 
	5, 6, 5, 6, 5, 27, 2, 0, 
	25, 5, 5, 1, 0, 6, 4, 1, 
	0, 27, 25, 6, 5, 27, 2, 0, 
	25, 5, 5, 1, 0, 27, 25, 27, 
	2, 0, 25, 23, 23, 3, 5, 5, 
	5, 5, 6, 5, 27, 2, 0, 25, 
	5, 5, 5, 5, 2, 0, 4, 2, 
	0, 4, 2, 0, 4, 2, 0, 4, 
	2, 0, 4, 2, 0, 4, 2, 0, 
	4, 2, 0, 4, 2, 0, 4, 2, 
	0, 4, 2, 0, 4, 2, 0, 4, 
	2, 0, 4, 2, 0, 25, 27, 5, 
	2, 0, 4, 6, 5, 3, 3, 2, 
	0, 3, 6, 5, 2, 0, 4, 6, 
	2, 0, 25, 2, 0, 4, 0
};

static const char _svg_path_range_lengths[] = {
	0, 1, 2, 2, 1, 1, 3, 3, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	1, 1, 3, 2, 3, 2, 3, 2, 
	2, 1, 2, 2, 2, 1, 3, 3, 
	1, 1, 2, 1, 1, 1, 1, 1, 
	1, 3, 1, 1, 3, 3, 1, 1, 
	2, 2, 2, 1, 1, 2, 2, 2, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	3, 2, 3, 2, 3, 2, 3, 2, 
	3, 2, 3, 1, 1, 3, 2, 2, 
	1, 1, 3, 3, 3, 1, 1, 3, 
	2, 2, 2, 1, 1, 3, 3, 1, 
	1, 3, 3, 1, 1, 3, 3, 1, 
	1, 3, 3, 3, 2, 3, 2, 3, 
	2, 3, 1, 1, 3, 2, 2, 1, 
	1, 3, 3, 1, 1, 3, 3, 1, 
	1, 3, 3, 1, 1, 3, 3, 3, 
	2, 3, 2, 3, 2, 3, 1, 1, 
	3, 2, 2, 1, 1, 3, 3, 1, 
	1, 3, 3, 3, 2, 3, 1, 1, 
	3, 2, 2, 1, 1, 3, 3, 3, 
	1, 1, 3, 1, 1, 2, 2, 2, 
	2, 2, 3, 2, 3, 1, 1, 3, 
	2, 2, 2, 2, 1, 1, 3, 1, 
	1, 3, 1, 1, 3, 1, 1, 3, 
	1, 1, 3, 1, 1, 3, 1, 1, 
	3, 1, 1, 3, 1, 1, 3, 1, 
	1, 3, 1, 1, 3, 1, 1, 3, 
	1, 1, 3, 1, 1, 2, 2, 2, 
	1, 1, 3, 3, 2, 1, 1, 1, 
	1, 2, 2, 2, 1, 1, 3, 3, 
	1, 1, 3, 1, 1, 3, 0
};

static const short _svg_path_index_offsets[] = {
	0, 0, 7, 15, 23, 26, 28, 38, 
	46, 49, 51, 82, 111, 114, 116, 126, 
	134, 137, 139, 170, 178, 188, 196, 227, 
	233, 239, 241, 250, 257, 263, 265, 275, 
	283, 286, 288, 296, 303, 309, 314, 321, 
	327, 332, 340, 343, 345, 355, 363, 366, 
	368, 398, 426, 435, 439, 441, 448, 456, 
	464, 467, 469, 479, 487, 490, 492, 502, 
	510, 513, 515, 525, 533, 536, 538, 548, 
	556, 559, 561, 571, 579, 582, 584, 615, 
	644, 654, 662, 672, 680, 690, 698, 708, 
	716, 726, 734, 765, 769, 771, 800, 808, 
	816, 819, 821, 852, 881, 912, 916, 918, 
	947, 955, 963, 971, 974, 976, 986, 994, 
	997, 999, 1009, 1017, 1020, 1022, 1032, 1040, 
	1043, 1045, 1076, 1105, 1115, 1123, 1133, 1141, 
	1151, 1159, 1190, 1194, 1196, 1225, 1233, 1241, 
	1244, 1246, 1256, 1264, 1267, 1269, 1279, 1287, 
	1290, 1292, 1302, 1310, 1313, 1315, 1346, 1375, 
	1385, 1393, 1403, 1411, 1421, 1429, 1460, 1464, 
	1466, 1495, 1503, 1511, 1514, 1516, 1526, 1534, 
	1537, 1539, 1570, 1599, 1609, 1617, 1648, 1652, 
	1654, 1683, 1691, 1699, 1702, 1704, 1735, 1764, 
	1795, 1799, 1801, 1830, 1855, 1880, 1886, 1894, 
	1902, 1910, 1918, 1928, 1936, 1967, 1971, 1973, 
	2002, 2010, 2018, 2026, 2034, 2038, 2040, 2048, 
	2052, 2054, 2062, 2066, 2068, 2076, 2080, 2082, 
	2090, 2094, 2096, 2104, 2108, 2110, 2118, 2122, 
	2124, 2132, 2136, 2138, 2146, 2150, 2152, 2160, 
	2164, 2166, 2174, 2178, 2180, 2188, 2192, 2194, 
	2202, 2206, 2208, 2216, 2220, 2222, 2250, 2280, 
	2288, 2292, 2294, 2302, 2312, 2320, 2325, 2330, 
	2334, 2336, 2342, 2351, 2359, 2363, 2365, 2373, 
	2383, 2387, 2389, 2418, 2422, 2424, 2432
};

static const short _svg_path_indicies[] = {
	0, 2, 2, 3, 4, 2, 1, 5, 
	5, 6, 6, 7, 5, 8, 1, 9, 
	9, 10, 10, 11, 9, 12, 1, 13, 
	14, 1, 15, 1, 16, 16, 18, 19, 
	20, 20, 16, 17, 15, 1, 21, 21, 
	23, 24, 21, 22, 25, 1, 26, 27, 
	1, 28, 1, 29, 30, 30, 32, 33, 
	34, 35, 36, 37, 38, 39, 40, 41, 
	42, 43, 44, 45, 46, 36, 47, 48, 
	49, 50, 51, 52, 53, 44, 30, 31, 
	28, 1, 0, 54, 54, 56, 57, 59, 
	60, 61, 62, 3, 63, 64, 65, 66, 
	67, 68, 69, 70, 71, 4, 72, 73, 
	74, 75, 67, 54, 55, 58, 1, 76, 
	77, 1, 78, 1, 79, 79, 81, 82, 
	83, 83, 79, 80, 78, 1, 84, 84, 
	86, 87, 84, 85, 88, 1, 89, 90, 
	1, 91, 1, 92, 93, 93, 95, 96, 
	97, 98, 99, 100, 101, 102, 103, 104, 
	105, 106, 107, 108, 109, 99, 110, 111, 
	112, 113, 114, 115, 116, 107, 93, 94, 
	91, 1, 56, 56, 55, 55, 57, 56, 
	58, 1, 79, 79, 81, 78, 83, 83, 
	79, 80, 77, 1, 86, 86, 85, 85, 
	87, 86, 88, 1, 92, 93, 93, 95, 
	91, 97, 98, 99, 100, 101, 102, 103, 
	104, 105, 106, 107, 108, 109, 99, 110, 
	111, 112, 113, 114, 115, 116, 107, 93, 
	94, 90, 1, 117, 117, 118, 117, 119, 
	1, 120, 120, 121, 120, 122, 1, 123, 
	1, 124, 124, 125, 126, 127, 127, 124, 
	123, 1, 128, 128, 129, 130, 128, 131, 
	1, 129, 129, 130, 129, 131, 1, 132, 
	1, 133, 133, 135, 136, 137, 137, 133, 
	134, 132, 1, 138, 138, 140, 141, 138, 
	139, 142, 1, 143, 144, 1, 145, 1, 
	146, 146, 147, 148, 148, 146, 145, 1, 
	149, 149, 150, 151, 152, 149, 1, 150, 
	150, 151, 152, 150, 1, 153, 153, 154, 
	153, 1, 155, 155, 156, 157, 158, 155, 
	1, 156, 156, 157, 158, 156, 1, 159, 
	159, 160, 159, 1, 161, 161, 163, 164, 
	161, 162, 165, 1, 166, 167, 1, 168, 
	1, 169, 169, 171, 172, 173, 173, 169, 
	170, 168, 1, 174, 174, 176, 177, 174, 
	175, 178, 1, 179, 180, 1, 181, 1, 
	182, 183, 183, 184, 185, 186, 187, 188, 
	189, 190, 191, 192, 193, 194, 195, 196, 
	197, 198, 188, 199, 200, 201, 202, 203, 
	204, 205, 196, 183, 181, 1, 0, 206, 
	206, 120, 121, 59, 60, 61, 62, 3, 
	63, 64, 65, 66, 67, 68, 69, 70, 
	71, 4, 72, 73, 74, 75, 67, 206, 
	122, 1, 124, 124, 125, 123, 127, 127, 
	124, 207, 1, 208, 208, 209, 1, 209, 
	1, 124, 124, 125, 126, 124, 209, 1, 
	210, 210, 211, 211, 212, 210, 213, 1, 
	214, 214, 215, 215, 216, 214, 217, 1, 
	218, 219, 1, 220, 1, 221, 221, 223, 
	224, 225, 225, 221, 222, 220, 1, 226, 
	226, 228, 229, 226, 227, 230, 1, 231, 
	232, 1, 233, 1, 234, 234, 236, 237, 
	238, 238, 234, 235, 233, 1, 239, 239, 
	241, 242, 239, 240, 243, 1, 244, 245, 
	1, 246, 1, 247, 247, 249, 250, 251, 
	251, 247, 248, 246, 1, 252, 252, 254, 
	255, 252, 253, 256, 1, 257, 258, 1, 
	259, 1, 260, 260, 262, 263, 264, 264, 
	260, 261, 259, 1, 265, 265, 267, 268, 
	265, 266, 269, 1, 270, 271, 1, 272, 
	1, 273, 273, 275, 276, 277, 277, 273, 
	274, 272, 1, 278, 278, 280, 281, 278, 
	279, 282, 1, 283, 284, 1, 285, 1, 
	286, 287, 287, 289, 290, 291, 292, 293, 
	294, 295, 296, 297, 298, 299, 300, 301, 
	302, 303, 293, 304, 305, 306, 307, 308, 
	309, 310, 301, 287, 288, 285, 1, 0, 
	311, 311, 214, 216, 59, 60, 61, 62, 
	3, 63, 64, 65, 66, 67, 68, 69, 
	70, 71, 4, 72, 73, 74, 75, 67, 
	311, 215, 217, 1, 221, 221, 223, 220, 
	225, 225, 221, 222, 219, 1, 228, 228, 
	227, 227, 229, 228, 230, 1, 234, 234, 
	236, 233, 238, 238, 234, 235, 232, 1, 
	241, 241, 240, 240, 242, 241, 243, 1, 
	247, 247, 249, 246, 251, 251, 247, 248, 
	245, 1, 254, 254, 253, 253, 255, 254, 
	256, 1, 260, 260, 262, 259, 264, 264, 
	260, 261, 258, 1, 267, 267, 266, 266, 
	268, 267, 269, 1, 273, 273, 275, 272, 
	277, 277, 273, 274, 271, 1, 280, 280, 
	279, 279, 281, 280, 282, 1, 286, 287, 
	287, 289, 285, 291, 292, 293, 294, 295, 
	296, 297, 298, 299, 300, 301, 302, 303, 
	293, 304, 305, 306, 307, 308, 309, 310, 
	301, 287, 288, 284, 1, 312, 312, 313, 
	1, 313, 1, 286, 287, 287, 289, 290, 
	291, 292, 294, 295, 296, 297, 298, 299, 
	300, 301, 302, 303, 304, 305, 306, 307, 
	308, 309, 310, 301, 287, 288, 313, 1, 
	314, 314, 315, 315, 316, 314, 317, 1, 
	318, 318, 319, 319, 320, 318, 321, 1, 
	322, 323, 1, 324, 1, 325, 326, 326, 
	328, 329, 330, 331, 332, 333, 334, 335, 
	336, 337, 338, 339, 340, 341, 342, 332, 
	343, 344, 345, 346, 347, 348, 349, 340, 
	326, 327, 324, 1, 0, 350, 350, 318, 
	320, 59, 60, 61, 62, 3, 63, 64, 
	65, 66, 67, 68, 69, 70, 71, 4, 
	72, 73, 74, 75, 67, 350, 319, 321, 
	1, 325, 326, 326, 328, 324, 330, 331, 
	332, 333, 334, 335, 336, 337, 338, 339, 
	340, 341, 342, 332, 343, 344, 345, 346, 
	347, 348, 349, 340, 326, 327, 323, 1, 
	351, 351, 352, 1, 352, 1, 325, 326, 
	326, 328, 329, 330, 331, 333, 334, 335, 
	336, 337, 338, 339, 340, 341, 342, 343, 
	344, 345, 346, 347, 348, 349, 340, 326, 
	327, 352, 1, 353, 353, 354, 354, 355, 
	353, 356, 1, 357, 357, 358, 358, 359, 
	357, 360, 1, 361, 361, 362, 362, 363, 
	361, 364, 1, 365, 366, 1, 367, 1, 
	368, 368, 370, 371, 372, 372, 368, 369, 
	367, 1, 373, 373, 375, 376, 373, 374, 
	377, 1, 378, 379, 1, 380, 1, 381, 
	381, 383, 384, 385, 385, 381, 382, 380, 
	1, 386, 386, 388, 389, 386, 387, 390, 
	1, 391, 392, 1, 393, 1, 394, 394, 
	396, 397, 398, 398, 394, 395, 393, 1, 
	399, 399, 401, 402, 399, 400, 403, 1, 
	404, 405, 1, 406, 1, 407, 408, 408, 
	410, 411, 412, 413, 414, 415, 416, 417, 
	418, 419, 420, 421, 422, 423, 424, 414, 
	425, 426, 427, 428, 429, 430, 431, 422, 
	408, 409, 406, 1, 0, 432, 432, 361, 
	363, 59, 60, 61, 62, 3, 63, 64, 
	65, 66, 67, 68, 69, 70, 71, 4, 
	72, 73, 74, 75, 67, 432, 362, 364, 
	1, 368, 368, 370, 367, 372, 372, 368, 
	369, 366, 1, 375, 375, 374, 374, 376, 
	375, 377, 1, 381, 381, 383, 380, 385, 
	385, 381, 382, 379, 1, 388, 388, 387, 
	387, 389, 388, 390, 1, 394, 394, 396, 
	393, 398, 398, 394, 395, 392, 1, 401, 
	401, 400, 400, 402, 401, 403, 1, 407, 
	408, 408, 410, 406, 412, 413, 414, 415, 
	416, 417, 418, 419, 420, 421, 422, 423, 
	424, 414, 425, 426, 427, 428, 429, 430, 
	431, 422, 408, 409, 405, 1, 433, 433, 
	434, 1, 434, 1, 407, 408, 408, 410, 
	411, 412, 413, 415, 416, 417, 418, 419, 
	420, 421, 422, 423, 424, 425, 426, 427, 
	428, 429, 430, 431, 422, 408, 409, 434, 
	1, 435, 435, 436, 436, 437, 435, 438, 
	1, 439, 439, 440, 440, 441, 439, 442, 
	1, 443, 444, 1, 445, 1, 446, 446, 
	448, 449, 450, 450, 446, 447, 445, 1, 
	451, 451, 453, 454, 451, 452, 455, 1, 
	456, 457, 1, 458, 1, 459, 459, 461, 
	462, 463, 463, 459, 460, 458, 1, 464, 
	464, 466, 467, 464, 465, 468, 1, 469, 
	470, 1, 471, 1, 472, 472, 474, 475, 
	476, 476, 472, 473, 471, 1, 477, 477, 
	479, 480, 477, 478, 481, 1, 482, 483, 
	1, 484, 1, 485, 486, 486, 488, 489, 
	490, 491, 492, 493, 494, 495, 496, 497, 
	498, 499, 500, 501, 502, 492, 503, 504, 
	505, 506, 507, 508, 509, 500, 486, 487, 
	484, 1, 0, 510, 510, 439, 441, 59, 
	60, 61, 62, 3, 63, 64, 65, 66, 
	67, 68, 69, 70, 71, 4, 72, 73, 
	74, 75, 67, 510, 440, 442, 1, 446, 
	446, 448, 445, 450, 450, 446, 447, 444, 
	1, 453, 453, 452, 452, 454, 453, 455, 
	1, 459, 459, 461, 458, 463, 463, 459, 
	460, 457, 1, 466, 466, 465, 465, 467, 
	466, 468, 1, 472, 472, 474, 471, 476, 
	476, 472, 473, 470, 1, 479, 479, 478, 
	478, 480, 479, 481, 1, 485, 486, 486, 
	488, 484, 490, 491, 492, 493, 494, 495, 
	496, 497, 498, 499, 500, 501, 502, 492, 
	503, 504, 505, 506, 507, 508, 509, 500, 
	486, 487, 483, 1, 511, 511, 512, 1, 
	512, 1, 485, 486, 486, 488, 489, 490, 
	491, 493, 494, 495, 496, 497, 498, 499, 
	500, 501, 502, 503, 504, 505, 506, 507, 
	508, 509, 500, 486, 487, 512, 1, 513, 
	513, 514, 514, 515, 513, 516, 1, 517, 
	517, 518, 518, 519, 517, 520, 1, 521, 
	522, 1, 523, 1, 524, 524, 526, 527, 
	528, 528, 524, 525, 523, 1, 529, 529, 
	531, 532, 529, 530, 533, 1, 534, 535, 
	1, 536, 1, 537, 538, 538, 540, 541, 
	542, 543, 544, 545, 546, 547, 548, 549, 
	550, 551, 552, 553, 554, 544, 555, 556, 
	557, 558, 559, 560, 561, 552, 538, 539, 
	536, 1, 0, 562, 562, 517, 519, 59, 
	60, 61, 62, 3, 63, 64, 65, 66, 
	67, 68, 69, 70, 71, 4, 72, 73, 
	74, 75, 67, 562, 518, 520, 1, 524, 
	524, 526, 523, 528, 528, 524, 525, 522, 
	1, 531, 531, 530, 530, 532, 531, 533, 
	1, 537, 538, 538, 540, 536, 542, 543, 
	544, 545, 546, 547, 548, 549, 550, 551, 
	552, 553, 554, 544, 555, 556, 557, 558, 
	559, 560, 561, 552, 538, 539, 535, 1, 
	563, 563, 564, 1, 564, 1, 537, 538, 
	538, 540, 541, 542, 543, 545, 546, 547, 
	548, 549, 550, 551, 552, 553, 554, 555, 
	556, 557, 558, 559, 560, 561, 552, 538, 
	539, 564, 1, 565, 565, 566, 566, 567, 
	565, 568, 1, 569, 569, 570, 570, 571, 
	569, 572, 1, 573, 574, 1, 575, 1, 
	576, 577, 577, 579, 580, 581, 582, 583, 
	584, 585, 586, 587, 588, 589, 590, 591, 
	592, 593, 583, 594, 595, 596, 597, 598, 
	599, 600, 591, 577, 578, 575, 1, 0, 
	601, 601, 569, 571, 59, 60, 61, 62, 
	3, 63, 64, 65, 66, 67, 68, 69, 
	70, 71, 4, 72, 73, 74, 75, 67, 
	601, 570, 572, 1, 576, 577, 577, 579, 
	575, 581, 582, 583, 584, 585, 586, 587, 
	588, 589, 590, 591, 592, 593, 583, 594, 
	595, 596, 597, 598, 599, 600, 591, 577, 
	578, 574, 1, 602, 602, 603, 1, 603, 
	1, 576, 577, 577, 579, 580, 581, 582, 
	584, 585, 586, 587, 588, 589, 590, 591, 
	592, 593, 594, 595, 596, 597, 598, 599, 
	600, 591, 577, 578, 603, 1, 604, 605, 
	605, 606, 607, 608, 609, 610, 611, 612, 
	613, 614, 615, 616, 617, 618, 619, 620, 
	621, 622, 623, 624, 615, 605, 1, 0, 
	625, 625, 59, 60, 61, 62, 3, 63, 
	64, 65, 66, 67, 68, 69, 70, 71, 
	4, 72, 73, 74, 75, 67, 625, 1, 
	626, 626, 627, 626, 628, 1, 629, 629, 
	630, 630, 631, 629, 632, 1, 633, 633, 
	634, 634, 635, 633, 636, 1, 637, 637, 
	638, 638, 639, 637, 640, 1, 641, 641, 
	642, 642, 643, 641, 644, 1, 16, 16, 
	18, 15, 20, 20, 16, 17, 14, 1, 
	23, 23, 22, 22, 24, 23, 25, 1, 
	29, 30, 30, 32, 28, 34, 35, 36, 
	37, 38, 39, 40, 41, 42, 43, 44, 
	45, 46, 36, 47, 48, 49, 50, 51, 
	52, 53, 44, 30, 31, 27, 1, 645, 
	645, 646, 1, 646, 1, 29, 30, 30, 
	32, 33, 34, 35, 37, 38, 39, 40, 
	41, 42, 43, 44, 45, 46, 47, 48, 
	49, 50, 51, 52, 53, 44, 30, 31, 
	646, 1, 647, 647, 648, 648, 649, 647, 
	650, 1, 651, 651, 652, 652, 653, 651, 
	654, 1, 655, 655, 656, 656, 657, 655, 
	658, 1, 659, 659, 660, 660, 661, 659, 
	662, 1, 663, 663, 664, 1, 664, 1, 
	16, 16, 18, 19, 16, 17, 664, 1, 
	665, 665, 666, 1, 666, 1, 524, 524, 
	526, 527, 524, 525, 666, 1, 667, 667, 
	668, 1, 668, 1, 472, 472, 474, 475, 
	472, 473, 668, 1, 669, 669, 670, 1, 
	670, 1, 459, 459, 461, 462, 459, 460, 
	670, 1, 671, 671, 672, 1, 672, 1, 
	446, 446, 448, 449, 446, 447, 672, 1, 
	673, 673, 674, 1, 674, 1, 394, 394, 
	396, 397, 394, 395, 674, 1, 675, 675, 
	676, 1, 676, 1, 381, 381, 383, 384, 
	381, 382, 676, 1, 677, 677, 678, 1, 
	678, 1, 368, 368, 370, 371, 368, 369, 
	678, 1, 679, 679, 680, 1, 680, 1, 
	273, 273, 275, 276, 273, 274, 680, 1, 
	681, 681, 682, 1, 682, 1, 260, 260, 
	262, 263, 260, 261, 682, 1, 683, 683, 
	684, 1, 684, 1, 247, 247, 249, 250, 
	247, 248, 684, 1, 685, 685, 686, 1, 
	686, 1, 234, 234, 236, 237, 234, 235, 
	686, 1, 687, 687, 688, 1, 688, 1, 
	221, 221, 223, 224, 221, 222, 688, 1, 
	689, 689, 690, 1, 690, 1, 182, 183, 
	183, 184, 185, 186, 187, 189, 190, 191, 
	192, 193, 194, 195, 196, 197, 198, 199, 
	200, 201, 202, 203, 204, 205, 196, 183, 
	690, 1, 182, 183, 183, 184, 181, 186, 
	187, 188, 189, 190, 191, 192, 193, 194, 
	195, 196, 197, 198, 188, 199, 200, 201, 
	202, 203, 204, 205, 196, 183, 180, 1, 
	176, 176, 175, 175, 177, 176, 178, 1, 
	691, 691, 692, 1, 692, 1, 169, 169, 
	171, 172, 169, 170, 692, 1, 169, 169, 
	171, 168, 173, 173, 169, 170, 167, 1, 
	163, 163, 162, 162, 164, 163, 165, 1, 
	693, 693, 694, 693, 1, 695, 695, 696, 
	695, 1, 697, 697, 698, 1, 698, 1, 
	146, 146, 147, 146, 698, 1, 146, 146, 
	147, 145, 148, 148, 146, 144, 1, 140, 
	140, 139, 139, 141, 140, 142, 1, 699, 
	699, 700, 1, 700, 1, 133, 133, 135, 
	136, 133, 134, 700, 1, 133, 133, 135, 
	132, 137, 137, 133, 134, 701, 1, 702, 
	702, 703, 1, 703, 1, 92, 93, 93, 
	95, 96, 97, 98, 100, 101, 102, 103, 
	104, 105, 106, 107, 108, 109, 110, 111, 
	112, 113, 114, 115, 116, 107, 93, 94, 
	703, 1, 704, 704, 705, 1, 705, 1, 
	79, 79, 81, 82, 79, 80, 705, 1, 
	1, 0
};

static const short _svg_path_trans_targs[] = {
	270, 0, 1, 2, 193, 3, 4, 5, 
	194, 3, 4, 5, 194, 5, 194, 6, 
	7, 8, 195, 9, 204, 7, 8, 195, 
	9, 196, 9, 196, 10, 270, 11, 12, 
	19, 13, 23, 54, 197, 94, 104, 2, 
	105, 133, 161, 177, 187, 189, 190, 191, 
	192, 193, 200, 201, 202, 203, 11, 12, 
	19, 13, 20, 23, 54, 94, 104, 105, 
	133, 161, 177, 187, 189, 190, 191, 192, 
	200, 201, 202, 203, 13, 20, 14, 15, 
	16, 21, 17, 267, 15, 16, 21, 17, 
	22, 17, 22, 18, 270, 11, 12, 19, 
	13, 23, 54, 264, 94, 104, 2, 105, 
	133, 161, 177, 187, 189, 190, 191, 192, 
	193, 200, 201, 202, 203, 24, 25, 50, 
	24, 25, 50, 26, 27, 28, 29, 51, 
	27, 28, 29, 263, 30, 31, 32, 259, 
	33, 260, 31, 32, 259, 33, 258, 33, 
	258, 34, 35, 36, 255, 35, 36, 37, 
	254, 38, 39, 38, 39, 40, 253, 41, 
	252, 41, 42, 252, 43, 251, 43, 251, 
	44, 45, 46, 247, 47, 248, 45, 46, 
	247, 47, 246, 47, 246, 48, 270, 49, 
	24, 25, 23, 54, 243, 94, 104, 2, 
	105, 133, 161, 177, 187, 189, 190, 191, 
	192, 193, 200, 201, 202, 203, 49, 50, 
	52, 53, 55, 56, 57, 80, 55, 56, 
	57, 80, 57, 80, 58, 59, 60, 81, 
	61, 240, 59, 60, 81, 61, 82, 61, 
	82, 62, 63, 64, 83, 65, 237, 63, 
	64, 83, 65, 84, 65, 84, 66, 67, 
	68, 85, 69, 234, 67, 68, 85, 69, 
	86, 69, 86, 70, 71, 72, 87, 73, 
	231, 71, 72, 87, 73, 88, 73, 88, 
	74, 75, 76, 89, 77, 228, 75, 76, 
	89, 77, 90, 77, 90, 78, 270, 79, 
	56, 55, 57, 23, 54, 91, 94, 104, 
	2, 105, 133, 161, 177, 187, 189, 190, 
	191, 192, 193, 200, 201, 202, 203, 79, 
	92, 93, 95, 96, 97, 100, 95, 96, 
	97, 100, 97, 100, 98, 270, 99, 96, 
	95, 97, 23, 54, 101, 94, 104, 2, 
	105, 133, 161, 177, 187, 189, 190, 191, 
	192, 193, 200, 201, 202, 203, 99, 102, 
	103, 19, 12, 13, 20, 106, 107, 108, 
	123, 106, 107, 108, 123, 108, 123, 109, 
	110, 111, 124, 112, 225, 110, 111, 124, 
	112, 125, 112, 125, 113, 114, 115, 126, 
	116, 222, 114, 115, 126, 116, 127, 116, 
	127, 117, 118, 119, 128, 120, 219, 118, 
	119, 128, 120, 129, 120, 129, 121, 270, 
	122, 107, 106, 108, 23, 54, 130, 94, 
	104, 2, 105, 133, 161, 177, 187, 189, 
	190, 191, 192, 193, 200, 201, 202, 203, 
	122, 131, 132, 134, 135, 136, 151, 134, 
	135, 136, 151, 136, 151, 137, 138, 139, 
	152, 140, 216, 138, 139, 152, 140, 153, 
	140, 153, 141, 142, 143, 154, 144, 213, 
	142, 143, 154, 144, 155, 144, 155, 145, 
	146, 147, 156, 148, 210, 146, 147, 156, 
	148, 157, 148, 157, 149, 270, 150, 135, 
	134, 136, 23, 54, 158, 94, 104, 2, 
	105, 133, 161, 177, 187, 189, 190, 191, 
	192, 193, 200, 201, 202, 203, 150, 159, 
	160, 162, 163, 164, 171, 162, 163, 164, 
	171, 164, 171, 165, 166, 167, 172, 168, 
	207, 166, 167, 172, 168, 173, 168, 173, 
	169, 270, 170, 163, 162, 164, 23, 54, 
	174, 94, 104, 2, 105, 133, 161, 177, 
	187, 189, 190, 191, 192, 193, 200, 201, 
	202, 203, 170, 175, 176, 178, 179, 180, 
	183, 178, 179, 180, 183, 180, 183, 181, 
	270, 182, 179, 178, 180, 23, 54, 184, 
	94, 104, 2, 105, 133, 161, 177, 187, 
	189, 190, 191, 192, 193, 200, 201, 202, 
	203, 182, 185, 186, 270, 188, 23, 54, 
	94, 104, 2, 105, 133, 161, 177, 187, 
	189, 190, 191, 192, 193, 200, 201, 202, 
	203, 188, 24, 25, 50, 55, 56, 57, 
	80, 95, 96, 97, 100, 19, 12, 13, 
	20, 3, 4, 5, 194, 198, 199, 106, 
	107, 108, 123, 134, 135, 136, 151, 162, 
	163, 164, 171, 178, 179, 180, 183, 205, 
	206, 208, 209, 211, 212, 214, 215, 217, 
	218, 220, 221, 223, 224, 226, 227, 229, 
	230, 232, 233, 235, 236, 238, 239, 241, 
	242, 244, 245, 249, 250, 41, 252, 38, 
	39, 256, 257, 261, 262, 263, 265, 266, 
	268, 269
};

static const char _svg_path_trans_actions[] = {
	15, 0, 0, 0, 0, 9, 47, 47, 
	47, 0, 1, 1, 1, 0, 0, 0, 
	3, 17, 3, 17, 0, 0, 1, 0, 
	1, 1, 0, 0, 0, 60, 20, 56, 
	20, 56, 20, 20, 0, 20, 20, 20, 
	20, 20, 20, 20, 20, 20, 20, 20, 
	20, 20, 20, 20, 20, 20, 0, 1, 
	0, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 3, 
	17, 3, 17, 0, 0, 1, 0, 1, 
	1, 0, 0, 0, 68, 23, 64, 23, 
	64, 23, 23, 0, 23, 23, 23, 23, 
	23, 23, 23, 23, 23, 23, 23, 23, 
	23, 23, 23, 23, 23, 9, 47, 47, 
	0, 1, 1, 0, 3, 3, 17, 0, 
	0, 0, 1, 1, 0, 3, 17, 3, 
	17, 0, 0, 1, 0, 1, 1, 0, 
	0, 0, 3, 3, 0, 0, 0, 0, 
	0, 7, 7, 0, 0, 0, 0, 7, 
	7, 0, 1, 0, 1, 1, 0, 0, 
	0, 3, 17, 3, 17, 0, 0, 1, 
	0, 1, 1, 0, 0, 0, 124, 44, 
	44, 120, 44, 44, 0, 44, 44, 44, 
	44, 44, 44, 44, 44, 44, 44, 44, 
	44, 44, 44, 44, 44, 44, 0, 0, 
	0, 0, 9, 47, 47, 47, 0, 1, 
	1, 1, 0, 0, 0, 3, 17, 3, 
	17, 0, 0, 1, 0, 1, 1, 0, 
	0, 0, 3, 17, 3, 17, 0, 0, 
	1, 0, 1, 1, 0, 0, 0, 3, 
	17, 3, 17, 0, 0, 1, 0, 1, 
	1, 0, 0, 0, 3, 17, 3, 17, 
	0, 0, 1, 0, 1, 1, 0, 0, 
	0, 3, 17, 3, 17, 0, 0, 1, 
	0, 1, 1, 0, 0, 0, 92, 32, 
	88, 32, 88, 32, 32, 0, 32, 32, 
	32, 32, 32, 32, 32, 32, 32, 32, 
	32, 32, 32, 32, 32, 32, 32, 0, 
	0, 0, 9, 47, 47, 47, 0, 1, 
	1, 1, 0, 0, 0, 76, 26, 72, 
	26, 72, 26, 26, 0, 26, 26, 26, 
	26, 26, 26, 26, 26, 26, 26, 26, 
	26, 26, 26, 26, 26, 26, 0, 0, 
	0, 9, 47, 47, 47, 9, 47, 47, 
	47, 0, 1, 1, 1, 0, 0, 0, 
	3, 17, 3, 17, 0, 0, 1, 0, 
	1, 1, 0, 0, 0, 3, 17, 3, 
	17, 0, 0, 1, 0, 1, 1, 0, 
	0, 0, 3, 17, 3, 17, 0, 0, 
	1, 0, 1, 1, 0, 0, 0, 108, 
	38, 104, 38, 104, 38, 38, 0, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	0, 0, 0, 9, 47, 47, 47, 0, 
	1, 1, 1, 0, 0, 0, 3, 17, 
	3, 17, 0, 0, 1, 0, 1, 1, 
	0, 0, 0, 3, 17, 3, 17, 0, 
	0, 1, 0, 1, 1, 0, 0, 0, 
	3, 17, 3, 17, 0, 0, 1, 0, 
	1, 1, 0, 0, 0, 100, 35, 96, 
	35, 96, 35, 35, 0, 35, 35, 35, 
	35, 35, 35, 35, 35, 35, 35, 35, 
	35, 35, 35, 35, 35, 35, 0, 0, 
	0, 9, 47, 47, 47, 0, 1, 1, 
	1, 0, 0, 0, 3, 17, 3, 17, 
	0, 0, 1, 0, 1, 1, 0, 0, 
	0, 116, 41, 112, 41, 112, 41, 41, 
	0, 41, 41, 41, 41, 41, 41, 41, 
	41, 41, 41, 41, 41, 41, 41, 41, 
	41, 41, 0, 0, 0, 9, 47, 47, 
	47, 0, 1, 1, 1, 0, 0, 0, 
	84, 29, 80, 29, 80, 29, 29, 0, 
	29, 29, 29, 29, 29, 29, 29, 29, 
	29, 29, 29, 29, 29, 29, 29, 29, 
	29, 0, 0, 0, 53, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 0, 11, 50, 50, 11, 50, 50, 
	50, 11, 50, 50, 50, 11, 50, 50, 
	50, 11, 50, 50, 50, 0, 0, 11, 
	50, 50, 50, 11, 50, 50, 50, 11, 
	50, 50, 50, 11, 50, 50, 50, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 5, 
	5, 0, 0, 0, 0, 0, 0, 0, 
	0, 0
};

static const int svg_path_start = 1;
static const int svg_path_first_final = 270;

//static const int svg_path_en_main = 1;


void Parser::parse(char const *str)
throw(SVGPathParseError)
{
    char const *p = str;
    char const *start = NULL;
    int cs;

    _reset();

	{
	cs = svg_path_start;
	}


	{
	int _klen;
	unsigned int _trans;
	const char *_acts;
	unsigned int _nacts;
	const char *_keys;

	if ( cs == 0 )
		goto _out;
_resume:
	_keys = _svg_path_trans_keys + _svg_path_key_offsets[cs];
	_trans = _svg_path_index_offsets[cs];

	_klen = _svg_path_single_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( (*p) < *_mid )
				_upper = _mid - 1;
			else if ( (*p) > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (unsigned int)(_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _svg_path_range_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( (*p) < _mid[0] )
				_upper = _mid - 2;
			else if ( (*p) > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += (unsigned int)((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _svg_path_indicies[_trans];
	cs = _svg_path_trans_targs[_trans];

	if ( _svg_path_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _svg_path_actions + _svg_path_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
	{
            start = p;
        }
	break;
	case 1:

	{
            char const *end=p;
            std::string buf(start, end);
            _push(g_ascii_strtod(buf.c_str(), NULL));
            start = NULL;
        }
	break;
	case 2:

	{
            _push(1.0);
        }
	break;
	case 3:

	{
            _push(0.0);
        }
	break;
	case 4:

	{
            _absolute = true;
        }
	break;
	case 5:
	{
            _absolute = false;
        }
	break;
	case 6:
	{
            _moveTo(_pop_point());
        }
	break;
	case 7:
	{
            _lineTo(_pop_point());
        }
	break;
	case 8:
	{
            _hlineTo(Point(_pop_coord(X), _current[Y]));
        }
	break;
	case 9:
	{
            _vlineTo(Point(_current[X], _pop_coord(Y)));
        }
	break;
	case 10:
	{
            Point p = _pop_point();
            Point c1 = _pop_point();
            Point c0 = _pop_point();
            _curveTo(c0, c1, p);
        }
	break;
	case 11:
	{
            Point p = _pop_point();
            Point c1 = _pop_point();
            _curveTo(_cubic_tangent, c1, p);
        }
	break;
	case 12:
	{
            Point p = _pop_point();
            Point c = _pop_point();
            _quadTo(c, p);
        }
	break;
	case 13:
	{
            Point p = _pop_point();
            _quadTo(_quad_tangent, p);
        }
	break;
	case 14:
	{
            Point point = _pop_point();
            bool sweep = _pop_flag();
            bool large_arc = _pop_flag();
            double angle = deg_to_rad(_pop());
            double ry = _pop();
            double rx = _pop();

            _arcTo(rx, ry, angle, large_arc, sweep, point);
        }
	break;
	case 15:
	{
            _closePath();
        }
	break;
	case 16:
	{{p++; goto _out; }}
	break;
		}
	}

_again:
	if ( cs == 0 )
		goto _out;
	p += 1;
	goto _resume;
	_out: {}
	}


    if ( cs < svg_path_first_final ) {
        throw SVGPathParseError();
    }
}

}

void parse_svg_path(char const *str, PathSink &sink)
throw(SVGPathParseError)
{
    Parser parser(sink);
    parser.parse(str);
    sink.flush();
}

}

/*
  Local Variables:
  mode:c++
  c-file-style:"stroustrup"
  c-file-offsets:((innamespace . 0)(inline-open . 0)(case-label . +))
  indent-tabs-mode:nil
  fill-column:99
  End:
*/
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:encoding=utf-8:textwidth=99 :
