/* Copyright (C) 2002 Bjoern Beutel. */

/* Description. =============================================================*/

/* Read in and display Malaga Results. */

/* Includes. ================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <setjmp.h>
#include <string.h>
#include <gtk/gtk.h>
#include "basic.h"
#include "scanner.h"
#include "input.h"
#include "canvas.h"
#include "result.h"

/* Types. ===================================================================*/

typedef struct 
{ 
  list_node_t *next;
  pos_string_t *index;
  pos_value_t *value;
} result_t;

/* Global variables. ========================================================*/

rectangle_t result_geometry;
string_t result_font_family;
int_t result_font_size;

/* Variables. ===============================================================*/

static pos_string_t *result_surf;
static list_t results;
static canvas_t *result_canvas;
static pos_string_t *colon;

/* Functions. ===============================================================*/

static void
configure_result( canvas_t *canvas, int_t *width_p, int_t *height_p )
{
  int_t width, height;
  result_t *result;
  int_t font_height = get_font_height( canvas );
  int_t space_width = get_space_width( canvas );
  int_t border_width = get_border_width( canvas );

  config_pos_string( colon, canvas );

  height = border_width;
  width = border_width;
  if (result_surf != NULL) 
  { 
    config_pos_string( result_surf, canvas );
    result_surf->x = border_width;
    result_surf->y = border_width;
    width = MAX( width, border_width + result_surf->width );
    height += result_surf->height;
  }
  
  FOREACH( result, results ) 
  { 
    height += font_height;

    config_pos_string( result->index, canvas );
    result->index->x = border_width;
    result->index->y = height;

    config_pos_value( result->value, canvas );
    result->value->x = (result->index->x + result->index->width 
			+ colon->width + space_width);
    result->value->y = height;
    width = MAX( width, result->value->x + result->value->width );
    height += result->value->height;

    result->index->y += (result->value->height - font_height) / 2;
  }

  *width_p = width + border_width;
  *height_p = height + border_width;
}

/*---------------------------------------------------------------------------*/

static void
expose_result( canvas_t *canvas, rectangle_t *area )
{
  result_t *result;

  set_color( BLACK );

  if (result_surf != NULL) 
    draw_pos_string( result_surf, canvas );
  
  FOREACH( result, results ) 
  { 
    colon->x = result->index->x + result->index->width;
    colon->y = result->index->y;
    draw_pos_string( result->index, canvas );
    draw_pos_string( colon, canvas );
    draw_pos_value( result->value, canvas );
  }
}

/*---------------------------------------------------------------------------*/

static void
free_result( canvas_t *canvas )
{
  result_t *result;
  
  free_pos_string( &colon );
  free_pos_string( &result_surf );
  FOREACH_FREE( result, results ) 
  { 
    free_pos_string( &result->index );
    free_pos_value( &result->value );
  }
}

/*---------------------------------------------------------------------------*/

void 
read_result( void )
/* Read new results from STDIN. */
{
  result_t *result;
  string_t line, string;

  free_result( NULL );
  line = read_line( stdin );
  result_surf = new_pos_string( line );
  free_mem( &line );

  while (TRUE) 
  { 
    line = read_line( stdin );
    if (line == NULL) 
      complain( "Premature EOF." );
    if (strcmp_no_case( line, "end" ) == 0) 
      break;

    /* Read a new result. */
    result = new_node( &results, sizeof( result_t ), LIST_END );
    set_scanner_input( line );

    /* Read result index. */
    test_token( TOK_NUMBER );
    string = int_to_string( token_number );
    result->index = new_pos_string( string );
    free_mem( &string );
    read_next_token();

    /* Read result value. */
    parse_token( '{' );
    result->value = parse_pos_value();
    parse_token( '}' );
    parse_token( EOF );
    set_scanner_input( NULL );
    free_mem( &line );
  }
  free_mem( &line );

  colon = new_pos_string( ":" );

  if (result_canvas == NULL) 
  { 
    result_canvas = create_canvas(
      "Malaga Result", "result.eps", &result_geometry, configure_result, 
      expose_result, free_result, NULL, TRUE, NULL, 0 );
  } 
  else
  {
    configure_canvas( result_canvas );
    show_canvas( result_canvas );
  }
}

/* End of file. =============================================================*/
