/*
 * Mesa 3-D graphics library
 *
 * Copyright (C) 2012-2013 LunarG, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *    Chia-I Wu <olv@lunarg.com>
 */

#include "genhw/genhw.h"
#include "core/ilo_state_surface.h"
#include "core/ilo_state_vf.h"
#include "ilo_format.h"

bool
ilo_format_support_vb(const struct ilo_dev *dev,
                      enum pipe_format format)
{
   const int idx = ilo_format_translate(dev, format, PIPE_BIND_VERTEX_BUFFER);

   return (idx >= 0 && ilo_state_vf_valid_element_format(dev, idx));
}

bool
ilo_format_support_sol(const struct ilo_dev *dev,
                       enum pipe_format format)
{
   const int idx = ilo_format_translate(dev, format, PIPE_BIND_STREAM_OUTPUT);

   return (idx >= 0 && ilo_state_surface_valid_format(dev,
            ILO_STATE_SURFACE_ACCESS_DP_SVB, idx));
}

bool
ilo_format_support_sampler(const struct ilo_dev *dev,
                           enum pipe_format format)
{
   const int idx = ilo_format_translate(dev, format, PIPE_BIND_SAMPLER_VIEW);

   return (idx >= 0 && ilo_state_surface_valid_format(dev,
            ILO_STATE_SURFACE_ACCESS_SAMPLER, idx));
}

bool
ilo_format_support_rt(const struct ilo_dev *dev,
                      enum pipe_format format)
{
   const int idx = ilo_format_translate(dev, format, PIPE_BIND_RENDER_TARGET);

   return (idx >= 0 && ilo_state_surface_valid_format(dev,
            ILO_STATE_SURFACE_ACCESS_DP_RENDER, idx));
}

bool
ilo_format_support_zs(const struct ilo_dev *dev,
                      enum pipe_format format)
{
   switch (format) {
   case PIPE_FORMAT_Z16_UNORM:
   case PIPE_FORMAT_Z24X8_UNORM:
   case PIPE_FORMAT_Z32_FLOAT:
   case PIPE_FORMAT_Z24_UNORM_S8_UINT:
   case PIPE_FORMAT_Z32_FLOAT_S8X24_UINT:
      return true;
   case PIPE_FORMAT_S8_UINT:
      /* TODO separate stencil */
   default:
      return false;
   }
}

/**
 * Translate a color (non-depth/stencil) pipe format to the matching hardware
 * format.  Return -1 on errors.
 */
int
ilo_format_translate_color(const struct ilo_dev *dev,
                           enum pipe_format format)
{
   static const int format_mapping[PIPE_FORMAT_COUNT] = {
      [PIPE_FORMAT_NONE]                  = 0,
      [PIPE_FORMAT_B8G8R8A8_UNORM]        = GEN6_FORMAT_B8G8R8A8_UNORM,
      [PIPE_FORMAT_B8G8R8X8_UNORM]        = GEN6_FORMAT_B8G8R8X8_UNORM,
      [PIPE_FORMAT_A8R8G8B8_UNORM]        = 0,
      [PIPE_FORMAT_X8R8G8B8_UNORM]        = 0,
      [PIPE_FORMAT_B5G5R5A1_UNORM]        = GEN6_FORMAT_B5G5R5A1_UNORM,
      [PIPE_FORMAT_B4G4R4A4_UNORM]        = GEN6_FORMAT_B4G4R4A4_UNORM,
      [PIPE_FORMAT_B5G6R5_UNORM]          = GEN6_FORMAT_B5G6R5_UNORM,
      [PIPE_FORMAT_R10G10B10A2_UNORM]     = GEN6_FORMAT_R10G10B10A2_UNORM,
      [PIPE_FORMAT_L8_UNORM]              = GEN6_FORMAT_L8_UNORM,
      [PIPE_FORMAT_A8_UNORM]              = GEN6_FORMAT_A8_UNORM,
      [PIPE_FORMAT_I8_UNORM]              = GEN6_FORMAT_I8_UNORM,
      [PIPE_FORMAT_L8A8_UNORM]            = GEN6_FORMAT_L8A8_UNORM,
      [PIPE_FORMAT_L16_UNORM]             = GEN6_FORMAT_L16_UNORM,
      [PIPE_FORMAT_UYVY]                  = GEN6_FORMAT_YCRCB_SWAPUVY,
      [PIPE_FORMAT_YUYV]                  = GEN6_FORMAT_YCRCB_NORMAL,
      [PIPE_FORMAT_Z16_UNORM]             = 0,
      [PIPE_FORMAT_Z32_UNORM]             = 0,
      [PIPE_FORMAT_Z32_FLOAT]             = 0,
      [PIPE_FORMAT_Z24_UNORM_S8_UINT]     = 0,
      [PIPE_FORMAT_S8_UINT_Z24_UNORM]     = 0,
      [PIPE_FORMAT_Z24X8_UNORM]           = 0,
      [PIPE_FORMAT_X8Z24_UNORM]           = 0,
      [PIPE_FORMAT_S8_UINT]               = 0,
      [PIPE_FORMAT_R64_FLOAT]             = GEN6_FORMAT_R64_FLOAT,
      [PIPE_FORMAT_R64G64_FLOAT]          = GEN6_FORMAT_R64G64_FLOAT,
      [PIPE_FORMAT_R64G64B64_FLOAT]       = GEN6_FORMAT_R64G64B64_FLOAT,
      [PIPE_FORMAT_R64G64B64A64_FLOAT]    = GEN6_FORMAT_R64G64B64A64_FLOAT,
      [PIPE_FORMAT_R32_FLOAT]             = GEN6_FORMAT_R32_FLOAT,
      [PIPE_FORMAT_R32G32_FLOAT]          = GEN6_FORMAT_R32G32_FLOAT,
      [PIPE_FORMAT_R32G32B32_FLOAT]       = GEN6_FORMAT_R32G32B32_FLOAT,
      [PIPE_FORMAT_R32G32B32A32_FLOAT]    = GEN6_FORMAT_R32G32B32A32_FLOAT,
      [PIPE_FORMAT_R32_UNORM]             = GEN6_FORMAT_R32_UNORM,
      [PIPE_FORMAT_R32G32_UNORM]          = GEN6_FORMAT_R32G32_UNORM,
      [PIPE_FORMAT_R32G32B32_UNORM]       = GEN6_FORMAT_R32G32B32_UNORM,
      [PIPE_FORMAT_R32G32B32A32_UNORM]    = GEN6_FORMAT_R32G32B32A32_UNORM,
      [PIPE_FORMAT_R32_USCALED]           = GEN6_FORMAT_R32_USCALED,
      [PIPE_FORMAT_R32G32_USCALED]        = GEN6_FORMAT_R32G32_USCALED,
      [PIPE_FORMAT_R32G32B32_USCALED]     = GEN6_FORMAT_R32G32B32_USCALED,
      [PIPE_FORMAT_R32G32B32A32_USCALED]  = GEN6_FORMAT_R32G32B32A32_USCALED,
      [PIPE_FORMAT_R32_SNORM]             = GEN6_FORMAT_R32_SNORM,
      [PIPE_FORMAT_R32G32_SNORM]          = GEN6_FORMAT_R32G32_SNORM,
      [PIPE_FORMAT_R32G32B32_SNORM]       = GEN6_FORMAT_R32G32B32_SNORM,
      [PIPE_FORMAT_R32G32B32A32_SNORM]    = GEN6_FORMAT_R32G32B32A32_SNORM,
      [PIPE_FORMAT_R32_SSCALED]           = GEN6_FORMAT_R32_SSCALED,
      [PIPE_FORMAT_R32G32_SSCALED]        = GEN6_FORMAT_R32G32_SSCALED,
      [PIPE_FORMAT_R32G32B32_SSCALED]     = GEN6_FORMAT_R32G32B32_SSCALED,
      [PIPE_FORMAT_R32G32B32A32_SSCALED]  = GEN6_FORMAT_R32G32B32A32_SSCALED,
      [PIPE_FORMAT_R16_UNORM]             = GEN6_FORMAT_R16_UNORM,
      [PIPE_FORMAT_R16G16_UNORM]          = GEN6_FORMAT_R16G16_UNORM,
      [PIPE_FORMAT_R16G16B16_UNORM]       = GEN6_FORMAT_R16G16B16_UNORM,
      [PIPE_FORMAT_R16G16B16A16_UNORM]    = GEN6_FORMAT_R16G16B16A16_UNORM,
      [PIPE_FORMAT_R16_USCALED]           = GEN6_FORMAT_R16_USCALED,
      [PIPE_FORMAT_R16G16_USCALED]        = GEN6_FORMAT_R16G16_USCALED,
      [PIPE_FORMAT_R16G16B16_USCALED]     = GEN6_FORMAT_R16G16B16_USCALED,
      [PIPE_FORMAT_R16G16B16A16_USCALED]  = GEN6_FORMAT_R16G16B16A16_USCALED,
      [PIPE_FORMAT_R16_SNORM]             = GEN6_FORMAT_R16_SNORM,
      [PIPE_FORMAT_R16G16_SNORM]          = GEN6_FORMAT_R16G16_SNORM,
      [PIPE_FORMAT_R16G16B16_SNORM]       = GEN6_FORMAT_R16G16B16_SNORM,
      [PIPE_FORMAT_R16G16B16A16_SNORM]    = GEN6_FORMAT_R16G16B16A16_SNORM,
      [PIPE_FORMAT_R16_SSCALED]           = GEN6_FORMAT_R16_SSCALED,
      [PIPE_FORMAT_R16G16_SSCALED]        = GEN6_FORMAT_R16G16_SSCALED,
      [PIPE_FORMAT_R16G16B16_SSCALED]     = GEN6_FORMAT_R16G16B16_SSCALED,
      [PIPE_FORMAT_R16G16B16A16_SSCALED]  = GEN6_FORMAT_R16G16B16A16_SSCALED,
      [PIPE_FORMAT_R8_UNORM]              = GEN6_FORMAT_R8_UNORM,
      [PIPE_FORMAT_R8G8_UNORM]            = GEN6_FORMAT_R8G8_UNORM,
      [PIPE_FORMAT_R8G8B8_UNORM]          = GEN6_FORMAT_R8G8B8_UNORM,
      [PIPE_FORMAT_R8G8B8A8_UNORM]        = GEN6_FORMAT_R8G8B8A8_UNORM,
      [PIPE_FORMAT_X8B8G8R8_UNORM]        = 0,
      [PIPE_FORMAT_R8_USCALED]            = GEN6_FORMAT_R8_USCALED,
      [PIPE_FORMAT_R8G8_USCALED]          = GEN6_FORMAT_R8G8_USCALED,
      [PIPE_FORMAT_R8G8B8_USCALED]        = GEN6_FORMAT_R8G8B8_USCALED,
      [PIPE_FORMAT_R8G8B8A8_USCALED]      = GEN6_FORMAT_R8G8B8A8_USCALED,
      [PIPE_FORMAT_R8_SNORM]              = GEN6_FORMAT_R8_SNORM,
      [PIPE_FORMAT_R8G8_SNORM]            = GEN6_FORMAT_R8G8_SNORM,
      [PIPE_FORMAT_R8G8B8_SNORM]          = GEN6_FORMAT_R8G8B8_SNORM,
      [PIPE_FORMAT_R8G8B8A8_SNORM]        = GEN6_FORMAT_R8G8B8A8_SNORM,
      [PIPE_FORMAT_R8_SSCALED]            = GEN6_FORMAT_R8_SSCALED,
      [PIPE_FORMAT_R8G8_SSCALED]          = GEN6_FORMAT_R8G8_SSCALED,
      [PIPE_FORMAT_R8G8B8_SSCALED]        = GEN6_FORMAT_R8G8B8_SSCALED,
      [PIPE_FORMAT_R8G8B8A8_SSCALED]      = GEN6_FORMAT_R8G8B8A8_SSCALED,
      [PIPE_FORMAT_R32_FIXED]             = GEN6_FORMAT_R32_SFIXED,
      [PIPE_FORMAT_R32G32_FIXED]          = GEN6_FORMAT_R32G32_SFIXED,
      [PIPE_FORMAT_R32G32B32_FIXED]       = GEN6_FORMAT_R32G32B32_SFIXED,
      [PIPE_FORMAT_R32G32B32A32_FIXED]    = GEN6_FORMAT_R32G32B32A32_SFIXED,
      [PIPE_FORMAT_R16_FLOAT]             = GEN6_FORMAT_R16_FLOAT,
      [PIPE_FORMAT_R16G16_FLOAT]          = GEN6_FORMAT_R16G16_FLOAT,
      [PIPE_FORMAT_R16G16B16_FLOAT]       = GEN6_FORMAT_R16G16B16_FLOAT,
      [PIPE_FORMAT_R16G16B16A16_FLOAT]    = GEN6_FORMAT_R16G16B16A16_FLOAT,
      [PIPE_FORMAT_L8_SRGB]               = GEN6_FORMAT_L8_UNORM_SRGB,
      [PIPE_FORMAT_L8A8_SRGB]             = GEN6_FORMAT_L8A8_UNORM_SRGB,
      [PIPE_FORMAT_R8G8B8_SRGB]           = GEN6_FORMAT_R8G8B8_UNORM_SRGB,
      [PIPE_FORMAT_A8B8G8R8_SRGB]         = 0,
      [PIPE_FORMAT_X8B8G8R8_SRGB]         = 0,
      [PIPE_FORMAT_B8G8R8A8_SRGB]         = GEN6_FORMAT_B8G8R8A8_UNORM_SRGB,
      [PIPE_FORMAT_B8G8R8X8_SRGB]         = GEN6_FORMAT_B8G8R8X8_UNORM_SRGB,
      [PIPE_FORMAT_A8R8G8B8_SRGB]         = 0,
      [PIPE_FORMAT_X8R8G8B8_SRGB]         = 0,
      [PIPE_FORMAT_R8G8B8A8_SRGB]         = GEN6_FORMAT_R8G8B8A8_UNORM_SRGB,
      [PIPE_FORMAT_DXT1_RGB]              = GEN6_FORMAT_DXT1_RGB,
      [PIPE_FORMAT_DXT1_RGBA]             = GEN6_FORMAT_BC1_UNORM,
      [PIPE_FORMAT_DXT3_RGBA]             = GEN6_FORMAT_BC2_UNORM,
      [PIPE_FORMAT_DXT5_RGBA]             = GEN6_FORMAT_BC3_UNORM,
      [PIPE_FORMAT_DXT1_SRGB]             = GEN6_FORMAT_DXT1_RGB_SRGB,
      [PIPE_FORMAT_DXT1_SRGBA]            = GEN6_FORMAT_BC1_UNORM_SRGB,
      [PIPE_FORMAT_DXT3_SRGBA]            = GEN6_FORMAT_BC2_UNORM_SRGB,
      [PIPE_FORMAT_DXT5_SRGBA]            = GEN6_FORMAT_BC3_UNORM_SRGB,
      [PIPE_FORMAT_RGTC1_UNORM]           = GEN6_FORMAT_BC4_UNORM,
      [PIPE_FORMAT_RGTC1_SNORM]           = GEN6_FORMAT_BC4_SNORM,
      [PIPE_FORMAT_RGTC2_UNORM]           = GEN6_FORMAT_BC5_UNORM,
      [PIPE_FORMAT_RGTC2_SNORM]           = GEN6_FORMAT_BC5_SNORM,
      [PIPE_FORMAT_R8G8_B8G8_UNORM]       = 0,
      [PIPE_FORMAT_G8R8_G8B8_UNORM]       = 0,
      [PIPE_FORMAT_R8SG8SB8UX8U_NORM]     = 0,
      [PIPE_FORMAT_R5SG5SB6U_NORM]        = 0,
      [PIPE_FORMAT_A8B8G8R8_UNORM]        = 0,
      [PIPE_FORMAT_B5G5R5X1_UNORM]        = GEN6_FORMAT_B5G5R5X1_UNORM,
      [PIPE_FORMAT_R10G10B10A2_USCALED]   = GEN6_FORMAT_R10G10B10A2_USCALED,
      [PIPE_FORMAT_R11G11B10_FLOAT]       = GEN6_FORMAT_R11G11B10_FLOAT,
      [PIPE_FORMAT_R9G9B9E5_FLOAT]        = GEN6_FORMAT_R9G9B9E5_SHAREDEXP,
      [PIPE_FORMAT_Z32_FLOAT_S8X24_UINT]  = 0,
      [PIPE_FORMAT_R1_UNORM]              = GEN6_FORMAT_R1_UNORM,
      [PIPE_FORMAT_R10G10B10X2_USCALED]   = GEN6_FORMAT_R10G10B10X2_USCALED,
      [PIPE_FORMAT_R10G10B10X2_SNORM]     = 0,
      [PIPE_FORMAT_L4A4_UNORM]            = 0,
      [PIPE_FORMAT_B10G10R10A2_UNORM]     = GEN6_FORMAT_B10G10R10A2_UNORM,
      [PIPE_FORMAT_R10SG10SB10SA2U_NORM]  = 0,
      [PIPE_FORMAT_R8G8Bx_SNORM]          = 0,
      [PIPE_FORMAT_R8G8B8X8_UNORM]        = GEN6_FORMAT_R8G8B8X8_UNORM,
      [PIPE_FORMAT_B4G4R4X4_UNORM]        = 0,
      [PIPE_FORMAT_X24S8_UINT]            = 0,
      [PIPE_FORMAT_S8X24_UINT]            = 0,
      [PIPE_FORMAT_X32_S8X24_UINT]        = 0,
      [PIPE_FORMAT_B2G3R3_UNORM]          = 0,
      [PIPE_FORMAT_L16A16_UNORM]          = GEN6_FORMAT_L16A16_UNORM,
      [PIPE_FORMAT_A16_UNORM]             = GEN6_FORMAT_A16_UNORM,
      [PIPE_FORMAT_I16_UNORM]             = GEN6_FORMAT_I16_UNORM,
      [PIPE_FORMAT_LATC1_UNORM]           = 0,
      [PIPE_FORMAT_LATC1_SNORM]           = 0,
      [PIPE_FORMAT_LATC2_UNORM]           = 0,
      [PIPE_FORMAT_LATC2_SNORM]           = 0,
      [PIPE_FORMAT_A8_SNORM]              = 0,
      [PIPE_FORMAT_L8_SNORM]              = 0,
      [PIPE_FORMAT_L8A8_SNORM]            = 0,
      [PIPE_FORMAT_I8_SNORM]              = 0,
      [PIPE_FORMAT_A16_SNORM]             = 0,
      [PIPE_FORMAT_L16_SNORM]             = 0,
      [PIPE_FORMAT_L16A16_SNORM]          = 0,
      [PIPE_FORMAT_I16_SNORM]             = 0,
      [PIPE_FORMAT_A16_FLOAT]             = GEN6_FORMAT_A16_FLOAT,
      [PIPE_FORMAT_L16_FLOAT]             = GEN6_FORMAT_L16_FLOAT,
      [PIPE_FORMAT_L16A16_FLOAT]          = GEN6_FORMAT_L16A16_FLOAT,
      [PIPE_FORMAT_I16_FLOAT]             = GEN6_FORMAT_I16_FLOAT,
      [PIPE_FORMAT_A32_FLOAT]             = GEN6_FORMAT_A32_FLOAT,
      [PIPE_FORMAT_L32_FLOAT]             = GEN6_FORMAT_L32_FLOAT,
      [PIPE_FORMAT_L32A32_FLOAT]          = GEN6_FORMAT_L32A32_FLOAT,
      [PIPE_FORMAT_I32_FLOAT]             = GEN6_FORMAT_I32_FLOAT,
      [PIPE_FORMAT_YV12]                  = 0,
      [PIPE_FORMAT_YV16]                  = 0,
      [PIPE_FORMAT_IYUV]                  = 0,
      [PIPE_FORMAT_NV12]                  = 0,
      [PIPE_FORMAT_NV21]                  = 0,
      [PIPE_FORMAT_A4R4_UNORM]            = 0,
      [PIPE_FORMAT_R4A4_UNORM]            = 0,
      [PIPE_FORMAT_R8A8_UNORM]            = 0,
      [PIPE_FORMAT_A8R8_UNORM]            = 0,
      [PIPE_FORMAT_R10G10B10A2_SSCALED]   = GEN6_FORMAT_R10G10B10A2_SSCALED,
      [PIPE_FORMAT_R10G10B10A2_SNORM]     = GEN6_FORMAT_R10G10B10A2_SNORM,
      [PIPE_FORMAT_B10G10R10A2_USCALED]   = GEN6_FORMAT_B10G10R10A2_USCALED,
      [PIPE_FORMAT_B10G10R10A2_SSCALED]   = GEN6_FORMAT_B10G10R10A2_SSCALED,
      [PIPE_FORMAT_B10G10R10A2_SNORM]     = GEN6_FORMAT_B10G10R10A2_SNORM,
      [PIPE_FORMAT_R8_UINT]               = GEN6_FORMAT_R8_UINT,
      [PIPE_FORMAT_R8G8_UINT]             = GEN6_FORMAT_R8G8_UINT,
      [PIPE_FORMAT_R8G8B8_UINT]           = GEN6_FORMAT_R8G8B8_UINT,
      [PIPE_FORMAT_R8G8B8A8_UINT]         = GEN6_FORMAT_R8G8B8A8_UINT,
      [PIPE_FORMAT_R8_SINT]               = GEN6_FORMAT_R8_SINT,
      [PIPE_FORMAT_R8G8_SINT]             = GEN6_FORMAT_R8G8_SINT,
      [PIPE_FORMAT_R8G8B8_SINT]           = GEN6_FORMAT_R8G8B8_SINT,
      [PIPE_FORMAT_R8G8B8A8_SINT]         = GEN6_FORMAT_R8G8B8A8_SINT,
      [PIPE_FORMAT_R16_UINT]              = GEN6_FORMAT_R16_UINT,
      [PIPE_FORMAT_R16G16_UINT]           = GEN6_FORMAT_R16G16_UINT,
      [PIPE_FORMAT_R16G16B16_UINT]        = GEN6_FORMAT_R16G16B16_UINT,
      [PIPE_FORMAT_R16G16B16A16_UINT]     = GEN6_FORMAT_R16G16B16A16_UINT,
      [PIPE_FORMAT_R16_SINT]              = GEN6_FORMAT_R16_SINT,
      [PIPE_FORMAT_R16G16_SINT]           = GEN6_FORMAT_R16G16_SINT,
      [PIPE_FORMAT_R16G16B16_SINT]        = GEN6_FORMAT_R16G16B16_SINT,
      [PIPE_FORMAT_R16G16B16A16_SINT]     = GEN6_FORMAT_R16G16B16A16_SINT,
      [PIPE_FORMAT_R32_UINT]              = GEN6_FORMAT_R32_UINT,
      [PIPE_FORMAT_R32G32_UINT]           = GEN6_FORMAT_R32G32_UINT,
      [PIPE_FORMAT_R32G32B32_UINT]        = GEN6_FORMAT_R32G32B32_UINT,
      [PIPE_FORMAT_R32G32B32A32_UINT]     = GEN6_FORMAT_R32G32B32A32_UINT,
      [PIPE_FORMAT_R32_SINT]              = GEN6_FORMAT_R32_SINT,
      [PIPE_FORMAT_R32G32_SINT]           = GEN6_FORMAT_R32G32_SINT,
      [PIPE_FORMAT_R32G32B32_SINT]        = GEN6_FORMAT_R32G32B32_SINT,
      [PIPE_FORMAT_R32G32B32A32_SINT]     = GEN6_FORMAT_R32G32B32A32_SINT,
      [PIPE_FORMAT_A8_UINT]               = 0,
      [PIPE_FORMAT_I8_UINT]               = GEN6_FORMAT_I8_UINT,
      [PIPE_FORMAT_L8_UINT]               = GEN6_FORMAT_L8_UINT,
      [PIPE_FORMAT_L8A8_UINT]             = GEN6_FORMAT_L8A8_UINT,
      [PIPE_FORMAT_A8_SINT]               = 0,
      [PIPE_FORMAT_I8_SINT]               = GEN6_FORMAT_I8_SINT,
      [PIPE_FORMAT_L8_SINT]               = GEN6_FORMAT_L8_SINT,
      [PIPE_FORMAT_L8A8_SINT]             = GEN6_FORMAT_L8A8_SINT,
      [PIPE_FORMAT_A16_UINT]              = 0,
      [PIPE_FORMAT_I16_UINT]              = 0,
      [PIPE_FORMAT_L16_UINT]              = 0,
      [PIPE_FORMAT_L16A16_UINT]           = 0,
      [PIPE_FORMAT_A16_SINT]              = 0,
      [PIPE_FORMAT_I16_SINT]              = 0,
      [PIPE_FORMAT_L16_SINT]              = 0,
      [PIPE_FORMAT_L16A16_SINT]           = 0,
      [PIPE_FORMAT_A32_UINT]              = 0,
      [PIPE_FORMAT_I32_UINT]              = 0,
      [PIPE_FORMAT_L32_UINT]              = 0,
      [PIPE_FORMAT_L32A32_UINT]           = 0,
      [PIPE_FORMAT_A32_SINT]              = 0,
      [PIPE_FORMAT_I32_SINT]              = 0,
      [PIPE_FORMAT_L32_SINT]              = 0,
      [PIPE_FORMAT_L32A32_SINT]           = 0,
      [PIPE_FORMAT_B10G10R10A2_UINT]      = GEN6_FORMAT_B10G10R10A2_UINT,
      [PIPE_FORMAT_ETC1_RGB8]             = GEN6_FORMAT_ETC1_RGB8,
      [PIPE_FORMAT_R8G8_R8B8_UNORM]       = 0,
      [PIPE_FORMAT_G8R8_B8R8_UNORM]       = 0,
      [PIPE_FORMAT_R8G8B8X8_SNORM]        = 0,
      [PIPE_FORMAT_R8G8B8X8_SRGB]         = 0,
      [PIPE_FORMAT_R8G8B8X8_UINT]         = 0,
      [PIPE_FORMAT_R8G8B8X8_SINT]         = 0,
      [PIPE_FORMAT_B10G10R10X2_UNORM]     = GEN6_FORMAT_B10G10R10X2_UNORM,
      [PIPE_FORMAT_R16G16B16X16_UNORM]    = GEN6_FORMAT_R16G16B16X16_UNORM,
      [PIPE_FORMAT_R16G16B16X16_SNORM]    = 0,
      [PIPE_FORMAT_R16G16B16X16_FLOAT]    = GEN6_FORMAT_R16G16B16X16_FLOAT,
      [PIPE_FORMAT_R16G16B16X16_UINT]     = 0,
      [PIPE_FORMAT_R16G16B16X16_SINT]     = 0,
      [PIPE_FORMAT_R32G32B32X32_FLOAT]    = GEN6_FORMAT_R32G32B32X32_FLOAT,
      [PIPE_FORMAT_R32G32B32X32_UINT]     = 0,
      [PIPE_FORMAT_R32G32B32X32_SINT]     = 0,
      [PIPE_FORMAT_R8A8_SNORM]            = 0,
      [PIPE_FORMAT_R16A16_UNORM]          = 0,
      [PIPE_FORMAT_R16A16_SNORM]          = 0,
      [PIPE_FORMAT_R16A16_FLOAT]          = 0,
      [PIPE_FORMAT_R32A32_FLOAT]          = 0,
      [PIPE_FORMAT_R8A8_UINT]             = 0,
      [PIPE_FORMAT_R8A8_SINT]             = 0,
      [PIPE_FORMAT_R16A16_UINT]           = 0,
      [PIPE_FORMAT_R16A16_SINT]           = 0,
      [PIPE_FORMAT_R32A32_UINT]           = 0,
      [PIPE_FORMAT_R32A32_SINT]           = 0,
      [PIPE_FORMAT_R10G10B10A2_UINT]      = GEN6_FORMAT_R10G10B10A2_UINT,
      [PIPE_FORMAT_B5G6R5_SRGB]           = GEN6_FORMAT_B5G6R5_UNORM_SRGB,
   };
   int sfmt = format_mapping[format];

   /* GEN6_FORMAT_R32G32B32A32_FLOAT happens to be 0 */
   if (!sfmt && format != PIPE_FORMAT_R32G32B32A32_FLOAT)
      sfmt = -1;

   return sfmt;
}
