#
# Copyright (C) 1997, 1998, 1999  Motoyuki Kasahara
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

# Whether BookView connects to server1 at start.
# (used in the setup window only)
set setup_start_server1 1

#
# `Setup...'
#
proc setup {} {
    global start_server1
    global setup_start_server1
    global server_names

    setup_make_window

    if {[llength $server_names] == 0} {
	set setup_start_server1 1
    } else {
	set setup_start_server1 $start_server1
    }

    set oldfocus [focus]

    #
    # Popup a sub window.
    #
    wm title .setup [_ "bookview setup"]
    wm iconname .setup "bookview setup"
    wm withdraw .setup

    set x [expr [winfo rootx .] + 24]
    set y [expr [winfo rooty .] + 24]
    wm geometry .setup +$x+$y
    wm deiconify .setup

    focus .setup
    grab .setup
    tkwait window .setup

    if {![catch {winfo .setup}]} {
	destroy .setup
    }
    focus $oldfocus
}

#
# Make a setup window.
#
proc setup_make_window {} {
    global max_servers
    global start_server1
    global setup_start_server1
    global server_names
    global server_ports
    global server_options

    #
    # Toplevel
    #
    toplevel .setup
    bind .setup <Return> setup_ok
    bind .setup <Control-m> setup_ok
    bind .setup <Escape> {destroy .setup}

    #
    # Server 1, 2, ... n
    #
    set length [llength $server_names]
    for {set i 1} {$i <= $max_servers} {incr i} {
	frame .setup.server$i -relief sunken -borderwidth 2
	pack .setup.server$i -side top -padx 2 -fill x

	label .setup.server$i.id -text [format [_ "Server %d:"] $i]
	pack .setup.server$i.id -side top -anchor w -pady 2

	frame .setup.server$i.hostname
	pack .setup.server$i.hostname -side top -fill x
	label .setup.server$i.hostname.label -text [_ "hostname:"] \
		-width 11
	pack .setup.server$i.hostname.label -side left
	entry .setup.server$i.hostname.entry -relief sunken -borderwidth 2 \
		-width 50
	pack .setup.server$i.hostname.entry -side left -anchor w -expand 1 \
		-fill x

	frame .setup.server$i.port
	pack .setup.server$i.port -side top -fill x
	label .setup.server$i.port.label -text [_ "    port:"] \
		-width 11
	pack .setup.server$i.port.label -side left
	entry .setup.server$i.port.entry -relief sunken -borderwidth 2 \
		-width 50
	pack .setup.server$i.port.entry -side left -anchor w -expand 1 \
		-fill x

	frame .setup.server$i.options
	pack .setup.server$i.options -side top -fill x
	label .setup.server$i.options.label -text [_ " options:"] \
		-width 11
	pack .setup.server$i.options.label -side left
	entry .setup.server$i.options.entry -relief sunken -borderwidth 2 \
		-width 50
	pack .setup.server$i.options.entry -side left -anchor w -expand 1 \
		-fill x

	#
	# Set default values.
	#
	if {$i <= $length} {
	    .setup.server$i.hostname.entry insert 0 \
		    [lindex $server_names [expr $i - 1]]
	    .setup.server$i.port.entry insert 0 \
		    [lindex $server_ports [expr $i - 1]]
	    .setup.server$i.options.entry insert 0 \
		    [lindex $server_options [expr $i - 1]]
	} else {
	    .setup.server$i.hostname.entry insert 0 ""
	    .setup.server$i.port.entry insert 0 "ndtp"
	    .setup.server$i.options.entry insert 0 "start=@all"
	}
    }

    #
    # Whether connect to server 1 at start of BookView.
    #
    frame .setup.at_start
    pack .setup.at_start -side top
    checkbutton .setup.at_start.check -variable setup_start_server1 \
  	    -text [_ "Connect to server 1 at start of BookView"] -onvalue 1 \
  	    -offvalue 0
    pack .setup.at_start.check -side top -pady 2 -fill x

    #
    # Ok and Cancel buttons.
    #
    frame .setup.ok-cancel
    pack .setup.ok-cancel -side top -pady 4 -fill x

    button .setup.ok-cancel.ok -text [_ "Ok"] -command setup_ok -default active
    pack .setup.ok-cancel.ok -side left -expand 1

    button .setup.ok-cancel.cancel -text [_ "Cancel"] -command {destroy .setup}
    pack .setup.ok-cancel.cancel -side left -expand 1
}

#
# `Ok' on the setup window.
#
proc setup_ok {} {
    global max_servers
    global start_server1
    global setup_start_server1
    global server_names
    global server_ports
    global server_options
    global server_starts
    global server_demand_loads
    global server_book_groups

    #
    # Close the current connection.
    #
    select_server 0

    #
    # Set a server list.
    #
    set server_names {}
    set server_ports {}
    set server_options {}
    set server_starts {}
    set server_demand_loads {}
    set server_book_groups {}
    for {set i 1} {$i <= $max_servers} {incr i} {
	set hostname [.setup.server$i.hostname.entry get]
	set port [.setup.server$i.port.entry get]
	set option [.setup.server$i.options.entry get]
	regsub {[ 	:].*} $hostname {} hostname
	regsub {[ 	:].*} $port {} port
	regsub -all ":" $option " " option

	if {![regexp {^[ 	]*$} $hostname]} {
	    if {[regexp {^[ 	]*$} $port]} {
		set $port "ndtp"
	    }
	    lappend server_names $hostname
	    lappend server_ports $port
	    lappend server_options $option
	    setup_parse_options $option
	}
    }

    #
    # Whether connect to server 1 at start of BookView.
    #
    if {![regexp {^[ 	]*$} [.setup.server1.hostname.entry get]] \
	    && $setup_start_server1} {
	set start_server1 1
    } else {
	set start_server1 0
    }

    #
    # Update `Server'.
    #
    set_server_menu

    #
    # Save a setup file.
    #
    setup_save_file

    destroy .setup
}

#
# Load a setup file.
#
proc setup_load_file {} {
    global system_setup_file
    global user_setup_file
    global server_names
    global server_ports
    global server_options
    global server_starts
    global server_demand_loads
    global server_book_groups
    global start_server1
    global bitmap_checkbutton
    global toolbar_checkbutton
    global method_radiobutton
    global max_servers
    global text_spacing

    set server_names {}
    set server_ports {}
    set server_starts {}
    set start_server1 0
    set server_demand_loads {}
    set server_book_groups {}

    #
    # Open a setup-file.
    # Try user setup file first, and then system setup file.
    #
    set setup_file $user_setup_file
    output_information [format [_ "Load %s..."] $setup_file]
    if {[catch {set file [open $setup_file "r"]}]} {
	set setup_file $system_setup_file
	output_information [format [_ "Load %s..."] $setup_file]
	if {[catch {set file [open $setup_file "r"]}]} {
	    output_information [_ "Cannot open a setup file."]
	    return 0
	}
    }

    #
    # Read a setup-file.
    #
    set lineno 0
    while {1} {
	if {[catch {gets $file line}]} {
	    output_information [format [_ "Cannot read %s."] $setup_file]
	    catch {close $file}
	    return 0
	}
	incr lineno
	if {[regexp {^server ([^:]+):([^:]+):(.*)} $line dummy \
		hostname port options]} {
	    lappend server_names $hostname
	    lappend server_ports $port
	    lappend server_options $options
	    setup_parse_options $options
	} elseif {[regexp {^start-server1 ([01])$} $line dummy arg]} {
	    set start_server1 $arg
	} elseif {[regexp {^display-bitmaps ([0-9]+)$} $line dummy arg]} {
	    set bitmap_checkbutton $arg
	} elseif {[regexp {^display-toolbar ([0-9]+)$} $line dummy arg]} {
	    set toolbar_checkbutton $arg
	} elseif {[regexp {^text-spacing ([0-9]+)$} $line dummy arg]} {
	    set text_spacing $arg
	} elseif {[regexp {^default-method (.+)$} $line dummy arg]} {
	    ndtp_set_default_method $arg
	    set method_radiobutton $arg
	}
	if {[eof $file]} {
	    break
	}
    }

    #
    # Close a setup-file.
    #
    output_information [format [_ "Load %s...done."] $setup_file]
    catch {close $file}

    #
    # Check for the setup.
    #
    set length [llength $server_names]
    if {$max_servers < $length} {
	set server_names [lrange $server_names 0 [expr $max_servers - 1]]
	set server_ports [lrange $server_ports 0 [expr $max_servers - 1]]
	set server_starts [lrange $server_starts 0 [expr $max_servers - 1]]
    } elseif {$length == 0} {
	set start_server1 0
    }

    if {$text_spacing < 0} {
	set text_spacing 0
    }

    return 1
}

#
# Save a setup file.
#
proc setup_save_file {} {
    global user_setup_file
    global server_names
    global server_ports
    global server_options
    global start_server1
    global bitmap_checkbutton
    global toolbar_checkbutton
    global method_radiobutton
    global text_spacing

    #
    # Open a setup file.
    #
    output_information [format [_ "Save %s..."] $user_setup_file]
    if {[catch {set file [open $user_setup_file "w"]}]} {
	output_information [format [_ "Cannot open %s"] $user_setup_file]
	return
    }

    #
    # Write to a setup file.
    #
    set length [llength $server_names]
    for {set i 0} {$i < $length} {incr i} {
	set hostname [lindex $server_names $i]
	set port [lindex $server_ports $i]
	set option [lindex $server_options $i]
	catch {puts $file "server $hostname:$port:$option"}
    }
    catch {puts $file "start-server1 $start_server1"}
    catch {puts $file "prefer-sizes 18"}
    catch {puts $file "display-bitmaps $bitmap_checkbutton"}
    catch {puts $file "display-toolbar $toolbar_checkbutton"}
    catch {puts $file "text-spacing $text_spacing"}
    catch {puts $file "default-method $method_radiobutton"}
    if {[catch {flush $file}]} {
	output_information [format [_ "Cannot write to %s."] $user_setup_file]
	catch {close $file}
	return
    }

    #
    # Close a setup file.
    #
    catch {close $file}
    output_information [format [_ "Save %s....done."] $user_setup_file]
}

#
# Parse a server option stinrg.
#
proc setup_parse_options {options} {
    global server_starts
    global server_demand_loads
    global server_book_groups

    set start ""
    set demand_load {}
    set book_group {}

    append options ";"
    while {[regexp {([^;]*);(.*)} $options dummy opt options]} {
	regsub {^[ 	](.*)[ 	]*$} $opt {\1} opt
	if {[regexp {^start=(.*)$} $opt dummy arg]} {
	    set start $arg
	} elseif {[regexp {^demand-load=(.*)$} $opt dummy arg]} {
	    regsub "," $arg " " arg
	    foreach i $arg {
		lappend demand_load $i
	    }
	} elseif {[regexp {^group=([^,]*),([^,]*),(.*)$} \
		$opt dummy name title arg]} {
	    if {![string match $name {@*}]} {
		set name "@$name"
	    }
	    regsub -all "," $arg " " arg
	    set expanded {}
	    foreach i $arg {
		lappend expanded $i
	    }
	    lappend book_group [list $name $title $expanded]
	}
    }

    lappend server_starts $start
    lappend server_demand_loads $demand_load
    lappend server_book_groups $book_group
}
