/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
   \file cdw_config.c
   \brief Functions operating on configuration file and configuration variable(s)

   This file defines functions that:
   \li initialize and destroy configuration module
   \li read and write configuration file (with error handling)
   \li set default configuration that can be used when program can't read configuration from file
   \li validate configuration
*/

#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* strcasestr() */

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/types.h>


#include "config.h"
#include "cdw_config.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_fs.h"
#include "cdw_widgets.h"
#include "cdw_disc.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_ext_tools.h"
#include "cdw_erase_disc.h"
#include "cdw_drive.h"
#include "cdw_iso9660.h"

extern const char *cdw_log_file_name;


#define CDW_CONFIG_VOLUME_ITEMS_MAX 6 /* 74min CD, 80min CD, DVD, DVD+R DL, custom, auto */


cdw_id_clabel_t cdw_config_volume_size_items[CDW_CONFIG_VOLUME_ITEMS_MAX] = {
	/* 2TRANS: this is dropdown item label: 650MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,               gettext_noop("74 min CD (650 MB)") },
	/* 2TRANS: this is dropdown item label: 700MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD80,               gettext_noop("80 min CD (700 MB)") },
	/* 2TRANS: this is dropdown item label: DVD */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC,        gettext_noop("Generic DVD (4.7 GB)") },
	/* 2TRANS: this is dropdown item label: DVD+R DL */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,          gettext_noop("DVD+R DL (8.5 GB)") },
	/* 2TRANS: this is dropdown item label: custom value of
	   ISO volume; user can enter arbitrary natural number */
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,             gettext_noop("Custom value") },
	/* 2TRANS: this is dropdown item label:
	   "automatic" = automatic detection/resolution of size of ISO volume */
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,               gettext_noop("Get sizes from disc") }
};

/** \brief Flag letting you know if options module works in abnormal mode
    Set to true if something went wrong during module setup -
    cdw will work without reading/writing configuration to disk file */
bool failsafe_mode = false;

/** \brief Project-wide variable holding current configuration of cdw */
cdw_config_t global_config;



static char *cdw_config_init_config_dir(const char *base_dir);
static cdw_rv_t cdw_config_read_from_file(const char *fullpath);



/** \brief Hardwired name of cdw configuration file */
static const char *old_config_file_name = ".cdw.conf";
static const char *config_file_name = "cdw.conf";
/** \brief Stores directory name where cdw config file is stored,
    also used as base path needed by some option values; if there are
    no errors during configuration of options module, this path is
    set to user home directory; since this path is initialized with
    value from cdw_fs.c module, ending slash is included */
static char *base_dir_fullpath = (char *) NULL;
/** \brief Full path to cdw configuration dir */
static char *config_dir_fullpath = (char *) NULL;
/** \brief Full path to cdw configuration file */
static char *config_file_fullpath = (char *) NULL;
/** \brief Full path to cdw configuration file in old localization ($HOME/.cdw.conf) */
static char *old_config_file_fullpath = (char *) NULL;



/* pair of low level functions that directly access configuration file;
   they are used by cdw_config_write_to_file() and
   cdw_config_read_from_file() respectively */
static void cdw_config_var_write_to_file(FILE *config_file, cdw_config_t *_config);
static cdw_rv_t cdw_config_var_read_from_file(FILE *config_file, cdw_config_t *_config);

static cdw_rv_t cdw_config_var_set_defaults(cdw_config_t *_config);
static cdw_rv_t cdw_config_module_set_paths(void);


struct cdw_config_vst_t {
	cdw_id_t id;
	const char *old_label;
	const char *new_label;
	long int size;
};



static struct cdw_config_vst_t cdw_config_volume_size_table[] = {
	/*         id;                      old_label             new_label;               size   */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,        "650",                "cd74",                681984000  / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_CD80,        "703",                "cd80",                737280000  / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC, "4482",               "dvd",                 4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_R,       "4489",               "dvd-r",               4707319808 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP,      "4482",               "dvd+r",               4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RW,      "4489",               "dvd-rw",              4707319808 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RWP,     "4482",               "dvd+rw",              4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_R_DL,    "dvd-r dl",           "dvd-r dl",            8543666176 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,   "dvd+r dl",           "dvd+r dl",            8547991552 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,      "custom",             "custom",              0      },
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,        "auto",               "auto",                0      },
	{ -1,                                 (char *) NULL,        (char *) NULL,         0      } };


long int cdw_config_get_volume_size_mb_by_id(cdw_id_t id)
{
	struct cdw_config_vst_t *table = cdw_config_volume_size_table;
	for (int i = 0; table[i].id != -1; i++) {
		if (table[i].id == id) {
			return table[i].size;
		}
	}
	cdw_vdm ("ERROR: id %lld not found\n", id);
	return 0;
}





cdw_id_t cdw_config_volume_id_by_label(const char *label)
{
	struct cdw_config_vst_t *table = cdw_config_volume_size_table;
	for (int i = 0; table[i].id != -1; i++) {
		if (!strcasecmp(label, table[i].old_label) || !strcasecmp(label, table[i].new_label)) {
			return table[i].id;
		}
	}
	cdw_vdm ("WARNING: label \"%s\" not found\n", label);
	return -1;
}





/**
 * \brief Initialize configuration module
 *
 * Initialize configuration module - prepare some default option values and set paths
 * to base directory (should be user home directory) and to cdw configuration
 * file (will be in base directory).
 *
 * \return CDW_OK on success
 * \return CDW_ERROR on errors - the module wasn't configured at all
 * \return CDW_CANCEL if module cannot access cdw configuration file (module is working in failsafe mode)
 */
cdw_rv_t cdw_config_module_init(void)
{
	cdw_rv_t crv = cdw_config_module_set_paths();
	if (crv == CDW_NO || crv == CDW_OK) { /* paths are set correctly... */
		if (crv == CDW_NO) { /* ... but not to $HOME */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot find home directory and settings file. Will use configuration file in temporary location."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		}

		failsafe_mode = false;
	} else { /* error situation; we can check what happened, but not in this release */
		failsafe_mode = true;
	}

	crv = cdw_config_var_init_fields(&global_config);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to initialize fields of config var\n");
		return CDW_ERROR;
	}

	crv = cdw_config_var_set_defaults(&global_config);
	if (crv != CDW_OK) {
		cdw_config_var_free_fields(&global_config);
		cdw_vdm ("ERROR: failed to set defaults for config variable\n");
		return CDW_ERROR;
	}

	if (failsafe_mode) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Config file error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot read or write settings file. Will use temporary configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return CDW_CANCEL;
	}


	/* This overwrites default option values with values found in config
	   file, don't create config file if it doesn't exist already */

	char *path = config_file_fullpath;
	if (old_config_file_fullpath != (char *) NULL) {
		if (!access(old_config_file_fullpath, F_OK)) {
			/* can't log into log file, since it is not configured yet */

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Information"),
					   /* 2TRANS: this is message in dialog window */
					   _("Migrating cdw configuration files to new location (into .cdw/ subdirectory in home dir)"),
					   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
			path = old_config_file_fullpath;
		}
	}

	crv = cdw_config_read_from_file(path);

	if (crv == CDW_ERROR || crv == CDW_CANCEL) {
		failsafe_mode = true;
		cdw_vdm ("ERROR: failed to read config from file, setting failsafe mode\n");
		return CDW_CANCEL;
	} else {
		return CDW_OK;
	}
}





/**
 * \brief Write current configuration to disk file
 *
 * Write content of config variable to disk file.
 *
 * This function only opens (for writing) config file, (in future: does
 * basic error checking) calls function doing actual writing, and then
 * closes config file.
 *
 * If configuration module works in failsafe mode, this function silently
 * skips writing to file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \return CDW_OK on success
 * \return CDW_ERROR if function failed to open file for writing
 * \return CDW_CANCEL if failsafe mode is in effect and writing was skipped
 */
cdw_rv_t cdw_config_write_to_file(void)
{
	if (failsafe_mode) {
		/* this might be redundant if caller checks for
		   failsafe_mode, but just to be sure: silently skip writing */

		/* emergency mode, don't work with filesystem */
		return CDW_CANCEL;
	}

	cdw_assert (config_file_fullpath != (char *) NULL, "full path to config file is null\n");

	FILE *config_file = fopen(config_file_fullpath, "w");
	if (config_file == (FILE *) NULL) {
		int e = errno;
		cdw_vdm ("ERROR: failed open config file \"%s\" for writing\n", config_file_fullpath);

		if (e == EACCES) { /* conf file has no write permissions */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("An error occurred when saving configuration. Please check config file permissions. Any changes will be lost after closing cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Unknown error occurred when saving configuration. Any changes will be lost after closing cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}

		return CDW_ERROR;
	} else {
		cdw_config_var_write_to_file(config_file, &global_config);
		fclose(config_file);
		config_file = (FILE *) NULL;
		return CDW_OK;
	}
}





/**
 * \brief Read config file, put config values into config variable (high-level)
 *
 * Open config file, call function reading config file, close config
 * file. If config file does not exists, it is not created.
 *
 * If configuration module works in failsafe mode, this function silently skips
 * reading from file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \return CDW_OK if configuration was read from file without problems
 * \return CDW_NO if there was no configuration file and now we are using default configuration
 * \return CDW_CANCEL if failsafe mode is in effect and reading was skipped (program is using default configuration now)
 * \return CDW_GEN_ERROR if function failed to open config file
 */
cdw_rv_t cdw_config_read_from_file(const char *fullpath)
{
	if (failsafe_mode) {
		cdw_vdm ("ERROR: failsafe mode is in effect, not reading config file\n");
		return CDW_CANCEL; /* emergency mode, don't work with filesystem */
	}

	cdw_assert (fullpath != (char *) NULL, "full path to config file is null\n");

	FILE *config_file = fopen(fullpath, "r");
	if (config_file == (FILE *) NULL) {
		int e = errno;
		cdw_vdm ("WARNING: failed open config file \"%s\" for reading (errno = \"%s\")\n",
			 fullpath, strerror(e));

		if (e == ENOENT) { /* no such file */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file problems"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot open config file. Will use default configuration."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			return CDW_NO;
		} else if (e == EACCES) { /* file permission problems */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot open config file. Please check file permissions."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_ERROR;
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Unknown error occurred when reading configuration. Default values will be used."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_ERROR;
		}
	} else { /* config file already exists and it is now open to read */
		cdw_rv_t r = cdw_config_var_read_from_file(config_file, &global_config);
		fclose(config_file);
		config_file = (FILE *) NULL;
		if (r == CDW_OK) {
			return CDW_OK; /* configuration read from existing configuration file */
		} else { /* problems with reading config file */
			cdw_vdm ("ERROR: failed to read from config file\n");
			return CDW_ERROR;
		}
	}
}





/**
   \brief Set path to cdw config file and to "base directory"

   Function uses call to cdw fs module to get path to home dir, which is
   used to initialize two paths used by config module:
   Base directory full path (char *base_dir_fullpath)
   Full config file path (char *config_file_fullpath)

   If path to user home directory can't be obtained, the function tries
   to get path to tmp directory. If this fails too, function returns
   CDW_ERROR and both paths are set to NULL.

   If function finishes successfully, dir paths in the two variables are
   the same, the difference is in file name in config_file_fullpath.
   Base dir path is ended with slash.

   On success, if function used user home directory, CDW_OK is returned.
   If function used tmp directory, CDW_NO is returned.

   \return CDW_OK if base_dir_fullpath is set to $HOME
   \return CDW_NO if base_dir_fullpath is set to tmp dir
   \return CDW_ERROR function can't get any path because of errors
*/
cdw_rv_t cdw_config_module_set_paths(void)
{
	cdw_assert (base_dir_fullpath == (char *) NULL,
		    "ERROR: called this function when base dir is already initialized\n");
	cdw_assert (config_file_fullpath == (char *) NULL,
		    "ERROR: called this function when base dir is already initialized\n");

	base_dir_fullpath = cdw_fs_get_home_or_tmp_dirpath();
	if (base_dir_fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create base dir fullpath\n");
		return CDW_ERROR;
	}

	/* at this point we have path to user home directory
	   or some tmp directory; now we set path to conf file */
	config_dir_fullpath = cdw_config_init_config_dir(base_dir_fullpath);
	if (config_dir_fullpath == (char *) NULL) {
		free(base_dir_fullpath);
		base_dir_fullpath = (char *) NULL;
		cdw_vdm ("ERROR: failed to create base dir fullpath\n");
		return CDW_ERROR;
	}

	config_file_fullpath = cdw_string_concat(config_dir_fullpath, config_file_name, (char *) NULL);
	if (config_file_fullpath == (char *) NULL) {
		free(base_dir_fullpath);
		base_dir_fullpath = (char *) NULL;
		free(config_dir_fullpath);
		config_dir_fullpath = (char *) NULL;
		cdw_vdm ("ERROR: failed to create config file fullpath\n");
		return CDW_ERROR;
	}

	old_config_file_fullpath = cdw_string_concat(base_dir_fullpath, old_config_file_name, (char *) NULL);
	if (old_config_file_fullpath == (char *) NULL) {
		free(base_dir_fullpath);
		base_dir_fullpath = (char *) NULL;
		free(config_dir_fullpath);
		config_dir_fullpath = (char *) NULL;
		free(config_file_fullpath);
		config_file_fullpath = (char *) NULL;
		cdw_vdm ("ERROR: failed to create config file fullpath\n");
		return CDW_ERROR;
	}

	cdw_vdm ("INFO: base directory:          \"%s\"\n", base_dir_fullpath);
	cdw_vdm ("INFO: configuration directory: \"%s\"\n", config_dir_fullpath);
	cdw_vdm ("INFO: configuration file:      \"%s\"\n", config_file_fullpath);

	if (cdw_string_starts_with_ci(base_dir_fullpath, "/tmp/")) {
		return CDW_NO; /* base dir is in tmp dir */
	} else {
		return CDW_OK; /* base dir is not in tmp dir */
	}
}





char *cdw_config_init_config_dir(const char *base_dir)
{
	char *dir = cdw_string_concat(base_dir, ".cdw/", (char *) NULL);
	if (dir == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat config dir\n");
		return (char *) NULL;
	}

	int rv = cdw_fs_check_existing_path(dir, R_OK | W_OK, CDW_FS_DIR);
	if (rv == 0) {
		;
	} else if (rv == ENOENT) {
		int d = mkdir(dir, S_IRWXU);
		int e = errno;
		if (d != 0) {
			cdw_vdm ("ERROR: failed to create new dir \"%s\", reason: \"%s\"\n", dir, strerror(e));
			free(dir);
			dir = (char *) NULL;
		}
	} else {
		cdw_vdm ("ERROR: invalid dir path \"%s\"\n", dir);
		free(dir);
		dir = (char *) NULL;
	}

	return dir;
}





const char *cdw_config_get_config_dir(void)
{
	return config_dir_fullpath;
}





/**
   \brief Deallocate 'configuration' module resources

   Deallocate all 'configuration' module resources that were allocated during
   program run and were not freed before.
*/
void cdw_config_module_clean(void)
{
	if (config_file_fullpath != (char *) NULL) {
		/* some changes could be made outside of cdw configuration window,
		   this line ensures that the changes will be saved */
		cdw_config_write_to_file();

		/* part of migrating cdw config file to new location */
		if (old_config_file_fullpath != (char *) NULL) {
			if (!access(old_config_file_fullpath, F_OK)) {
				unlink(old_config_file_fullpath);
			}
		}
	} else {
		/* may happen if config module wasn't correctly
		   initialized, e.g. when there are some problems
		   at startup of application */
		cdw_vdm ("ERROR: path to config file not initialized\n");
	}

	if (config_file_fullpath != (char *) NULL) {
		free(config_file_fullpath);
		config_file_fullpath = (char *) NULL;
	}

	if (old_config_file_fullpath != (char *) NULL) {
		free(old_config_file_fullpath);
		old_config_file_fullpath = (char *) NULL;
	}

	if (config_dir_fullpath != (char *) NULL) {
		free(config_dir_fullpath);
		config_dir_fullpath = (char *) NULL;
	}

	if (base_dir_fullpath != (char *) NULL) {
		free(base_dir_fullpath);
		base_dir_fullpath = (char *) NULL;
	}

	cdw_config_var_free_fields(&global_config);

	return;
}





/**
   \brief Copy values of options from one configuration variable to other

   Function does not validate \p src nor \p dest

   \param dest - config variable to be used as destination of copied values
   \param src - config variable to be used as source of copied values

   \return CDW_ERROR if function failed to copy one of fields
   \return CDW_OK on success
*/
cdw_rv_t cdw_config_var_copy(cdw_config_t *dest, cdw_config_t *src)
{
	cdw_rv_t ss = CDW_ERROR;


	/* page A (writing) */
	dest->write_pad = src->write_pad;
	dest->write_pad_size = src->write_pad_size;
	dest->erase_mode = src->erase_mode;
	dest->eject = src->eject;
	dest->burnproof = src->burnproof;
	ss = cdw_string_set(&(dest->other_cdrecord_options), src->other_cdrecord_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_cdrecord_options from src = \"%s\"\n", src->other_cdrecord_options);
		return CDW_ERROR;
	}
	ss = cdw_string_set(&(dest->other_growisofs_options), src->other_growisofs_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_growisofs_options from src = \"%s\"\n", src->other_growisofs_options);
		return CDW_ERROR;
	}
	ss = cdw_string_set(&(dest->other_xorriso_burn_options), src->other_xorriso_burn_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_xorriso_burn_options from src = \"%s\"\n", src->other_xorriso_burn_options);
		return CDW_ERROR;
	}


	/* page B (hardware) */
	strncpy(dest->custom_drive, src->custom_drive, OPTION_FIELD_LEN_MAX);
	dest->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->selected_drive, src->selected_drive, OPTION_FIELD_LEN_MAX);
	dest->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->scsi, src->scsi, OPTION_FIELD_LEN_MAX);
	dest->scsi[OPTION_FIELD_LEN_MAX] = '\0';


	/* page C (audio) */
	ss = cdw_string_set(&(dest->audiodir), src->audiodir);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set audiodir from src = \"%s\"\n", src->audiodir);
		return CDW_ERROR;
	}


	/* page D (iso filesystem) */
	strcpy(dest->volume_id, src->volume_id);
	strcpy(dest->volume_set_id, src->volume_set_id);
	strcpy(dest->preparer, src->preparer);
	strcpy(dest->publisher, src->publisher);
	strcpy(dest->system_id, src->system_id);
	strcpy(dest->copyright, src->copyright);
	strcpy(dest->abstract, src->abstract);

	dest->iso_level = src->iso_level;
	dest->joliet = src->joliet;
	dest->rock_ridge = src->rock_ridge;
	dest->joliet_long = src->joliet_long;
	dest->follow_symlinks = src->follow_symlinks;
	ss = cdw_string_set(&(dest->iso_image_full_path), src->iso_image_full_path);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set iso_image_full path from src = \"%s\"\n", src->iso_image_full_path);
		return CDW_ERROR;
	}
	ss = cdw_string_set(&(dest->boot_disc_options), src->boot_disc_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set boot_disc options = \"%s\"\n", src->boot_disc_options);
		return CDW_ERROR;
	}
	ss = cdw_string_set(&(dest->mkisofs_root_dir), src->mkisofs_root_dir);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set root_dir from src = \"%s\"\n", src->mkisofs_root_dir);
		return CDW_ERROR;
	}
	ss = cdw_string_set(&(dest->other_mkisofs_options), src->other_mkisofs_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_mkisofs_options from src = \"%s\"\n", src->other_mkisofs_options);
		return CDW_ERROR;
	}
	ss = cdw_string_set(&(dest->other_xorriso_iso_options), src->other_xorriso_iso_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_xorriso_options from src = \"%s\"\n", src->other_xorriso_iso_options);
		return CDW_ERROR;
	}


	/* page F (log and other) */
	ss = cdw_string_set(&(dest->log_full_path), src->log_full_path);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set log_full_path from src = \"%s\"\n", src->log_full_path);
		return CDW_ERROR;
	}
	dest->showlog = src->showlog;

	dest->volume_size_id = src->volume_size_id;
	dest->volume_size_custom_value = src->volume_size_custom_value;

	/* other, without field in config window */
	dest->support_dvd_rp_dl = src->support_dvd_rp_dl;
	dest->show_dvd_rw_support_warning = src->show_dvd_rw_support_warning;
	dest->fs.display_hidden_files = src->fs.display_hidden_files;

	return CDW_OK;
}





/**
   \brief Check if values in config variable are valid

   Check values stored in given configuration variable.
   Function checks only one specified subset of all config variable fields,
   or all verifiable config variable fields, depending on value of \p page.
   Use page symbolic name PAGE_{A,B,C,D,E}_INDEX to select one, specific
   subset corresponding to page in configuration window (see definitions
   in cdw_config_ui.h). Set \p page to -1 if you want to check all
   verifiable values from all pages. Page numbers correspond to
   configuration pages created in options_ui.c file, and fields of
   configuration variable are grouped just like fields in options_ui.c.

   If page, for which caller requested validation (or any page if \p page
   is set to -1) contains errors, then \p page will be set to number of
   page that has errors, and \p field will be set to number of invalid
   field in that page. CDW_NO is returned.

   If there are no invalid fields, value of \p field is unspecified, and
   CDW_OK is returned.

   Currently only part of field values are checked. E.g. 'pad' field is
   not checked.

   \param config - configuration variable to be checked
   \param page - number of page, from which parameters should be checked
   \param field - index of field with invalid value

   \return CDW_OK if all checked fields are valid
   \return CDW_NO if some field is invalid and the error must be fixed
*/
cdw_rv_t cdw_config_var_validate(cdw_config_t *config, cdw_id_t *page, int *field)
{
	/* IDs are defined in cdw_config.h  */

	cdw_rv_t crv;
	/* hardware */
	if (*page == -1 || *page == CONFIG_PAGE_ID_HW) {
		/* FIXME: use ID instead of string */
		if (!strcmp(config->selected_drive, "custom")) {
			/* user has selected "use custom drive path, and
			   the path cannot be empty; we only check if
			   the path is empty, we don't check if path is
			   in any way valid */

			if (!strlen(config->custom_drive)) {
				*field = f_custom_drive_i;
				*page = CONFIG_PAGE_ID_HW;
				return CDW_NO;
			}
		}

		crv = cdw_string_security_parser(config->custom_drive, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_custom_drive_i;
			*page = CONFIG_PAGE_ID_HW;
			return CDW_NO;
		}
		/* device path with ending slashes is not accepted by
		   other modules; info in config window asks to provide
		   device path without ending slashes, but let's
		   validate and fix path entered by user */
		size_t len = strlen(config->custom_drive);
		if (len > 1) {
			size_t i = 0;
			for (i = len - 1; i > 0; i--) {
				if (config->custom_drive[i] == '/') {
					config->custom_drive[i] = '\0';
				} else {
					break;
				}
			}
		}

		if (strlen(config->scsi)) {
			crv = cdw_string_security_parser(config->scsi, (char *) NULL);
			if (crv == CDW_NO) {
				*field = f_scsi_i;
				*page = CONFIG_PAGE_ID_HW;
				return crv;
			}
		}

	}

	if (*page == -1 || *page == CONFIG_PAGE_ID_AUDIO) { /* "audio" page */
		crv = cdw_string_security_parser(config->audiodir, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_audiodir_i;
			*page = CONFIG_PAGE_ID_AUDIO;
			return CDW_NO;
		}
	}


	if (*page == -1 || *page == CONFIG_PAGE_ID_TOOLS) { /* "tools" page */
		; /* tool paths are read from "which" output, I won't validate this */
		; /* WARNING for the future: keep in mind that at some
		     occasions "tools" page is hidden, and it must not be
		     validated then, because in case of errors user will
		     be unable to visit the page to fix the errors */
	}

	if (*page == -1 || *page == CONFIG_PAGE_ID_OTHER) { /* "log and other" page */
		if (!strlen(config->log_full_path)) {
			/* there has to be some log path specified */
			*field = f_log_fp_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}
		crv = cdw_string_security_parser(config->log_full_path, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_log_fp_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}
		if (config->volume_size_custom_value == -1) {
			*field = f_cust_volume_size_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}
	}

	return CDW_OK;
}





/**
   \brief Properly initialize fields of data structure of type cdw_config_t

   Set/initialize/allocate some fields of given variable of type
   cdw_config_t, so that it can be used safely by other C functions,
   i.e. there are no uninitialized pointers, and all non-NULL
   pointers point to some malloc()ed space.

   It is a good idea to call this function right after defining
   a cdw_config_t variable. You have to call this function before
   performing any operations on the variable.

   This function is different than cdw_config_var_set_defaults(),
   because cdw_config_var_set_defaults() is aware of cdw option values
   and sets higher-level values of fields that cdw_config_var_init_fields()
   only initialize with simple, low-level states.

   \p config must be valid, non-null pointer

   \param config - configuration variable with fields to initialize;

   \return CDW_OK on success
   \return CDW_ERROR on malloc() error
*/
cdw_rv_t cdw_config_var_init_fields(cdw_config_t *config)
{
	cdw_assert (config != (cdw_config_t *) NULL, "passing null config variable\n");

	/* page A (writing) */
	config->other_cdrecord_options = (char *) NULL;
	config->other_growisofs_options = (char *) NULL;
	config->other_xorriso_burn_options = (char *) NULL;
	config->log_full_path = (char *) NULL;


	/* page C (audio) */
	config->audiodir = (char *) NULL;


	/* page D (iso filesystem) */
	config->volume_id[0] = '\0'; /* empty string */
	config->volume_set_id[0] = '\0';
	config->preparer[0] = '\0';
	config->publisher[0] = '\0';
	config->system_id[0] = '\0';
	config->copyright[0] = '\0';
	config->abstract[0] = '\0';

	config->iso_image_full_path = (char *) NULL;
	config->boot_disc_options = (char *) NULL;
	config->mkisofs_root_dir = (char *) NULL;
	config->other_mkisofs_options = (char *) NULL;
	config->other_xorriso_iso_options = (char *) NULL;


	/* page E (external tools) */
	/* paths external tools are initialized by other module;
	   the paths aren't tightly related to options in this file */

	return CDW_OK;
}





/**
   \brief Deallocate all resources used by given cdw_config_t variable

   Free all memory that is referenced by fields (pointers) in given
   cdw_config_t variable.

   This function can recognize unallocated resources
   and can handle them properly.

   \param _config - config variable with fields to free

   \return CDW_OK otherwise
*/
cdw_rv_t cdw_config_var_free_fields(cdw_config_t *config)
{
	cdw_assert (config != (cdw_config_t *) NULL, "passing null config pointer\n");

	/* page A - writing */
	if (config->other_cdrecord_options != (char *) NULL) {
		free(config->other_cdrecord_options);
		config->other_cdrecord_options = (char *) NULL;
	}
	if (config->other_growisofs_options != (char *) NULL) {
		free(config->other_growisofs_options);
		config->other_growisofs_options = (char *) NULL;
	}
	if (config->other_xorriso_burn_options != (char *) NULL) {
		free(config->other_xorriso_burn_options);
		config->other_xorriso_burn_options = (char *) NULL;
	}


	/* page C - audio */
	if (config->audiodir != (char *) NULL) {
		free(config->audiodir);
		config->audiodir = (char *) NULL;
	}


	/* page D - iso filesystem */
	if (config->iso_image_full_path != (char *) NULL) {
		free(config->iso_image_full_path);
		config->iso_image_full_path = (char *) NULL;
	}
	if (config->boot_disc_options != (char *) NULL) {
		free(config->boot_disc_options);
		config->boot_disc_options = (char *) NULL;
	}
	if (config->mkisofs_root_dir != (char *) NULL) {
		free(config->mkisofs_root_dir);
		config->mkisofs_root_dir = (char *) NULL;
	}
	if (config->other_mkisofs_options != (char *) NULL) {
		free(config->other_mkisofs_options);
		config->other_mkisofs_options = (char *) NULL;
	}
	if (config->other_xorriso_iso_options != (char *) NULL) {
		free(config->other_xorriso_iso_options);
		config->other_xorriso_iso_options = (char *) NULL;
	}


	/* page F - log */
	if (config->log_full_path != (char *) NULL) {
		free(config->log_full_path);
		config->log_full_path = (char *) NULL;
	}

	return CDW_OK;
}





bool cdw_config_has_scsi_device(void)
{
	if (!strlen(global_config.scsi)) {
		return false;
	} else {
		return true;
	}
}




/* ***************************** */
/* ***** private functions ***** */
/* ***************************** */





/**
   \brief Set default values in given cdw configuration variable

   Set default values of fields in given configuration variable.
   This function can be called to make sure that some default values
   in configuration variable exist - just in case if config file does
   not exists or is broken or incomplete.

   This function depends on base_dir_fullpath to be set. If it is not set to some
   valid directory (which is indicated by failsafe_mode set to true), last
   attempt is made to set base dir to sane value: is is set to "/tmp".

   \param config - config variable in which to set default values

   \return CDW_ERROR on malloc() errors
   \return CDW_OK on success
 */
cdw_rv_t cdw_config_var_set_defaults(cdw_config_t *config)
{
	cdw_assert (base_dir_fullpath != (char *) NULL, "base_dir_fullpath must not be null\n");


	/* page A - writing */
	config->write_pad = true;
	/* the "150" value is selected after small tests with DVD; originally it was 63, which
	   worked just fine for CDs, but for DVDs it was insufficient;
	   TODO: the same value is used in code creating configuration forms,
	   so it should be a constant defined in some header */
	/* OLD COMMENT: value of 63 is taken from here:
	   http://www.troubleshooters.com/linux/coasterless.htm */
	config->write_pad_size = 150;
	config->erase_mode = CDW_ERASE_MODE_FAST;
	config->eject = false;
	config->burnproof = true;

	/* can't set to (char *) NULL because these options are used as
	argument of concat() in some places */
	cdw_string_set(&(config->other_cdrecord_options), " ");
	if (config->other_cdrecord_options == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set other_cdrecord_options\n");
		return CDW_ERROR;
	}
	cdw_string_set(&(config->other_growisofs_options), " ");
	if (config->other_growisofs_options == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set other_growisofs_options\n");
		return CDW_ERROR;
	}
	cdw_string_set(&(config->other_xorriso_burn_options), " ");
	if (config->other_xorriso_burn_options == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set other_xorriso_burn_options\n");
		return CDW_ERROR;
	}


	/* page B - hardware */
	strcpy(config->custom_drive, "");
	strcpy(config->selected_drive, "default");
	strcpy(config->scsi,""); /* some (most?) users will prefer to leave it empty */


	/* page C - audio */
	cdw_rv_t crv = CDW_NO;
	if (failsafe_mode) {
		crv = cdw_string_set(&(config->audiodir), "/tmp/audio/");
	} else {
		char *tmp2 = cdw_string_concat(base_dir_fullpath, "audio/", (char *) NULL);
		if (tmp2 == (char *) NULL) {
			cdw_vdm ("ERROR: failed to concat string for audiodir\n");
			return CDW_ERROR;
		} else {
			crv = cdw_string_set(&(config->audiodir), tmp2);

			free(tmp2);
			tmp2 = (char *) NULL;
		}
	}
	if (config->audiodir == (char *) NULL || crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set audiodir\n");
		return CDW_ERROR;
	}



	/* page D - iso filesystem */
	/* 2TRANS: this is a default label for ISO9660 volume */
	strncpy(config->volume_id, _("my volume"), CDW_ISO9660_VOLI_LEN);
	config->volume_id[CDW_ISO9660_VOLI_LEN] = '\0';
	config->iso_level = 3;
	config->joliet = true;
	config->rock_ridge = CDW_ISO9660_RR_USEFUL;
	config->joliet_long = false; /* breaks a standard, use with care */
	config->follow_symlinks = false;

	cdw_string_set(&(config->iso_image_full_path), "/tmp/image.iso");
	if (config->iso_image_full_path == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set iso_image_full_path\n");
		return CDW_ERROR;
	}
	cdw_string_set(&(config->boot_disc_options), "");
	/* can't set to (char *) NULL because these options are used as
	   argument of concat() in some places */
	cdw_string_set(&(config->mkisofs_root_dir), " ");
	cdw_string_set(&(config->other_mkisofs_options), " ");
	cdw_string_set(&(config->other_xorriso_iso_options), " ");


	/* page F - log and other */
#ifndef NDEBUG
	size_t len = strlen(base_dir_fullpath);
	cdw_assert (base_dir_fullpath[len - 1] == '/',
		    "ERROR: base_dir_fullpath is not ended with slash: \"%s\"\n",
		    base_dir_fullpath);
#endif
	char *tmp = cdw_string_concat(config_dir_fullpath, cdw_log_file_name, (char *) NULL);
	if (tmp == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat string for log_full_path\n");
		return CDW_ERROR;
	} else {
		cdw_string_set(&(config->log_full_path), tmp);
		free(tmp);
		tmp = (char *) NULL;

		if (config->log_full_path == (char *) NULL) {
			cdw_vdm ("ERROR: failed to set log_full_path to \"%s\"\n", tmp);
			return CDW_ERROR;
		}
	}

	config->showlog = true;

	config->volume_size_id = CDW_CONFIG_VOLUME_SIZE_CD74;
	if (config->volume_size_custom_value < 0) {
		config->volume_size_custom_value = 0;
	}

	/* other - not saved in config file */
	config->support_dvd_rp_dl = false;
	/* show warning dialog; dialog code will display a message box,
	   and will set this flag to "false", to not to annoy user with
	   repeated messages */
	config->show_dvd_rw_support_warning = true;
	config->fs.display_hidden_files = true;

	return CDW_OK;
}





/**
   \brief Write values from given configuration variable to cdw config file

   Write values from given configuration variable config to hard disk
   configuration file config_file. This is low level function that does actual
   writing to file. The file should be opened for writing before calling this
   function. This function will not check for validity of file nor
   configuration variable.

   This is the only function that writes to configuration file. If you
   would like to change layout of config file - you can do it here.

   \param config_file - config file already opened for writing
   \param config - configuration variable with option values to be written to file
*/
void cdw_config_var_write_to_file(FILE *config_file, cdw_config_t *config)
{
	fprintf(config_file, "####################################\n");
	fprintf(config_file, "#   %s %s configuration file       \n", PACKAGE, VERSION);
	fprintf(config_file, "####################################\n");


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "### \"Writing\" options\n");
	fprintf(config_file, "pad=%d\n", config->write_pad ? 1 : 0);
	fprintf(config_file, "pad_size=%d\n", config->write_pad_size);
	fprintf(config_file, "# fast / all\n");
	fprintf(config_file, "blank=%s\n", config->erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");
	fprintf(config_file, "eject=%d\n", config->eject ? 1 : 0);
	fprintf(config_file, "burnproof=%d\n", config->burnproof ? 1 : 0);

	fprintf(config_file, "other_cdrecord_options=%s\n", (config->other_cdrecord_options != (char *) NULL) ? config->other_cdrecord_options : "");
	fprintf(config_file, "other_growisofs_options=%s\n", (config->other_growisofs_options != (char *) NULL) ? config->other_growisofs_options : "");
	fprintf(config_file, "other_xorriso_burn_options=%s\n", (config->other_xorriso_burn_options != (char *) NULL) ? config->other_xorriso_burn_options : "");



	fprintf(config_file, "\n\n");
	fprintf(config_file, "### \"Hardware\" options\n");

	fprintf(config_file, "# string indicating drive to be used by cdw\n");
	fprintf(config_file, "selected_drive=%s\n", config->selected_drive);
	fprintf(config_file, "# manually entered path to a drive\n");
	fprintf(config_file, "custom_drive=%s\n", config->custom_drive);
	fprintf(config_file, "# SCSI device descriptor, used by cdrecord\n");
	fprintf(config_file, "scsi=%s\n", config->scsi);


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "### \"Audio\" options\n");
	fprintf(config_file, "audiodir=%s\n", (config->audiodir != (char *) NULL) ? config->audiodir : "");


	fprintf(config_file, "\n\n");
	fprintf(config_file, "### \"ISO9660 file system\" options\n");
	fprintf(config_file, "# allowed values for iso_level are 1, 2, 3, 4\n");
	fprintf(config_file, "iso_level=%lld\n", config->iso_level);

	fprintf(config_file, "joliet=%d\n", config->joliet ? 1 : 0);
	fprintf(config_file, "rock_ridge=%lld\n", config->rock_ridge);
	fprintf(config_file, "joliet_long=%d\n", config->joliet_long ? 1 : 0);
	fprintf(config_file, "follow_symlinks=%d\n", config->follow_symlinks ? 1 : 0);

	fprintf(config_file, "iso_image_full_path=%s\n", (config->iso_image_full_path != (char *) NULL) ? config->iso_image_full_path : "");
	fprintf(config_file, "# well, in fact this field stores all options related to boot disc\n");
	fprintf(config_file, "boot_image_path=%s\n", (config->boot_disc_options != (char *) NULL) ? config->boot_disc_options : "");
	fprintf(config_file, "other_mkisofs_options=%s\n", (config->other_mkisofs_options != (char *) NULL) ? config->other_mkisofs_options : "");
	fprintf(config_file, "other_xorriso_iso_options=%s\n", (config->other_xorriso_iso_options != (char *) NULL) ? config->other_xorriso_iso_options : "");

	fprintf(config_file, "\n\n");
	fprintf(config_file, "### \"Log and other\" options\n");

	fprintf(config_file, "showlog=%d\n", config->showlog ? 1 : 0);
	fprintf(config_file, "logfile=%s\n", (config->log_full_path != (char *) NULL) ? config->log_full_path : "");


	fprintf(config_file, "\n\n");
	fprintf(config_file, "cdsize=%s\n", cdw_config_volume_size_table[config->volume_size_id].new_label);
	fprintf(config_file, "cdsize=%s\n", cdw_config_volume_size_table[config->volume_size_id].old_label);
	fprintf(config_file, "user_cdsize=%ld\n", config->volume_size_custom_value);

	return;
}





/**
   \brief Read config file, store configuration in data structure (low-level code)

   Read config file line by line and store values found in that file into data
   structure. This is low level function that deals with disk file. It should
   be called by other function that will open file with correct permissions
   before passing it to this function.

   Data read from file is stored in \p config variable.

   This function calls cdw_string_security_parser() for every option
   value and additionally performs some basic validation of _some_ fields.
   When function finds some invalid value in config file, it discards it
   and doesn't modify existing value of field of \p config variable.

   \param config_file - config file opened for reading
   \param config - variable into which store values that were read

   \return CDW_OK on success
   \return CDW_ERROR on malloc() error
 */
cdw_rv_t cdw_config_var_read_from_file(FILE *config_file, cdw_config_t *config)
{
	cdw_rv_t ss = CDW_ERROR;

	char *line = (char *) NULL;
	while (1) {
		if (line != (char *) NULL) {
			free(line);
			line = (char *) NULL;
		}

		line = my_readline_10k(config_file);
		if (line == (char *) NULL) {
			break;
		}

		cdw_option_t option;
		if (!cdw_config_split_options_line(&option, line)) {
			continue; /* empty or invalid line, or comment */
		}

		cdw_rv_t v = cdw_string_security_parser(option.value, (char *) NULL);
		if (v != CDW_OK) {
			cdw_vdm ("WARNING: option value \"%s=%s\" from config file rejected as insecure\n",
				 option.name, option.value);
			continue;
		}


		/* page A - writing */
		if (!strcasecmp(option.name, "pad")) {
			cdw_string_get_bool_value(option.value, &(config->write_pad));
			continue;
		}

		if (!strcasecmp(option.name, "pad_size")) {
			config->write_pad_size = atoi(option.value);
			continue;
		}


		if (!strcasecmp(option.name, "blank")) {
			/* compare ignoring case, but store with correct case */
			if (!strcasecmp(option.value, "fast")) {
				config->erase_mode = CDW_ERASE_MODE_FAST;
			} else if (!strcasecmp(option.value, "all")) {
				config->erase_mode = CDW_ERASE_MODE_ALL;
			} else {
				; /* there should be some default value in option.name already */
			}
			continue;
		}

		if (!strcasecmp(option.name, "eject")) {
			cdw_string_get_bool_value(option.value, &(config->eject));
			continue;
		}

		if (!strcasecmp(option.name, "burnproof")) {
			cdw_string_get_bool_value(option.value, &(config->burnproof));
			continue;
		}

		if (!strcasecmp(option.name, "other_cdrecord_options")) {
			ss = cdw_string_set(&(config->other_cdrecord_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_cdrecord_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "other_growisofs_options")) {
			ss = cdw_string_set(&(config->other_growisofs_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_growisofs_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}

			continue;
		}
		if (!strcasecmp(option.name, "other_xorriso_burn_options")) {
			ss = cdw_string_set(&(config->other_xorriso_burn_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_xorriso_burn_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}

			continue;
		}


		/* page B - hardware */
		/* ** hardware ** */

		if (!strcasecmp(option.name, "custom_drive")) {
			if (!strlen(option.value)) {
				/* don't erase value that may already be set
				   after reading cdrw_device field from config file */
				;
			} else {
				strncpy(config->custom_drive, option.value, OPTION_FIELD_LEN_MAX);
				config->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
			}
			continue;
		}

		if (!strcasecmp(option.name, "selected_drive")) {
			strncpy(config->selected_drive, option.value, OPTION_FIELD_LEN_MAX);
			config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "scsi")) {
			strncpy(config->scsi, option.value, OPTION_FIELD_LEN_MAX);
			config->scsi[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}


		/* page C - audio */
		if (!strcasecmp(option.name, "audiodir")) {
			ss = cdw_string_set(&(config->audiodir), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set audiodir from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}



		/* page D - iso filesystem */
		if (!strcasecmp(option.name, "iso_level")) {
			int i = atoi(option.value);
			if (i < 1 || i > 4) {
				; /* probably "trash" value, rejecting completely */
			} else {
				config->iso_level = i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "joliet")) {
			cdw_string_get_bool_value(option.value, &(config->joliet));
			continue;
		}

		if (!strcasecmp(option.name, "rock_ridge")) {
			int i = atoi(option.value);
			if (i < 0 || i > 2) {
				; /* probably "trash" value, rejecting completely */
			} else {
				config->rock_ridge = i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "joliet_long")) {
			cdw_string_get_bool_value(option.value, &(config->joliet_long));
			continue;
		}

		if (!strcasecmp(option.name, "follow_symlinks")) {
			cdw_string_get_bool_value(option.value, &(config->follow_symlinks));
			continue;
		}

		if (!strcasecmp(option.name, "iso_image_full_path")) {
			ss = cdw_string_set(&(config->iso_image_full_path), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set iso_image_full_path from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "boot_image_path")) {
			ss = cdw_string_set(&(config->boot_disc_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set boot_disc_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}

			continue;
		}

		if (!strcasecmp(option.name, "other_mkisofs_options")) {
			ss = cdw_string_set(&(config->other_mkisofs_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_mkisofs_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "other_xorriso_iso_options")) {
			ss = cdw_string_set(&(config->other_xorriso_iso_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_xorriso_iso_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}


		/* page F - log and other */
		if (!strcasecmp(option.name, "logfile")) {
			ss = cdw_string_set(&(config->log_full_path), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set log_full_path from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "showlog")) {
			cdw_string_get_bool_value(option.value, &(config->showlog));
			continue;
		}

		if (!strcasecmp(option.name, "cdsize")) {
			cdw_id_t id = cdw_config_volume_id_by_label(option.value);
			if (id != -1) {
				config->volume_size_id = id;
			}
			continue;
		}

		if (!strcasecmp(option.name, "user_cdsize")) {
			config->volume_size_custom_value = strtol(option.value, (char **) NULL, 10);
			continue;
		}

	} /* while () */
	if (line != (char *) NULL) {
		free(line);
		line = (char *) NULL;
	}

	return CDW_OK;
}





/**
   \brief Split line containing '=' char into 'name' and 'value' part

   \date Function's top-level comment reviewed on 2012-02-07
   \date Function's body reviewed on 2012-02-07

   The function takes one 'char *' string, recognizes where is first
   '=' char in it. What is on the left side of the char is treated
   as option name, and what is on the right side of the char is treated
   as option value. Pointers to beginning of both name and value are placed
   in \p option. These pointers point to substrings of original string.

   Original string is modified so that strings representing name and value
   don't have any white chars at the beginning and end. '\0' chars are
   also inserted into original string (function's argument) to properly
   delimit the two substrings.

   The function heavily modifies its \p line argument!

   The function recognizes '#' as comment char and erases everything from
   line starting from '#' char;

   \p line must be proper char * string ending with '\0'

   Both fields of \p option are set to NULL if:
   \li function's \p line argument is NULL;
   \li function's \p line argument is empty;
   \li function's \p line argument does not contain '=' char;
   \li function's \p line argument is a comment line.
   \li function's \p line argument has '=' char but don't have any option name

   Otherwise the fields of \p option are set with valid pointers pointing
   to two substrings in function's \p line argument. Again: these substrings
   are proper C strings and are ended with '\0'.

   Don't use cdw_config_option_free_new() on \p option!

   This function is similar to cdw_config_split_options_line_new(),
   the difference being that name and value are substrings in \p line,
   and not freshly allocated strings.

   \param option - variable where pointers to substrings are placed
   \param line - line that you want to extract option from

   \return true on success (both fields of \p option are set)
   \return false otherwise
*/
bool cdw_config_split_options_line(cdw_option_t *option, char *line)
{
	cdw_sdm ("input line = '%s'\n", line);

	option->name = (char *) NULL;
	option->value = (char *) NULL;

	if (line == (char *) NULL) {
		return false;
	}

	/* this will make sure that comment, starting at any position
	   in line, will be erased from line */
	char *comment = strstr(line, "#"); /* beginning of in-line comment */
	if (comment != (char *) NULL) {
		size_t len = strlen(comment);
		size_t i = 0;
		for (i = 0; i < len; i++) {
			*(comment + i) = '\0';
		}
	}

	char *tline = cdw_string_ltrim(cdw_string_rtrim(line));
	char *eq = strstr(tline, "="); /* first occurrence of '=' in line */

	if (eq == (char *) NULL) {
		cdw_vdm ("ERROR: line is invalid: \"%s\"\n", line);

		return false;
	} else {
		option->name = tline;
		*eq = '\0';
		option->name = cdw_string_rtrim(option->name);

		if (!strcmp(option->name, "")) {
			option->name = (char *) NULL;
			option->value = (char *) NULL;
			return false;
		}

		option->value = cdw_string_ltrim(eq + 1);
		/* value may be empty string too, but this is not an error
		   condition, since given option value can be empty
		   (no value set) - this is correct situation */

		cdw_sdm ("   option.name = '%s'\n", option->name);
		cdw_sdm ("   option.value = '%s'\n", option->value);

		return true;
	}
}





/**
   \brief Split line containing '=' char into 'name' and 'value' part

   \date Function's top-level comment reviewed on 2012-02-07
   \date Function's body reviewed on 2012-02-07

   The function takes one 'char *' string, recognizes where is first
   '=' char in it. What is on the left side of the char is treated
   as option name, and what is on the right side of the char is treated
   as option value. Both name and value of an option are duplicated
   into appropriate fields of \p option. Both strings are trimmed of
   whitespaces.

   The function modifies its \p line argument!

   The function recognizes '#' as comment char and erases everything from
   line starting from '#' char;

   \p line must be proper char * string ending with '\0'

   Both fields of \p option are set to NULL if:
   \li function's \p line argument is NULL;
   \li function's \p line argument is empty;
   \li function's \p line argument does not contain '=' char;
   \li function's \p line argument is a comment line.
   \li function's \p line argument has '=' char but don't have any option name

   Otherwise the fields of \p option are set with freshly allocated
   strings: name and value of option.

   Use cdw_config_option_free_new() on \p option to conveniently
   deallocate both strings.

   This function is similar to cdw_config_split_options_line(), the
   difference being that name and value are freshly allocated strings,
   and not substrings in \p line.

   \param option - variable where new strings with name and value are placed
   \param line - line that you want to extract option from

   \return true on success (both fields of \p option are set)
   \return false otherwise
*/
bool cdw_config_split_options_line_new(cdw_option_t *option, char *line)
{
	if (!line) {
		cdw_vdm ("ERROR: input line is NULL\n");
		return false;
	}

	cdw_vdm ("INFO: input line = '%s'\n", line);

	option->name = (char *) NULL;
	option->value = (char *) NULL;


	/* this will make sure that comment, starting at any position
	   in line, will be erased from line */
	size_t n = strcspn(line, "#"); /* number of chars before start of comment */
	if (n != 0) {
		line[n] = '\0';
	}

	char *eq = strstr(line, "=");
	if (!eq) {
		cdw_vdm ("WARNING: line is invalid (\"%s\")\n", line);
		return false;
	}
	*eq = '\0';

	option->name = cdw_string_trim(line);
	option->value = cdw_string_trim(eq + 1);
	cdw_vdm ("INFO: trimmed name: \"%s\"\n", option->name);
	cdw_vdm ("INFO: trimmed value: \"%s\"\n", option->value);

	if (!option->name || !option->value || !strcmp(option->name, "")) {
		if (option->name) {
			free(option->name);
			option->name = (char *) NULL;
		}
		if (option->value) {
			free(option->value);
			option->value = (char *) NULL;
		}
		cdw_vdm ("ERROR: some string was empty\n");
		return false;
	}

	return true;
}





/**
   \brief Deallocate strings from an option

   \date Function's top-level comment reviewed on 2012-02-07
   \date Function's body reviewed on 2012-02-07

   Function deallocates two strings from given \p option.
   It checks for NULL pointers prior to deallocation, and sets
   pointers to NULL after deallocation.

   Call this function only for \p option processed by
   cdw_config_split_options_line_new().

   \param option - variable which you want to process
*/
void cdw_config_option_free_new(cdw_option_t *option)
{
	cdw_assert (option, "ERROR: pointer is NULL\n");

	if (option->value) {
		free(option->value);
		option->value = (char *) NULL;
	}

	if (option->name) {
		free(option->name);
		option->name = (char *) NULL;
	}

	return;
}





/* simple getter - some files use global config variable just to get this field */
const char *cdw_config_get_custom_drive(void)
{
	return global_config.custom_drive;
}





const char *cdw_config_get_scsi_drive(void)
{
	return global_config.scsi;
}





/* get either custom value of ISO9660 volume size,
   or one of predefined volume size values */
long int cdw_config_get_current_volume_size_value(void)
{
	cdw_assert (global_config.volume_size_id != CDW_CONFIG_VOLUME_SIZE_AUTO,
		    "ERROR: current volume size ID is \"auto\", so you should fetch volume size from disc\n");

	if (global_config.volume_size_id == CDW_CONFIG_VOLUME_SIZE_CUSTOM) {
		/* value entered by user in configuration window */
		cdw_vdm ("INFO: returning custom value %ld\n", global_config.volume_size_custom_value);
		return global_config.volume_size_custom_value;
	} else {
		/* one of predefined values, corresponding to total
		   capacity of selected disc type */
		return cdw_config_get_volume_size_mb_by_id(global_config.volume_size_id);
	}
}




void cdw_config_debug_print_config(cdw_config_t *config)
{
	cdw_vdm ("\n\nINFO: configuration:\n\n");
#if 1
	cdw_vdm ("INFO: Page A, \"writing\" settings:\n");
	cdw_vdm ("INFO:                 erase mode = \"%s\"\n", config->erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");
	cdw_vdm ("INFO:                      eject = \"%s\"\n", config->eject ? "true" : "false");
	cdw_vdm ("INFO:                  write pad = \"%s\"\n", config->write_pad ? "true" : "false");
	cdw_vdm ("INFO:             write pad size = \"%d\"\n", config->write_pad_size);
	cdw_vdm ("INFO:                  burnproof = \"%s\"\n", config->burnproof ? "true" : "false");
	cdw_vdm ("INFO:     other cdrecord options = \"%s\"\n", config->other_cdrecord_options);
	cdw_vdm ("INFO:    other growisofs options = \"%s\"\n", config->other_growisofs_options);
	cdw_vdm ("INFO: other xorriso burn options = \"%s\"\n\n", config->other_xorriso_burn_options);
#endif


#if 1
	cdw_vdm ("INFO: Page B, \"hardware\" settings:\n");
	cdw_vdm ("INFO:   custom drive = \"%s\"\n", config->custom_drive);
	cdw_vdm ("INFO: selected drive = \"%s\"\n", config->selected_drive);
	cdw_vdm ("INFO:           scsi = \"%s\"\n\n", config->scsi);
#if 0 /* unused, maybe will be enabled in future */
	cdw_vdm ("INFO:     mountpoint = \"%s\"\n", config->mountpoint);
	cdw_vdm ("INFO:          cdrom = \"%s\"\n", config->cdrom);
#endif
#endif


#if 1

	cdw_vdm ("INFO: Page C, \"audio\" settings:\n");
	cdw_vdm ("INFO: audiodir = \"%s\"\n\n", config->audiodir);
#endif


#if 1
	cdw_vdm ("INFO: Page D, \"ISO filesystem\" settings:\n");
	cdw_vdm ("INFO:                 volume id = \"%s\"\n", config->volume_id);
	cdw_vdm ("INFO:                 iso level = \"%lld\"\n", config->iso_level);
	cdw_vdm ("INFO:                    joliet = \"%s\"\n", config->joliet ? "true" : "false");
	cdw_vdm ("INFO:                rock_ridge = \"%lld\"\n", config->rock_ridge);
	cdw_vdm ("INFO:               joliet long = \"%s\"\n", config->joliet_long ? "true" : "false");
	cdw_vdm ("INFO:           follow_symlinks = \"%s\"\n", config->follow_symlinks ? "true" : "false");
	cdw_vdm ("INFO:       iso image full path = \"%s\"\n", config->iso_image_full_path);
	cdw_vdm ("INFO:         boot disc options = \"%s\"\n", config->boot_disc_options);
	cdw_vdm ("INFO:          mkisofs root dir = \"%s\"\n", config->mkisofs_root_dir);
	cdw_vdm ("INFO:     other mkisofs options = \"%s\"\n", config->other_mkisofs_options);
	cdw_vdm ("INFO: other xorriso iso_options = \"%s\"\n\n", config->other_xorriso_iso_options);
#endif


#if 1
	cdw_vdm ("INFO: Page E, \"tools\" settings:\n");
	cdw_ext_tools_debug_print_config();
#endif


#if 1
	cdw_vdm ("INFO: Page F, \"logging and other\" settings:\n");
	cdw_vdm ("INFO:            log full path = \"%s\"\n", config->log_full_path);
	cdw_vdm ("INFO:                 show log = \"%s\"\n", config->showlog ? "true" : "false");
	cdw_vdm ("INFO:           volume size id = \"%s\" (%lld)\n",
		 cdw_utils_id_label_table_get_label(cdw_config_volume_size_items, config->volume_size_id),
		 config->volume_size_id);
	cdw_vdm ("INFO: volume size custom value = %ld\n", config->volume_size_custom_value);

#endif

#if 1
	cdw_vdm ("INFO: Other setting:\n");
	cdw_vdm ("INFO:           support DVD+RP DL = \"%s\"\n", config->support_dvd_rp_dl ? "true" : "false");
	cdw_vdm ("INFO: show DVD-RW support warning = \"%s\"\n", config->show_dvd_rw_support_warning ? "true" : "false");
	cdw_vdm ("INFO:        display hidden files = \"%s\"\n", config->fs.display_hidden_files ? "true" : "false");
#endif

	return;
}





bool cdw_config_support_dvd_rp_dl(void)
{
	return global_config.support_dvd_rp_dl;
}





bool cdw_config_follow_symlinks(void)
{
	return global_config.follow_symlinks;
}





#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */


static void test_cdw_config_split_options_line(void);
static void test_cdw_config_split_options_line_new(void);


void cdw_config_run_tests(void)
{
	fprintf(stderr, "testing cdw_config.c\n");

	test_cdw_config_split_options_line();
	test_cdw_config_split_options_line_new();

	fprintf(stderr, "done\n\n");

	return;
}





void test_cdw_config_split_options_line(void)
{
	fprintf(stderr, "\ttesting cdw_config_split_options_line()... ");

	struct {
		const char *line;
		bool expected_return_value;
		bool expected_null_name;
		bool expected_null_value;
		const char *expected_name;
		const char *expected_value;

	} input_data[] = {
		/* correct line */
		{ "option name=value string",
		  true,
		  false, false,
		  "option name", "value string" },

		/* correct line */
		{ "\t option name \t\t = \t value string \t",
		  true,
		  false, false,
		  "option name", "value string" },

		/* correct line, but no value */
		{ "\t option name = \t \t",
		  true,
		  false, false,
		  "option name", "" },

		/* correct line, but no value (2) */
		{ "\t option name = #value string\t \t",
		  true,
		  false, false,
		  "option name", "" },

		/* incorrect line, no '=' char (1) */
		{ "\t option name \t # = value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, no '=' char (2) */
		{ "\t option name \t  value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, all content is comment */
		{ " \t #\t option name = value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, empty line */
		{ "",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, empty line (2) */
		{ " \t \t    \t\t\t\t     \t               \t#    ",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, line = NULL */
		{ (char *) NULL,
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		{ (char *) NULL,
		  false,
		  false, true, /* <-- guard */
		  (char *) NULL, (char *) NULL }
	};

	/* remember that option fields are pointers to places inside
	   input strings, not pointers to newly allocated strings */
	cdw_option_t option;

	int i = 0;
	while (input_data[i].expected_null_name == input_data[i].expected_null_value) {
		char *line = (char *) NULL;
		if (input_data[i].line) {
			line = strdup(input_data[i].line);
		}

		bool rv = cdw_config_split_options_line(&option, line);
		cdw_assert (input_data[i].expected_return_value == rv, "ERROR: failed at function call#%d\n", i);

		if (input_data[i].expected_null_name) {
			cdw_assert (!option.name, "ERROR: name is not NULL (#%d)\n", i);
		} else {
			cdw_assert (option.name, "ERROR: name is NULL (#%d)\n", i);
		}

		if (input_data[i].expected_null_value) {
			cdw_assert (!option.value, "ERROR: value is not NULL (#%d)\n", i);
		} else {
			cdw_assert (option.value, "ERROR: value is NULL (#%d)\n", i);
		}

		if (!input_data[i].expected_null_name && input_data[i].expected_null_value) {

			cdw_assert (!strcmp(option.name, input_data[i].expected_name), "ERROR: unexpected name #%d: \"%s\"\n", i, option.name);
			cdw_assert (!strcmp(option.value, input_data[i].expected_value), "ERROR: unexpected value #%d: \"%s\"\n", i, option.value);
		}

		if (line) {
			free(line);
			line = (char *) NULL;
		}

		i++;
	}


	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_config_split_options_line_new(void)
{
	fprintf(stderr, "\ttesting cdw_config_split_options_line_new()... ");

	struct {
		const char *line;
		bool expected_return_value;
		bool expected_null_name;
		bool expected_null_value;
		const char *expected_name;
		const char *expected_value;

	} input_data[] = {
		/* correct line */
		{ "option name=value string",
		  true,
		  false, false,
		  "option name", "value string" },

		/* correct line */
		{ "\t option name \t\t = \t value string \t",
		  true,
		  false, false,
		  "option name", "value string" },

		/* correct line, but no value */
		{ "\t option name = \t \t",
		  true,
		  false, false,
		  "option name", "" },

		/* correct line, but no value (2) */
		{ "\t option name = #value string\t \t",
		  true,
		  false, false,
		  "option name", "" },

		/* incorrect line, no '=' char (1) */
		{ "\t option name \t # = value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, no '=' char (2) */
		{ "\t option name \t  value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, all content is comment */
		{ " \t #\t option name = value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, empty line */
		{ "",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, empty line (2) */
		{ " \t \t    \t\t\t\t     \t               \t#    ",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, line = NULL */
		{ (char *) NULL,
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		{ (char *) NULL,
		  false,
		  false, true, /* <-- guard */
		  (char *) NULL, (char *) NULL }
	};

	cdw_option_t option;
	option.name = (char *) NULL;
	option.value = (char *) NULL;
	int i = 0;
	while (input_data[i].expected_null_name == input_data[i].expected_null_value) {
		char *line = (char *) NULL;
		if (input_data[i].line) {
			line = strdup(input_data[i].line);
		}

		bool rv = cdw_config_split_options_line_new(&option, line);
		cdw_assert (input_data[i].expected_return_value == rv, "ERROR: failed at function call#%d\n", i);

		if (input_data[i].expected_null_name) {
			cdw_assert (!option.name, "ERROR: name is not NULL (#%d)\n", i);
		} else {
			cdw_assert (option.name, "ERROR: name is NULL (#%d)\n", i);
		}

		if (input_data[i].expected_null_value) {
			cdw_assert (!option.value, "ERROR: value is not NULL (#%d)\n", i);
		} else {
			cdw_assert (option.value, "ERROR: value is NULL (#%d)\n", i);
		}

		if (!input_data[i].expected_null_name && input_data[i].expected_null_value) {
			cdw_assert (!strcmp(option.name, input_data[i].expected_name), "ERROR: unexpected name #%d: \"%s\"\n", i, option.name);
			cdw_assert (!strcmp(option.value, input_data[i].expected_value), "ERROR: unexpected value #%d: \"%s\"\n", i, option.value);
		}

		cdw_config_option_free_new(&option);
		if (line) {
			free(line);
			line = (char *) NULL;
		}

		i++;
	}


	fprintf(stderr, "OK\n");

	return;
}



#endif /* #ifdef CDW_UNIT_TEST_CODE */
