/*
 * Copyright (C) 2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <diagnostics/extensions/stacktrace/frame_visitor.hpp>

#include <diagnostics/extensions/stacktrace/posix.hpp>
#include <diagnostics/extensions/stacktrace/process.hpp>
#include <diagnostics/extensions/stacktrace/error.hpp>

#include <vector>

#include <cstdio>

#define __STDC_FORMAT_MACROS
#include <inttypes.h>

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

//////////////////////////////////////////////////////////////////////
// Traceback::SourceVisitor

static const char *
addr2line_path()
{
  static std::string path;
  // Not totally thread-safe, but okay because the whole facility is
  // not thread-safe.
  if (path.empty()) {
    try {
      path = POSIX::SearchCommand
        ("addr2line", "/bin:/usr/bin:/usr/local/bin");
    } catch (POSIX::Error &) {
      path.assign("", 1);       // string with NUL character
    }
  }
  const char *p = path.c_str();
  if (*p) {
    return p;
  }
  return 0;
}

Traceback::SourceVisitor::~SourceVisitor()
{
}

// Workaround for addr2line not picking up some DSO debug symbols.
static std::string
message_dso_name(const char *dso)
{
  std::string name(POSIX::RealPath(dso));
  std::string name_dbg("/usr/lib/debug");
  name_dbg += name;
  if (POSIX::Exists(name_dbg)) {
    return name_dbg;
  } else {
    return name;
  }
}

bool
Traceback::SourceVisitor::WalkSource(const char *dso, void *addr)
{
  const char *addr2line_cmd = addr2line_path();
  if (!addr2line_cmd) {
    return false;
  }

  std::string realdso(message_dso_name(dso));
  char saddr[3 * sizeof(void *) + 1];
  snprintf(saddr, sizeof(saddr), "0x%" PRIxPTR,
           reinterpret_cast<uintptr_t>(addr));
  const char *argv[] = {
    "addr2line", "-C", "-f", "-i", "-e", realdso.c_str(), saddr, NULL
  };

  POSIX::ProcessResult res;
  try {
    POSIX::Run(addr2line_cmd, const_cast<char *const *>(argv),
                     NULL, res);
  } catch (POSIX::Error &) {
    return false;
  }
  if (!res.Error.empty()) {
    return false;
  }

  bool found = false;
  std::string::iterator p = res.Output.begin();
  const std::string::iterator end = res.Output.end();
  while (p != end) {
    // Line with the function name.
    std::string::iterator start = p;
    while (p != end && *p != '\n') {
      ++p;
    }
    if (p == end) {
      break;
    }
    std::string function(start, p);
    ++p;

    // Line with FILENAME:LINENO.
    start = p;
    while (p != end && *p != '\n') {
      ++p;
    }
    if (p == end) {
      break;
    }
    std::string::iterator nl = p;
    while (p != start && *p != ':') {
      --p;
    }
    if (p == start) {
      // no colon
      break;
    }
    std::string file(start, p);
    unsigned line;
    if (sscanf(res.Output.c_str() + (p - res.Output.begin()) + 1,
               "%u", &line) != 1) {
      break;
    }
    if (function == "??" && file == "??") {
      break;
    }
    Frame(function.c_str(), file.c_str(), line);
    found = true;
    p = nl + 1;
  }

  return found;
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

