/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file dtmf.c
 * \brief Dtmf actions related functions
 */

#include <ffgtk.h>

/** global dtmf action list */
static GList *psDtmfActionList = NULL;

/**
 * \brief Get dtmf action list
 * \return global dtmf action list
 */
GList *getDtmfActions( void ) {
	return psDtmfActionList;
}

void executeDtmfActions( gchar *pnCaller, gchar *pnCalled, gchar *pnCode ) {
	GList *psList = getDtmfActions();
	struct sDtmf *psDtmf = NULL;

	while ( psList != NULL ) {
		psDtmf = psList -> data;

		if ( psDtmf != NULL ) {
			gchar *pnTarget = pnCalled;

			pnTarget = strchr( pnCalled, '#' );
			if ( pnTarget == NULL ) {
				pnTarget = pnCalled;
			} else {
				pnTarget++;
			}

			Debug( KERN_DEBUG, "Checking caller: %s <-> %s\n", psDtmf -> pnCaller, pnCaller );
			Debug( KERN_DEBUG, "&& Checking caller: %s <-> %s\n", psDtmf -> pnCalled, pnTarget );
			Debug( KERN_DEBUG, "&& Checking code: %s <-> %s\n", psDtmf -> pnCode, pnCode );
			if ( !strcmp( psDtmf -> pnCaller, pnCaller ) && !strcmp( psDtmf -> pnCalled, pnTarget ) && !strcmp( psDtmf -> pnCode, pnCode ) ) {
				Debug( KERN_DEBUG, "Executing: %s\n", psDtmf -> pnExecute );
				g_spawn_command_line_async( psDtmf -> pnExecute, NULL );
			}
		}

		psList = psList -> next;
	}
}

gboolean validDtmfAction( gchar *pnCaller, gchar *pnCalled ) {
	GList *psList = getDtmfActions();
	struct sDtmf *psDtmf = NULL;

	while ( psList != NULL ) {
		psDtmf = psList -> data;

		if ( psDtmf != NULL ) {
			if ( !strcmp( psDtmf -> pnCaller, pnCaller ) && !strcmp( psDtmf -> pnCalled, pnCalled ) ) {
				return TRUE;
			}
		}

		psList = psList -> next;
	}

	return FALSE;
}

/**
 * \brief Add dtmf action from xmlnode
 * \param psNode xml node structure
 */
static void addDtmfAction( xmlnode *psNode ) {
	struct sDtmf *psDtmf = NULL;
	xmlnode *psChild = NULL;
	gchar *pnName = NULL;
	gchar *pnCaller = NULL;
	gchar *pnCalled = NULL;
	gchar *pnCode = NULL;
	gchar *pnExec = NULL;

	psChild = xmlnode_get_child( psNode, "name" );
	if ( psChild != NULL ) {
		pnName = xmlnode_get_data( psChild );

		psChild = xmlnode_get_child( psNode, "caller" );
		if ( psChild != NULL ) {
			pnCaller = xmlnode_get_data( psChild );

			psChild = xmlnode_get_child( psNode, "called" );
			if ( psChild != NULL ) {
				pnCalled = xmlnode_get_data( psChild );

				psChild = xmlnode_get_child( psNode, "code" );
				if ( psChild != NULL ) {
					pnCode = xmlnode_get_data( psChild );

					psChild = xmlnode_get_child( psNode, "exec" );
					if ( psChild != NULL ) {
						pnExec = xmlnode_get_data( psChild );

						psDtmf = g_malloc0( sizeof( struct sDtmf ) );

						if ( psDtmf != NULL ) {
							/* Ok, now add information to psDtmf and prepend it to dtmf action list */
							psDtmf -> pnName = g_strdup( pnName );
							psDtmf -> pnCaller = g_strdup( pnCaller );
							psDtmf -> pnCalled = g_strdup( pnCalled );
							psDtmf -> pnCode = g_strdup( pnCode );
							psDtmf -> pnExecute = g_strdup( pnExec );
							psDtmfActionList = g_list_prepend( psDtmfActionList, psDtmf );
						}

						g_free( pnExec );
					}

					g_free( pnCode );
				}

				g_free( pnCalled );
			}

			g_free( pnCaller );
		}

		g_free( pnName );
	}
}

/**
 * \brief Find dtmf action by name
 * \param pnName name to search for
 * \return dtmf structure or NULL
 */
struct sDtmf *findDtmfAction( const gchar *pnName ) {
	GList *psList = NULL;
	struct sDtmf *psDtmf = NULL;

	for ( psList = getDtmfActions(); psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psDtmf = psList -> data;
		if ( !strcmp( psDtmf -> pnName, pnName ) ) {
			return psDtmf;
		}
	}

	return NULL;
}

/**
 * \brief Convert dtmf action to xml node
 * \param psDtmf action structure
 * \return action xml node
 */
static xmlnode *dtmfActionToXmlnode( struct sDtmf *psDtmf ) {
	xmlnode *psNode = NULL;
	xmlnode *psChild = NULL;

	psNode = xmlnode_new( "dtmf" );

	psChild = xmlnode_new_child( psNode, "name" );
	xmlnode_insert_data( psChild, psDtmf -> pnName, -1 );
	psChild = xmlnode_new_child( psNode, "caller" );
	xmlnode_insert_data( psChild, psDtmf -> pnCaller, -1 );
	psChild = xmlnode_new_child( psNode, "called" );
	xmlnode_insert_data( psChild, psDtmf -> pnCalled, -1 );
	psChild = xmlnode_new_child( psNode, "code" );
	xmlnode_insert_data( psChild, psDtmf -> pnCode, -1 );
	psChild = xmlnode_new_child( psNode, "exec" );
	xmlnode_insert_data( psChild, psDtmf -> pnExecute, -1 );

	return psNode;
}

/**
 * \brief Convert dtmf actions to xml node
 * \return xml node containing actions
 */
static xmlnode *dtmfActionsToXmlnode(void){
	xmlnode *psNode = NULL;
	xmlnode *psChild = NULL;
	GList *psCurrent = NULL;

	psNode = xmlnode_new( "dtmf" );
	xmlnode_set_attrib( psNode, "version", "1.0" );

	for ( psCurrent = getDtmfActions(); psCurrent != NULL && psCurrent -> data != NULL; psCurrent = psCurrent -> next ) {
		psChild = dtmfActionToXmlnode( psCurrent -> data );
		xmlnode_insert_child( psNode, psChild );
	}

	return psNode;
}

/**
 * \brief Save dtmf action to disk
 */
void saveDtmfActions( void ) {
	xmlnode *psNode = NULL;
	gchar *pnData = NULL;

	psNode = dtmfActionsToXmlnode();

	pnData = xmlnode_to_formatted_str( psNode, NULL );

	if ( pnData != NULL ) {
		gint nFile = -1;
		gint nResult = 0;
		gchar *pnFile = g_build_filename( getUserDir(), "dtmf.xml", NULL );

		nFile = open( pnFile, O_RDWR | O_CREAT | O_TRUNC, 0600 );
		if ( nFile > 0 ) {
			nResult = write( nFile, pnData, strlen( pnData ) );
			if ( nResult != strlen( pnData ) ) {
				Debug( KERN_WARNING, "Could not save file %s\n", pnFile );
			}
			close( nFile );
		}
	}

	g_free( pnData );
	xmlnode_free( psNode );
}

/**
 * \brief Remove dtmf action by name
 * \param pnName dtmf action name we want to remove
 */
void removeDtmfAction( const gchar *pnName ) {
	struct sDtmf *psDtmf = findDtmfAction( pnName );

	if ( psDtmf != NULL ) {
		psDtmfActionList = g_list_remove( psDtmfActionList, psDtmf );
	}
}

/**
 * \brief Create new dtmf action with name
 * \param pnName dtmf action name
 * \return new dtmf structure
 */
struct sDtmf *createDtmfAction( const gchar *pnName ) {
	struct sDtmf *psDtmf = NULL;

	psDtmf = g_malloc0( sizeof( struct sDtmf ) );

	if ( psDtmf != NULL ) {
		psDtmf -> pnName = g_strdup( pnName );
		psDtmfActionList = g_list_prepend( psDtmfActionList, psDtmf );
	} else {
		Debug( KERN_WARNING, "Could not add '%s'\n", pnName );
	}

	return psDtmf;
}

/**
 * \brief Load dtmf from disk
 */
void DtmfActionsLoad( void ) {
	xmlnode *psNode = NULL;
	xmlnode *psChild = NULL;
	DIR *psDir = NULL;

	psDir = opendir( getUserDir() );
	if ( psDir == NULL ) {
		g_mkdir( getUserDir(), 0755 );
	} else {
		closedir( psDir );
	}

	psNode = readXmlFromFile( "dtmf.xml", _( "dtmf" ) );
	if ( psNode == NULL ) {
		Debug( KERN_DEBUG, "Could not read dtmf.xml\n" );
		return;
	}

	for ( psChild = xmlnode_get_child( psNode, "dtmf" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		addDtmfAction( psChild );
	}

	xmlnode_free( psNode );
}
