/*
 * =============================================================================
 * 
 *   Copyright (c) 2007-2010, The JASYPT team (http://www.jasypt.org)
 * 
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 * 
 *       http://www.apache.org/licenses/LICENSE-2.0
 * 
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 * 
 * =============================================================================
 */
package org.jasypt.spring3.properties;

import java.util.Properties;
import java.util.prefs.Preferences;

import org.jasypt.commons.CommonUtils;
import org.jasypt.encryption.StringEncryptor;
import org.jasypt.properties.PropertyValueEncryptionUtils;
import org.jasypt.util.text.TextEncryptor;
import org.springframework.beans.factory.config.PreferencesPlaceholderConfigurer;

/**
 * <p>
 * Subclass of
 * <tt>org.springframework.beans.factory.config.PreferencesPlaceholderConfigurer</tt>
 * which can make use of a {@link org.jasypt.encryption.StringEncryptor} or
 * {@link org.jasypt.util.text.TextEncryptor} object to decrypt preferences values
 * if they are encrypted in the loaded resource locations.
 * </p>
 * <p>
 * A value is considered "encrypted" when it appears surrounded by
 * <tt>ENC(...)</tt>, like:
 * </p>
 * <p>
 * <center> <tt>ENC(!"DGAS24FaIO$)</tt> </center>
 * </p>
 * 
 * @since 1.9.0 (existed in package org.jasypt.spring.properties since 1.7)
 * 
 * @author Daniel Fern&aacute;ndez
 * 
 */
public final class EncryptablePreferencesPlaceholderConfigurer 
        extends PreferencesPlaceholderConfigurer {
	/*
	 * Only one of these instances will be initialized, the other one will be
	 * null.
	 */
	private final StringEncryptor stringEncryptor;
	private final TextEncryptor textEncryptor;

	/**
	 * <p>
	 * Creates an <tt>EncryptablePreferencesPlaceholderConfigurer</tt> instance
	 * which will use the passed {@link StringEncryptor} object to decrypt
	 * encrypted values.
	 * </p>
	 * 
	 * @param stringEncryptor
	 *            the {@link StringEncryptor} to be used do decrypt values. It
	 *            can not be null.
	 */
	public EncryptablePreferencesPlaceholderConfigurer(
	        final StringEncryptor stringEncryptor) {
		super();
		CommonUtils.validateNotNull(stringEncryptor, "Encryptor cannot be null");
		this.stringEncryptor = stringEncryptor;
		this.textEncryptor = null;
	}

	/**
	 * <p>
	 * Creates an <tt>EncryptablePreferencesPlaceholderConfigurer</tt> instance which will use the
	 * passed {@link TextEncryptor} object to decrypt encrypted values.
	 * </p>
	 * 
	 * @param textEncryptor
	 *            the {@link TextEncryptor} to be used do decrypt values. It can
	 *            not be null.
	 */
	public EncryptablePreferencesPlaceholderConfigurer(final TextEncryptor textEncryptor) {
		super();
		CommonUtils.validateNotNull(textEncryptor, "Encryptor cannot be null");
		this.stringEncryptor = null;
		this.textEncryptor = textEncryptor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.springframework.beans.factory.config.PropertyResourceConfigurer#convertPropertyValue(java.lang.String)
	 */
	@Override
	protected String convertPropertyValue(final String originalValue) {
		if (!PropertyValueEncryptionUtils.isEncryptedValue(originalValue)) {
			return originalValue;
		}
		if (this.stringEncryptor != null) {
			return PropertyValueEncryptionUtils.decrypt(originalValue,
					this.stringEncryptor);

		}
		return PropertyValueEncryptionUtils.decrypt(originalValue, this.textEncryptor);
	}

    
    /*
     * (non-Javadoc)
     * 
     * @see org.springframework.beans.factory.config.PreferencesPlaceholderConfigurer#resolvePlaceholder(java.lang.String, java.util.Properties)
     */
    @Override
    protected String resolvePlaceholder(final String placeholder, final Properties props) {
        final String result = super.resolvePlaceholder(placeholder, props);
        return convertPropertyValue(result);
    }

    
    /*
     * (non-Javadoc)
     * 
     * @see org.springframework.beans.factory.config.PreferencesPlaceholderConfigurer#resolvePlaceholder(java.lang.String, java.lang.String, java.util.prefs.Preferences)
     */
    @Override
    protected String resolvePlaceholder(
            final String path, final String key, final Preferences preferences) {
        final String result = super.resolvePlaceholder(path, key, preferences);
        return convertPropertyValue(result);
    }
	
}
