/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


/***************************************************************************
   grid-proxy-verify.c

 Sample C program that verifies a Globus GSI proxy.
 The following checks are performed:
 - certificate chain is verified , including proxy certs
 - proxy itself is verified (SUBJ/CN=proxy vs ISSUER etc)
 - proxy private key is matched against proxy public key
 - file permissions of proxy file are checked

 Build instructions:
    gcc -o grid-proxy-verify grid-proxy-verify.c \
        -I<OPENSSL-INCLUDE> -L<OPENSSL-LIB> -lssl -lcrypto
 (code is CFLAGS="-Wall -g -Wuninitialized -O" clean)

 ***************************************************************************/

#define _GNU_SOURCE

#ifndef INTERNAL_VERIFY_X509
 #define INTERNAL_VERIFY_X509

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>


/* Include DataTypes and defines for verify_x509 */
#include "verify_x509_datatypes.h"


/* OpenSSL uses a default depth of 9.
 * To cope with Subordinate CAs we have to extend the verification depth
 * to be able to hold the certificate chain (could contain a lot of delegations)
 * and all the CA certificate, which might not be added to the certificate chain itself
 * but would still be lingering in the X509 CA directory lookup functions
 */
#define VERIFICATION_CHAIN_DEPTH_EXTENSION 9

unsigned long process_internal_verify_data (internal_verify_x509_data_t **);
unsigned long verify_x509_readPrivateKeyFromFile (char *filename, EVP_PKEY **pkey);
unsigned long verify_x509_readPrivateKeyFromPEM  (unsigned char * pem, EVP_PKEY **pkey);
unsigned long verify_x509_readPublicCertChain    (char *filename, STACK_OF(X509) **certstack);


size_t u_strlen (const unsigned char * s);
unsigned char * u_strcpy (unsigned char * dest, const unsigned char * src);



/* ASN1 time string (in a char *) to time_t */
/**
 *  (Use ASN1_STRING_data() to convert ASN1_GENERALIZEDTIME to char * if
 *   necessary)
 */

time_t grid_asn1TimeToTimeT(unsigned char *asn1time, size_t len);


/* Check if certificate can be used as a CA to sign standard X509 certs */
/*
 *  Return 1 if true; 0 if not.
 */

int grid_x509IsCA(X509 *cert);
int grid_X509_empty_callback(char *buf, int buf_size, int verify, void *cb_tmp);


unsigned long grid_X509_knownCriticalExts(X509 *cert);
unsigned long grid_readProxy( char *filename, STACK_OF(X509) **certstack, EVP_PKEY **pkey );

#if OPENSSL_VERSION_NUMBER < 0x00908000L
    /* We change the default callback to use our wrapper and discard errors
       due to GSI proxy chains (ie where users certs act as CAs) */
    int grid_X509_check_issued_wrapper(X509_STORE_CTX *ctx,X509 *x,X509 *issuer);
#endif


/******************************************************************************
Function:   grid_verifyProxy
Description:
    Tries to verify the proxies in the certstack
******************************************************************************/
unsigned long grid_verifyProxy( STACK_OF(X509) *certstack );


/******************************************************************************
Function:   verify_certificate_type_str
Description:
            lcmaps_proxy_type_t to human readable string
******************************************************************************/
const char *
verify_certificate_type_str(lcmaps_proxy_type_t cert_type);

/******************************************************************************
Function:   verify_PROXYCERTINFO_get_policy
Description:
            Get a policy from the PROXYCERTINFO structure
******************************************************************************/
PROXYPOLICY *
verify_PROXYCERTINFO_get_policy(PROXYCERTINFO *cert_info);

/******************************************************************************
Function:   verify_PROXYPOLICY_get_policy_language
Description:
            Get the proxy language from the proxy policy
******************************************************************************/
ASN1_OBJECT *
verify_PROXYPOLICY_get_policy_language(PROXYPOLICY *policy);

/******************************************************************************
Function:   grid_verifyPathLenConstraints
Description:
            This function will check the certificate chain on CA based and
            RFC3820 Proxy based Path Length Constraints.
Parameters:
    chain of certificates
Returns:
    0       : Not ok, failure in the verification or the verification failed
    1       : Ok, verification has succeeded and positive
******************************************************************************/
int grid_verifyPathLenConstraints (STACK_OF(X509) * chain);


/******************************************************************************
Function:   verify_callback
Description:
    Callback function for OpenSSL to put the errors
Parameters:
    ok, X509_STORE_CTX
Returns:
******************************************************************************/
/* static int grid_X509_verify_callback(int ok, X509_STORE_CTX *ctx); */
int grid_X509_verify_callback(int ok, X509_STORE_CTX *ctx);


/******************************************************************************
Function:   grid_verifyCert
Description:
    Validates a certificate with the CRL list and the signing CA
******************************************************************************/
unsigned long grid_verifyCert( char * CA_DIR, STACK_OF(X509) *certstack );


/******************************************************************************
Function:   verifyPrivateyKey
Description:
    Tries to verify a privatekey with the first cert in the certstack
******************************************************************************/
unsigned long grid_verifyPrivateKey( STACK_OF(X509) *certstack, EVP_PKEY *pkey );


/******************************************************************************
Function:   startVerifyProcess
Description:
    Start the read-in and processing of the rst cert in the certstack
******************************************************************************/
int startVerifyProcess (char * proxyfilename, char * CA_dir);


#endif /* INTERNAL_VERIFY_X509 */

