/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_verify_account_from_pem.c
    \brief  This interface returns the uid, gids and poolindex, i.e. leaseid,
            using LCMAPS and takes the user credential in PEM format as input
    \author Martijn Steenbakkers for the EU DataGrid.

    This interface returns the uid, gids and poolindex, i.e. leaseid, using LCMAPS.
    As input it requires the user credential in PEM format
    -# lcmaps_verify_account_from_pem: interface function
*/

#ifndef LCMAPS_VERIFY_ACCOUNT_FROM_PEM_C
#define LCMAPS_VERIFY_ACCOUNT_FROM_PEM_C

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

/* LCMAPS includes */
#include "lcmaps.h"
#include "lcmaps_config.h"
#include "lcmaps_log.h"
#include "lcmaps_verify_account_from_pem.h"

/******************************************************************************
                             Module specific defines
******************************************************************************/
/* Default settings */
#define LCMAPS_MAX_POLICIES 10
/* #define LCMAPS_LOG_FILE "/var/log/lcmaps-suexec.log" */
/* #define LCMAPS_LOG_STRING "verify_account_from_pem" */
/* #define LCMAPS_BUFSIZ 8192 */
/* #define LCMAPS_LOG_BUFSIZ 2048 */
/* #define LCMAPS_DEFAULT_DB_FILE "lcmaps.db" */
/* #define LCMAPS_REQUIRED_DB_PATH LCMAPS_ETC_HOME */

/* Flags to indicate which type of logging is required */
#define DO_USRLOG           ((unsigned short)0x0001)
#define DO_SYSLOG           ((unsigned short)0x0002)

/******************************************************************************
                       Define module specific variables
******************************************************************************/
/* static int              do_syslog = 1; */                /* use syslog for LCMAPS logging */
/*static int              lcmaps_log_stream_exists = 0; */  /* 1 if log_stream already exists at startup */
#if 0
static FILE *           lcmaps_log_stream = NULL;      /* stream for LCMAPS logging */
static char *           lcmaps_log_string = NULL;      /* (constant) log string to be prepended to log information */
static char *           lcmaps_log_file = NULL;
#endif

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
extern int lcmaps_tokenize( const char * command, char ** args, int * n, char * sep);
extern int lcmaps_verify_account_from_pem_va (int narg, ...);



/******************************************************************************
Function:   lcmaps_verify_account_from_pem_va
Description:
    Verify the account mapping from the supplied user credential.

Parameters:
    narg:  the number of arguments that follow
    ap:    vararg list, that consists of
        - input:  The PEM-encoded string containing the user proxy
        - input:  a structure that contains account information:
                  uid, gids (primary and secondary) and the poolindex
Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_verify_account_from_pem_va(
    int        narg,
    ...
)
{
    va_list                             ap;
    char *                              pem_string = NULL;        /* input variable */
    lcmaps_account_info_t               lcmaps_account;           /* input variable */
    char *                              lcmaps_policy_string = NULL;
    char *                              lcmaps_policies[LCMAPS_MAX_POLICIES];
    int                                 lcmaps_npols = LCMAPS_MAX_POLICIES;
    /* char *                              lcmaps_db_file_env = NULL; */
    /* char *                              lcmaps_db_file = NULL; */
    int                                 rc = -1;
    int                                 jj = 0;

#if 0
    time_t                              mclock;
    struct tm  *                        timestamp = NULL;
    size_t                              string_size;
    char *                              lcmaps_log_string_default = NULL;
#endif
    /* struct stat                         prg_info; */   /* program info holder       */
    /* struct stat                         dir_info; */   /* directory info holder     */

    /*
     * Take log file from environment variables.
     * If not defined take the default ones
     */
#if 0
    lcmaps_log_file = getenv("LCMAPS_LOG_FILE");
    lcmaps_log_file = (lcmaps_log_file ? lcmaps_log_file : LCMAPS_LOG_FILE );
#endif

    /*
     * The other environment variables should in principle not be touched,
     * because reasonably defaults are provided by the LCMAPS core.
     * An exception is made for LCMAPS_LOG_STRING, because we want to enforce
     * that it is logged which interface is used.
     * For traceability also add a timestamp.
     * setenv with overwrite=0 only overwrites if the variable is non-existant
     */
#if 0
    time(&mclock);
    /* timestamp = localtime(&mclock); */
    timestamp = gmtime(&mclock);
    string_size = strlen(LCMAPS_LOG_STRING) + strlen(": ") + 19 + 1;
    lcmaps_log_string_default = (char *) malloc(string_size * sizeof(char));

    snprintf(lcmaps_log_string_default, string_size, "%s: %04d-%02d-%02d.%02d:%02d:%02d",
        LCMAPS_LOG_STRING,
        timestamp->tm_year + 1900, timestamp->tm_mon + 1, timestamp->tm_mday,
        timestamp->tm_hour, timestamp->tm_min, timestamp->tm_sec);

    setenv("LCMAPS_LOG_STRING", lcmaps_log_string_default, 0);
    lcmaps_log_string = getenv("LCMAPS_LOG_STRING");

    if (lcmaps_log_string_default != NULL)
    {
        free(lcmaps_log_string_default);
        lcmaps_log_string_default = NULL;
    }
#endif
#if 0
    /*
     * Another exception is made for the lcmaps.db file: This interface is used by
     * VO services (i.e. not site-controlled services), so the VO should not be able
     * to determine which lcmaps.db file (or in the future a general lcmaps config file)
     * they take.
     */
    if ( (lcmaps_db_file_env = getenv("LCMAPS_DB_FILE")) != NULL)
    {
        /*
         * OK, one tries to set the location of the lcmaps.db file.
         * We don't allow paths to be included in the filename. LCMAPS will add the
         * site-controlled path to it.
         * If a path is found: error
         * If no path is found: add the default path to it and performs various checks
         */
        if (strchr(lcmaps_db_file_env, '/') != NULL)
        {
            fprintf(stderr, "%s: (LCMAPS_DB_FILE = %s)\n", lcmaps_log_string, lcmaps_db_file_env);
            fprintf(stderr,
                "%s: Attempt to include path to lcmaps.db file by environment variable\n", lcmaps_log_string);
            fprintf(stderr, "%s: This is not allowed for this lcmaps interface\n", lcmaps_log_string);
            return 1;
        }
    }
    else
    {
        lcmaps_db_file_env = LCMAPS_DEFAULT_DB_FILE;
    }
    /* prepend required path */
    lcmaps_db_file = (char *) malloc(strlen(LCMAPS_REQUIRED_DB_PATH) + 2 + strlen(lcmaps_db_file_env));
    sprintf(lcmaps_db_file, "%s/%s", LCMAPS_REQUIRED_DB_PATH, lcmaps_db_file_env);

    /*
     * Check lcmaps.db file:
     * - Should be in a directory owned by root, which non-writable for others.
     * - Should be owned by root and non-writable for others.
     * If checks are ok reset environment variable
     */
    /*
     * Stat the directory of lcmaps.db and verify it is a directory, or error out.
     */
    if (((lstat(LCMAPS_REQUIRED_DB_PATH, &dir_info)) != 0) || !(S_ISDIR(dir_info.st_mode)))
    {
        fprintf(stderr, "%s: cannot stat directory: (%s)\n", lcmaps_log_string, LCMAPS_REQUIRED_DB_PATH);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the directory is not owned by root
     */
    if ((dir_info.st_uid != 0) || (dir_info.st_gid != 0))
    {
        fprintf(stderr, "%s: directory is not owned by root: (%s)\n", lcmaps_log_string, LCMAPS_REQUIRED_DB_PATH);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the directory is writable by others.
     */
    if (dir_info.st_mode & S_IWOTH)
    {
        fprintf(stderr, "%s: directory is writable by others: (%s)\n", lcmaps_log_string, LCMAPS_REQUIRED_DB_PATH);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if we cannot stat the lcmaps.db file.
     */
    if (((lstat(lcmaps_db_file, &prg_info)) != 0) || (S_ISLNK(prg_info.st_mode)))
    {
        fprintf(stderr, "%s: cannot stat program: (%s)\n", lcmaps_log_string, lcmaps_db_file);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the lcmaps.db file is not owned by root
     */
    if ((prg_info.st_uid != 0) || (prg_info.st_gid != 0))
    {
        fprintf(stderr, "%s: file is not owned by root: (%s)\n", lcmaps_log_string, lcmaps_db_file);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the lcmaps.db file is writable by others.
     */
    if (prg_info.st_mode & S_IWOTH)
    {
        fprintf(stderr, "%s: file is writable by others: (%s)\n", lcmaps_log_string, lcmaps_db_file);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /* Everything's fine, so set (overwrite!) the environment variable. */
    setenv("LCMAPS_DB_FILE", lcmaps_db_file, 1);
    if (lcmaps_db_file)
    {
        free(lcmaps_db_file);
        lcmaps_db_file = NULL;
    }
#endif

    /* Parse arguments from va_list */
    va_start(ap, narg);
    if (narg == 2)
    {
        pem_string = va_arg(ap, char *);
        lcmaps_account = va_arg(ap, lcmaps_account_info_t);
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: The number of arguments (%d) should be 2\n", __func__, narg);
        return 1;
    }
    va_end(ap);

#if 0
    if (start_logging())
    {
        fprintf(stderr, "%s: Cannot start logging\n", lcmaps_log_string);
        return 1;
    }
#endif

    /* Parse policy environment variable */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++) lcmaps_policies[jj] = 0;
    lcmaps_policy_string = getenv("LCMAPS_POLICY_NAME");
    if ((rc = lcmaps_tokenize(lcmaps_policy_string, lcmaps_policies, &lcmaps_npols, ":")) != 0)
    {
        lcmaps_log(LOG_ERR, "%s: Cannot parse LCMAPS_POLICY_NAME environment variable, because\n", __func__);
        switch (rc)
        {
            case -1:
                lcmaps_log(LOG_ERR, "%s: of a malloc error\n", __func__);
                break;
            case -2:
                lcmaps_log(LOG_ERR, "%s: the policy list is too large (max = %d)\n",
                    __func__, LCMAPS_MAX_POLICIES);
                break;
            case -3:
                lcmaps_log(LOG_ERR, "%s: of a non-matching quote\n", __func__);
                break;
            case -4:
                lcmaps_log(LOG_ERR, "%s: invalid input\n", __func__);
                break;
            default:
                lcmaps_log(LOG_ERR, "%s: of an unknown error\n", __func__);
                break;
        }
        goto verify_account_from_pem_error;
    }

    /* First initialize LCMAPS: 1st arg used to be lcmaps_log_stream, but always NULL*/
    if (lcmaps_init_and_log(NULL, DO_USRLOG|DO_SYSLOG))
    {
        lcmaps_log(LOG_ERR, "%s: LCMAPS initialization failure\n", __func__);
        goto verify_account_from_pem_error;
    }


    /*
     * Now that we have the credential let's run (and terminate) LCMAPS !
     */
    rc = lcmaps_run_and_verify_account_from_pem(
        NULL,
        pem_string,
        lcmaps_account.uid,
        lcmaps_account.pgid_list,
        lcmaps_account.npgid,
        lcmaps_account.sgid_list,
        lcmaps_account.nsgid,
        lcmaps_account.poolindex,
        NULL,
        lcmaps_npols,
        lcmaps_policies
    );
    if (rc != 0)
    {
        lcmaps_log(LOG_NOTICE, "LCMAPS failed to verify the account mapping\n");
        if (lcmaps_term())
        {
            lcmaps_log(LOG_ERR, "LCMAPS termination failure\n");
            goto verify_account_from_pem_error;
        }
        goto verify_account_from_pem_error;
    }

    rc = lcmaps_term();
    if (rc)
    {
        lcmaps_log(LOG_ERR, "LCMAPS termination failure\n");
        goto verify_account_from_pem_error;
    }

#if 0
    if (end_logging())
    {
        fprintf(stderr, "%s: Cannot cleanly end lcmaps logging\n", lcmaps_log_string);
        goto verify_account_from_pem_error;
    }
#endif

    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }

    return 0;

 verify_account_from_pem_error:
    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }
#if 0
    if (end_logging())
    {
        fprintf(stderr, "%s: Cannot cleanly end lcmaps logging\n", lcmaps_log_string);
    }
#endif

    return 1;
}

/******************************************************************************
Function:   lcmaps_verify_account_from_pem
Description:
    Verify the account mapping from the supplied user credential.

Parameters:
    pem_string:    :  The PEM-encoded string containing the user proxy
    lcmaps_account :  A structure that contains account information: (input)
                      uid, gids (primary and secondary).
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_verify_account_from_pem(
        char * pem_string,
        lcmaps_account_info_t lcmaps_account
)
{
    return (lcmaps_verify_account_from_pem_va(2, pem_string, lcmaps_account));
}

#endif /* LCMAPS_VERIFY_ACCOUNT_FROM_PEM_C */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps_verify_account_from_pem.c,v $
    $Date: 2012-01-26 12:07:08 +0100 (Thu, 26 Jan 2012) $
    $Revision: 15876 $
    $Author: msalle $
******************************************************************************/
