package tests::ReportColumnInfoTest;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::Report::TableInfo;

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->{'table_info'} = new Lire::Report::TableInfo();
    $self->{'num_col_info'} =
      new Lire::Report::ColumnInfo( $self->{'table_info'},
                                    'test', 'numerical',
                                    'int', 'A Label' );
    return $self;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    return $self;
}

sub test_new {
    my $self = $_[0];

    my @arg_names = ( 'group_info', 'name', 'class', 'type' );
    my @args = ( $self->{'table_info'}, 'test', 'categorical', 'string' );
    for ( my $i=0; $i < @arg_names; $i++ ) {
        my @args = ($i == 0) ? () : @args[0..($i-1)];
        $self->assert_died( sub { new Lire::Report::ColumnInfo( @args ) },
                            qr/missing '$arg_names[$i]' parameter/ );
    }

    $self->assert_died( sub { new Lire::Report::ColumnInfo( $self->{'table_info'},
                                                            'invalid%',
                                                            'categorical',
                                                            'string' ) },
                        qr/'name' parameter isn't a valid XML name: 'invalid%'/ );

    $self->assert_died( sub { new Lire::Report::ColumnInfo( {},
                                                            'test',
                                                            'categorical',
                                                            'string' ) },
                        qr/\'group_info\' parameter should be a \'Lire::Report::GroupInfo\' instance, not \'HASH/ );

    $self->assert_died( sub { new Lire::Report::ColumnInfo( $self->{'table_info'},
                                                            'test',
                                                            'invalid',
                                                            'string' ) },
                        qr/'class' parameter should either be 'numerical' or 'categorical'/ );

    $self->assert_died( sub { new Lire::Report::ColumnInfo( $self->{'table_info'},
                                                            'test',
                                                            'categorical',
                                                            'nosuchtype' ) },
                        qr/'type' parameter isn't a valid Lire type: 'nosuchtype'/ );

    my $col_info = new Lire::Report::ColumnInfo( $self->{'table_info'},
                                                 'test', 'categorical',
                                                 'string', 'A Label' );

    $self->assert_not_null( $col_info,
                            "new() returned undef" );
    $self->assert_isa( 'Lire::Report::ColumnInfo', $col_info );
    $self->assert_equals( "test", $col_info->name() );
    $self->assert_equals( "categorical", $col_info->class() );
    $self->assert_equals( "string", $col_info->type() );
    $self->assert_equals( $self->{'table_info'}, $col_info->group_info() );
    $self->assert_equals( 'A Label', $col_info->label() );
    $self->assert_null( $col_info->{'max_chars'},
                        'max_chars() should be undef' );
    $self->assert_null( $col_info->{'index'},
                        "'index' attribute should be undef" );
    $self->assert_equals( 0, $col_info->{'col_width'} );
    $self->assert_equals( 0, $col_info->{'padding'} );
    $self->assert_null( $col_info->{'avg_chars'},
                        "'avg_chars' attribute should be undef" );
}

sub test_avg_chars {
    my $self = $_[0];

    my $col_info = $self->{'num_col_info'};
    $self->assert_died( sub { $col_info->avg_chars() },
                        qr/avg_chars\(\) called while stat is undefined. Was subreport's finalize\(\) called\?/ );
    $col_info->avg_chars( 10 );
    $self->assert_equals( 10, $col_info->{'avg_chars'} );
    $self->assert_equals( 10, $col_info->avg_chars() );
}

sub test_max_chars {
    my $self = $_[0];

    my $col_info = $self->{'num_col_info'};
    $self->assert_died( sub { $col_info->max_chars() },
                        qr/max_chars\(\) called while stat is undefined. Was subreport's finalize\(\) called\?/ );
    $col_info->max_chars( 10 );
    $self->assert_equals( 10, $col_info->{'max_chars'} );
    $self->assert_equals( 10, $col_info->max_chars() );
}

sub test_label {
    my $self = $_[0];

    my $col_info = $self->{'num_col_info'};
    $self->assert_equals( 'A Label', $col_info->label() );
    $col_info->label( undef );
    $self->assert_null( $col_info->{'label'}, "'label' should be undef" );
    $self->assert_equals( 'test', $col_info->label() );

    $self->assert_equals( 'New label', $col_info->label( 'New label' ) );
}

sub test_init_chars_stats {
    my $self = $_[0];

    my $col_info = $self->{'num_col_info'};

    $col_info->init_chars_stats();

    $self->assert_equals( 0, $col_info->{'total_chars'} );
    $self->assert_equals( 0, $col_info->{'column_count'} );
    $self->assert_equals( length( $col_info->label() ),
                          $col_info->{'max_chars'} );
}

sub test_finish_chars_stats {
    my $self = $_[0];

    my $col_info = $self->{'num_col_info'};
    $col_info->{'total_chars'} = 10;
    $col_info->{'column_count'} = 2;

    $col_info->finish_chars_stats();
    $self->assert_equals( 6, $col_info->avg_chars() );
    $self->assert_equals( 6, $col_info->{'avg_chars'} );
    $self->assert_null( $col_info->{'total_chars'},
                        "'total_chars' attribute should be removed" );
    $self->assert_null( $col_info->{'column_count'},
                        "'column_count' attribute should be removed" );
    $self->assert_equals( 6, $col_info->avg_chars() );

    $col_info->avg_chars( 8 );
    $self->assert_equals( 8, $col_info->{'avg_chars'} );
}

sub test_update_chars_stats {
    my $self = $_[0];

    my $col_info = $self->{'num_col_info'};
    $self->assert_null( $col_info->{'total_chars'},
                        "'total_chars' attribute should be undef" );
    $self->assert_null( $col_info->{'column_count'},
                        "'column_count' attribute should be undef" );

    my $len1 = length 'A longer value';
    $col_info->init_chars_stats();
    $col_info->update_chars_stats( 'A longer value' );
    $self->assert_equals( $len1, $col_info->{'total_chars'} );
    $self->assert_equals( $len1, $col_info->{'max_chars'} );
    $self->assert_equals( 1, $col_info->{'column_count'} );

    my $len2 = length 'A short';
    $col_info->update_chars_stats( 'A short' );
    $self->assert_equals( $len1+$len2, $col_info->{'total_chars'} );
    $self->assert_equals( $len1, $col_info->{'max_chars'} );
    $self->assert_equals( 2, $col_info->{'column_count'} );
}

1;

